package org.lsst.ccs.subsystem.vacuum;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.twistorr.TwisTorr84;
import org.lsst.ccs.subsystem.common.devices.turbopump.TwisTorr84Device;
import org.lsst.ccs.subsystem.vacuum.constants.DeviceState;
import org.lsst.ccs.subsystem.vacuum.constants.Devices;

/**
 * Device class which identifies the cryo system turbo pump
 * 
 * @author Owen Saxton
 */
public class VacTurboDevice  extends TwisTorr84Device implements SwitchDevice {

    private static final Map<TwisTorr84.PumpStatus, DeviceState> turboStateMap = new HashMap<>();
    static {
        turboStateMap.put(TwisTorr84.PumpStatus.STOP, DeviceState.STOPPED);
        turboStateMap.put(TwisTorr84.PumpStatus.WAIT_INTLK, DeviceState.WAITING);
        turboStateMap.put(TwisTorr84.PumpStatus.STARTING, DeviceState.STARTNG);
        turboStateMap.put(TwisTorr84.PumpStatus.NORMAL, DeviceState.NORMAL);
        turboStateMap.put(TwisTorr84.PumpStatus.BRAKING, DeviceState.BRAKING);
        turboStateMap.put(TwisTorr84.PumpStatus.FAIL, DeviceState.FAILED);
        turboStateMap.put(TwisTorr84.PumpStatus.AUTO_TUNING, DeviceState.AUTOTUN);
    }

    private int deviceId = Devices.DEVC_NO_SWITCH;


    /**
     *  Gets the switch device ID.
     * 
     *  @return  The device ID
     */
    @Override
    public int getSwitchDevice()
    {
        return deviceId;
    }


    /**
     *  Turns a switch on or off.
     * 
     *  @param  sw  The switch ID
     *  @param  on  Whether to turn on or off
     *  @throws  DriverException
     */
    @Override
    public void setSwitch(int sw, boolean on) throws DriverException
    {
        if (on) {
            startTurboPump();
        }
        else {
            stopTurboPump();
        }
    }


    /**
     *  Gets whether a switch is on.
     * 
     *  @param  sw  The switch ID
     *  @return  Whether on, or null if device is offline
     */
    @Override
    public Boolean isSwitchOn(int sw)
    {
        DeviceState st = getDeviceState();
        return st == null ? null : st != DeviceState.STOPPED && st != DeviceState.BRAKING;
    }


    /**
     *  Gets the device state.
     * 
     *  @return  The device state
     */
    public DeviceState getDeviceState()
    {
        try {
            return turboStateMap.get(readTurboStatus());
        }
        catch (DriverException e) {
            return null;
        }
    }
    
}
