package org.lsst.ccs.subsystem.vacuum;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystem.vacuum.constants.Conditions;
import org.lsst.ccs.subsystem.vacuum.constants.DeviceState;
import org.lsst.ccs.subsystem.vacuum.constants.Latches;

/**
 *  Handles the vacuum system simulated Pluto PLC gateway.
 *
 *  @author Owen Saxton
 */
public class VacSimPlutoDevice extends VacPlutoDevice {

    /**
     *  Data fields.
     */
    private static final Logger LOG = Logger.getLogger(VacSimPlutoDevice.class.getName());
    private final boolean[] switches = new boolean[NUM_SWITCHES];
    private final DeviceState[] switchStates = new DeviceState[NUM_SWITCHES];
    private final Boolean[] latches = new Boolean[Latches.NUM_LATCHES];
    private final boolean[] conditions = new boolean[Conditions.NUM_CONDITIONS];


    /**
     *  Performs basic initialization.
     */
    @Override
    protected void initDevice()
    {
        switchStates[SW_OPEN_VCR00] = DeviceState.SHUT;
        switchStates[SW_OPEN_VHX00] = DeviceState.SHUT;
    }


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        LOG.log(Level.INFO, "Connected to simulated Pluto gateway");
        setOnline(true);
    }


    /**
     *  Closes device connection.
     */
    @Override
    protected void close()
    {
    }


    /**
     *  Checks a channel's parameters for validity.
     *
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel number
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string
     *  @return  A two-element array containing the encoded type [0] and subtype [1] values.
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype)
    {
        return new int[]{0, 0};
    }


    /**
     *  Reads all referenced channels.
     */
    @Override
    protected void readChannelGroup()
    {
    }


    /**
     *  Reads a channel.
     *
     *  @param  hwChan   The hardware channel number.
     *  @param  type     The encoded channel type returned by checkChannel.
     *  @return  The read value
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        return 0.0;
    }


    /**
     *  Tests whether the PLC is active
     *
     *  @return  Whether the PLC is active, or null if offline
     */
    @Override
    public Boolean isPLCActive()
    {
        return true;
    }


    /**
     *  Sets a switch on or off.
     *
     *  For the vacuum Pluto, this is implemented as a pair of push buttons,
     *  one for on, one for off.
     *
     *  @param  sw  The switch number.
     *  @param  on  The on state to set: true or false
     */
    @Override
    public void setSwitch(int sw, boolean on)
    {
        switches[sw] = on;
        switchStates[sw] = on ? DeviceState.OPEN : DeviceState.SHUT;
    }


    /**
     *  Gets the on state of a switch.
     *
     *  The state is not the state of the bit that was toggled, but is read back
     *  either directly from the controlled hardware, or from the PLC output line.
     *
     *  @param  sw  The switch number.
     *  @return  Whether the switch is on
     */
    @Override
    public Boolean isSwitchOn(int sw)
    {
        return switches[sw];
    }


    /**
     *  Gets the detailed device state of a switch.
     * 
     *  Used only for gate valves
     * 
     *  @param  sw  The switch number
     *  @return  The device state, or null if not a gate valve
     */
    @Override
    public DeviceState getSwitchState(int sw)
    {
        return switchStates[sw];
    }


    /**
     *  Gets whether a latched condition is active.
     * 
     *  @param  cond  The condition number
     *  @return  Whether active - indicated by the bit being 0
     */
    @Override
    public Boolean isLatchActive(int cond)
    {
        Boolean latch = latches[cond];
        return latch != null && latch;
    }


    /**
     *  Gets whether a latched condition is latched.
     * 
     *  @param  cond  The condition number
     *  @return  Whether latched - indicated by the bit being 1
     */
    @Override
    public Boolean isLatchLatched(int cond)
    {
        Boolean latch = latches[cond];
        return latch != null && !latch;
    }


    /**
     *  Clears a latched condition.
     * 
     *  @param  cond  The condition number
     */
    @Override
    public void clearLatch(int cond)
    {
        Boolean latch = latches[cond];
        latches[cond] = latch != null && !latch ? null : latch;
    }


    /**
     *  Gets whether a condition is active.
     * 
     *  @param  cond  The condition number
     *  @return  Whether active - indicated by the bit being 1
     */
    @Override
    public Boolean isConditionActive(int cond)
    {
        return conditions[cond];
    }


    @Command(type = Command.CommandType.ACTION)
    public void setSwitchState(@Argument(description="Switch number") int sw,
                               @Argument(description="Device state") DeviceState state) {
        switchStates[sw] = state;
    }

    @Command(type = Command.CommandType.ACTION)
    public void setLatchActive(@Argument(description="Latched condition number") int cond) {
        latches[cond] = true;
    }

    @Command(type = Command.CommandType.ACTION)
    public void setLatchLatched(@Argument(description="Latched condition number") int cond) {
        latches[cond] = false;
    }

    @Command(type = Command.CommandType.ACTION)
    public void setCondition(@Argument(description="Running condition number") int cond,
                             @Argument(description="Condition on state") boolean set) {
        conditions[cond] = set;
    }

}
