package org.lsst.ccs.subsystem.vacuum;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Logger;
import java.util.logging.Level;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.twistorr.TwisTorr;
import org.lsst.ccs.subsystem.vacuum.constants.Switches;
import org.lsst.ccs.subsystem.common.devices.turbopump.TwisTorrDevice;
import org.lsst.ccs.subsystem.vacuum.constants.DeviceState;
import org.lsst.ccs.subsystem.vacuum.constants.Devices;

import org.lsst.ccs.config.ConfigurationParameterDescription;


/**
 * Device class which identifies the cryo system turbo pump
 * 
 * @author Owen Saxton
 */
public class VacTurboDevice  extends TwisTorrDevice implements SwitchDevice {

    private static final Logger LOG = Logger.getLogger(VacTurboDevice.class.getName());

    /**
     *  Constants.
     */
    public static final int
            SW_PUMP = 0,
            SW_VENT = 1;

    private static final Map<TwisTorr.PumpStatus, DeviceState> turboStateMap = new HashMap<>();
    static {
        turboStateMap.put(TwisTorr.PumpStatus.STOP, DeviceState.STOPPED);
        turboStateMap.put(TwisTorr.PumpStatus.WAIT_INTLK, DeviceState.WAITING);
        turboStateMap.put(TwisTorr.PumpStatus.STARTING, DeviceState.STARTNG);
        turboStateMap.put(TwisTorr.PumpStatus.NORMAL, DeviceState.NORMAL);
        turboStateMap.put(TwisTorr.PumpStatus.BRAKING, DeviceState.BRAKING);
        turboStateMap.put(TwisTorr.PumpStatus.FAIL, DeviceState.FAILED);
        turboStateMap.put(TwisTorr.PumpStatus.AUTO_TUNING, DeviceState.AUTOTUN);
    }

    private static final Map<String, TwisTorr.CmndBool> registerBoolMap = new HashMap<>();
    static {
        registerBoolMap.put("lowSpeedMode",TwisTorr.CmndBool.LOW_SPEED_MODE);
        registerBoolMap.put("softStartMode",TwisTorr.CmndBool.SOFT_START_MODE);
        registerBoolMap.put("interlockType",TwisTorr.CmndBool.INTERLOCK_TYPE);
        registerBoolMap.put("ventvalveByCmnd",TwisTorr.CmndBool.VENTVALVE_BY_CMND);
        registerBoolMap.put("speedReadActive",TwisTorr.CmndBool.SPEED_READ_ACTIVATE);
        registerBoolMap.put("setpointActive",TwisTorr.CmndBool.SETPOINT_ACTIVE);
        registerBoolMap.put("ventvalveType",TwisTorr.CmndBool.VENTVALVE_TYPE);
        registerBoolMap.put("activeStopMode",TwisTorr.CmndBool.ACTIVE_STOP_MODE);
        registerBoolMap.put("waterCooling",TwisTorr.CmndBool.WATER_COOLING);
        registerBoolMap.put("gasType",TwisTorr.CmndBool.GAS_TYPE_ARGON);
    }
    private static final Map<String, TwisTorr.CmndNumeric> registerNumMap = new HashMap<>();
    static {
        registerNumMap.put("setpointType",TwisTorr.CmndNumeric.SETPOINT_TYPE);
        registerNumMap.put("setpointThresh",TwisTorr.CmndNumeric.SETPOINT_THRESH);
        registerNumMap.put("ventDelay",TwisTorr.CmndNumeric.VENT_DELAY);
        registerNumMap.put("rotfreq_set",TwisTorr.CmndNumeric.ROTFREQ_SET);
        registerNumMap.put("rotfreq_low",TwisTorr.CmndNumeric.ROTFREQ_LOW);
        registerNumMap.put("gasType",TwisTorr.CmndNumeric.GAS_TYPE);
    }
    private int deviceId = Devices.DEVC_NO_SWITCH;

    private TwisTorr.CntlrModels model = null;
    
    /*
     * VacTurboDevice - construct class using deviceId and turbo model name provided by call from the groovy file
     *
     */

   public VacTurboDevice(int deviceId, String modelName) throws DriverException {
        LOG.fine("modelName = "+modelName);
        if (modelName != null) {
            model = TwisTorr.CntlrModels.valueOf(modelName);
            setModel(model);
        }
        LOG.fine("model = "+model);
        this.deviceId = deviceId;
    }
   
    
    /**
     *  Gets the switch device ID.
     * 
     *  @return  The device ID
     */
    @Override
    public int getSwitchDevice()
    {
        return deviceId;
    }


    /**
     *  Turns a switch on or off.
     * 
     *  @param  sw  The switch ID
     *  @param  on  Whether to turn on or off
     *  @throws  DriverException
     */
    @Override
    public void setSwitch(int sw, boolean on) throws DriverException
    {
        if (sw == SW_PUMP) {
            if (on) {
                startTurboPump();
            } else {
                stopTurboPump();
            }
        } else if (sw == SW_VENT) {
            if (on) {
                timedPulseVentValve();
            } else {
                closeVentValve();
            } 
        } else {
            LOG.log(Level.WARNING, String.format("setSwitch(sw=%d, on=%s): invalid argument sw", sw, on));
        }
    }


    /**
     *  Gets whether a switch is on.
     * 
     *  @param  sw  The switch ID
     *  @return  Whether on, or null if device is offline
     */
    @Override
    public Boolean isSwitchOn(int sw)
    {
        if (sw == SW_PUMP) {
            DeviceState st = getDeviceState();  // treating the turbopump as a switch
            return st == null ? null : st != DeviceState.STOPPED && st != DeviceState.BRAKING;
        } else if (sw == SW_VENT) {
            return getVentValveOpenState();
        } else {
            LOG.log(Level.WARNING, String.format("isSwitchon(sw=%d): invalid argument sw", sw));
            return null;
        }
    }


    /**
     *  Gets the device state.
     * 
     *  @return  The device state
     */
    public DeviceState getDeviceState()
    {
        try {
            return turboStateMap.get(readTurboStatus());
        }
        catch (DriverException e) {
            return null;
        }
    }
    
    /**
     *  Gets the vent valve state
     * 
     *  @return  true for valve open
     */
    public Boolean getVentValveOpenState()
    {
        try {
            return(isVentValveOpen());
        }
        catch (DriverException e) { // offline
            return null;
        }
    }
    
}
