package org.lsst.ccs.subsystem.vacuum;

import java.util.ArrayList;
import java.time.Duration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import java.lang.Exception;
import java.util.concurrent.TimeUnit;

import org.lsst.ccs.utilities.taitime.CCSTimeStamp;
import org.lsst.ccs.Agent;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.Measurement;
import org.lsst.ccs.bus.states.AlertState;

import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.CommandResult;
import org.lsst.ccs.messaging.CommandOriginator;

import org.lsst.ccs.bus.messages.CommandAck;
import org.lsst.ccs.bus.messages.CommandNack;

import org.lsst.ccs.bus.messages.EmbeddedObjectDeserializationException;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.services.MessagingService;

import org.lsst.ccs.messaging.ConcurrentMessagingUtils;

import org.lsst.ccs.subsystem.power.data.QuadBoxState;
import org.lsst.ccs.subsystem.power.constants.PowerAgentProperties;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.alert.AlertEvent;
import org.lsst.ccs.services.alert.AlertListener;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.MonitorTaskControl;
import org.lsst.ccs.subsystem.common.actions.RefrigAction;
import org.lsst.ccs.subsystem.vacuum.constants.VacuumAlert;
import org.lsst.ccs.subsystem.vacuum.constants.ConditionState;
import org.lsst.ccs.subsystem.vacuum.constants.Conditions;
import org.lsst.ccs.subsystem.vacuum.constants.DeviceState;
import org.lsst.ccs.subsystem.vacuum.constants.Devices;
import org.lsst.ccs.subsystem.vacuum.constants.LatchState;
import org.lsst.ccs.subsystem.vacuum.constants.Latches;
import org.lsst.ccs.subsystem.vacuum.constants.MonitorControl;
import org.lsst.ccs.subsystem.vacuum.constants.PLCState;
import org.lsst.ccs.subsystem.vacuum.constants.SwitchEnable;
import org.lsst.ccs.subsystem.vacuum.constants.Switches;
import org.lsst.ccs.subsystem.vacuum.constants.SwitchState;
import org.lsst.ccs.subsystem.vacuum.constants.VacuumAgentProperties;
import org.lsst.ccs.subsystem.vacuum.constants.CryoVacuumState;
import org.lsst.ccs.subsystem.vacuum.constants.HxVacuumState;
import org.lsst.ccs.subsystem.vacuum.data.VacuumException;
import org.lsst.ccs.subsystem.vacuum.data.VacSysState;
import org.lsst.ccs.subsystem.power.constants.QuadBoxSwitches;


import org.lsst.ccs.drivers.twistorr.TwisTorr;


/**
 * The camera vacuum system
 *
 * @author The LSST CCS Team
 */
public class VacuumMain extends Subsystem implements HasLifecycle, AlertListener, AgentPresenceListener, StatusMessageListener {

    //    private final CommandSender sender;

    
    
    
    volatile QuadBoxState qbs = null;

    @Override
    public void onStatusMessage(StatusMessage msg) {
        try {
            if (msg instanceof StatusSubsystemData) {
                StatusSubsystemData sd = (StatusSubsystemData)msg;
                LOG.log(Level.FINE,"onStatusMessage key = "+QuadBoxState.KEY+" | d.sDataKey = "+sd.getDataKey());
                if (sd.getDataKey().equals(QuadBoxState.KEY) ) {
                    LOG.log(Level.FINE,"QuadBoxState sd received");
                    qbs = (QuadBoxState) ( ((KeyValueData)sd.getSubsystemData()).getValue() );
                }
            }
        } catch (EmbeddedObjectDeserializationException x) {
        }
    }


    static class SensorData {
        Channel channel;
        int maxErrors;
        int numErrors = 0;
        boolean getLimits = false;
        double value = 0.0;
        CCSTimeStamp valueCCSTimestamp;
        double goodValue = Double.NaN;
        //        double previousValue = Double.NaN;
        CCSTimeStamp goodValueCCSTimestamp;
        double lowLimit;
        double highLimit;

        private SensorData(Channel channel, int maxErrors, boolean getLimits) {
            this.channel = channel;
            this.maxErrors = maxErrors;
            this.getLimits = getLimits;
        }

        private SensorData(Channel channel, int maxErrors) {
            this(channel, maxErrors, false);
        }
    }

    private static final double
        PRESS_ATMOS = 759.0,
        SPEED_CRYO_TURBO_MAX = 60000,
        SPEED_HX_TURBO_MAX = 81000,
        PUMPCART_RESPONSE_TIME = 2.0;

    private static final int
        MAX_PRESS_ERRORS = 2,
        MAX_SPEED_ERRORS = 0,
        MAX_TEMP_ERRORS = 0,
        MAX_POWER_ERRORS = 0,
        MAX_AIR_ERRORS = 0,
        DELAY_ION_OFF = 10000,
        DELAY_VACUUM_BAD = 10000,
        MAX_PCPUMP_ERRORS = 2;

    private int turboFailureCode = TwisTorr.PumpStatus.FAIL.ordinal();

    private boolean testingModeOn = false; // run in test mode ... only set true by command during runtime

    // ---------------------------------------------------------------------------
    private static final Map<Integer, VacuumAlert> latchAlertMap = new HashMap<>();
    static {
        latchAlertMap.put(Latches.LATCH_CR_VACUUM, VacuumAlert.CRYO_VACUUM_BAD);
        latchAlertMap.put(Latches.LATCH_CR_GATE_NFC, VacuumAlert.CRYO_GATE_FORCED_SHUT);
        latchAlertMap.put(Latches.LATCH_CR_GATE_AO, VacuumAlert.CRYO_GATE_CANNOT_OPEN);
        latchAlertMap.put(Latches.LATCH_CR_PUMP, VacuumAlert.CRYO_TURBO_PUMP_BAD);
        latchAlertMap.put(Latches.LATCH_HX_VACUUM, VacuumAlert.HX_VACUUM_BAD);
        latchAlertMap.put(Latches.LATCH_HX_GATE_NFC, VacuumAlert.HX_GATE_FORCED_SHUT);
        latchAlertMap.put(Latches.LATCH_HX_GATE_AO, VacuumAlert.HX_GATE_CANNOT_OPEN);
        latchAlertMap.put(Latches.LATCH_HX_PUMP, VacuumAlert.HX_TURBO_PUMP_BAD);
    }
    private static final Map<String, Integer> revLatchAlertMap = new HashMap<>();
    static {
        for (int cond : latchAlertMap.keySet()) {
            revLatchAlertMap.put(latchAlertMap.get(cond).getId(), cond);
        }
    }
    private static final List<Integer> cryoIonPumps = new ArrayList<>();
    static {
        cryoIonPumps.add(Switches.SW_CRYO_ION_PUMP1);
        cryoIonPumps.add(Switches.SW_CRYO_ION_PUMP2);
        cryoIonPumps.add(Switches.SW_CRYO_ION_PUMP3);
        cryoIonPumps.add(Switches.SW_CRYO_ION_PUMP4);
        cryoIonPumps.add(Switches.SW_CRYO_ION_PUMP5);
        cryoIonPumps.add(Switches.SW_CRYO_ION_PUMP6);
    }
    private static final List<Integer> hxIonPumps = new ArrayList<>();
    static {
        hxIonPumps.add(Switches.SW_HX_ION_PUMP1);
        hxIonPumps.add(Switches.SW_HX_ION_PUMP2);
    }
    private static final List<Integer> utSwitches = new ArrayList<>();
    static {
        utSwitches.add(Switches.SW_INST_FTH_VALVE);
        utSwitches.add(Switches.SW_INST_FTPP_VALVE);
        utSwitches.add(Switches.SW_INST_L3H_VALVE);
        utSwitches.add(Switches.SW_INST_L3LF_VALVE);
        utSwitches.add(Switches.SW_INST_SCROLL_PUMP);
        utSwitches.add(Switches.SW_CRYO_SCROLL_PUMP);
        utSwitches.add(Switches.SW_HX_SCROLL_PUMP);
        utSwitches.add(Switches.SW_INST_FLINE_VALVE);
        utSwitches.add(Switches.SW_CRYO_FLINE_VALVE);
        utSwitches.add(Switches.SW_HX_FLINE_VALVE);
    }

    @LookupField(strategy=LookupField.Strategy.TOP)
    private Subsystem subsys;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService aps;


    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService propertiesService;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private ASM380Device pumpCartDevc;

    @LookupField(strategy=LookupField.Strategy.DESCENDANTS)
    private final List<SwitchDevice> switchDevcs = new ArrayList<>();

    // From Groovy file
    private Channel cryoVacPressure, cryoTurboPressure, cryoTurboTemp, cryoTurboPower, cryoFlinePressure, cryoTurboSpeed, cryoTurboStatus,
        hxVacPressure, hxTurboPressure, hxTurboTemp, hxTurboPower, hxFlinePressure, hxTurboSpeed, hxTurboStatus, airPressure,
        pcCyclingStatusChan, pcVentingStatusChan, cryoFlineValveStatusChan, hxFlineValveStatusChan, pcInletPressureChan;

    // Configuration parameters

    @ConfigurationParameter(category="Vacuum", isFinal=false, units="Torr", description="cryostat pressure change pumpcart panel update trigger")
    private volatile double pChangeStateUpdate = 0.1;
    @ConfigurationParameter(category="Vacuum", isFinal=false, units="Torr", description="foreline pressure change pumpcart panel update trigger")
    private volatile double pFLChangeStateUpdate = 1.0;
    @ConfigurationParameter(category="Vacuum", isFinal=false, units="Torr", description="pumpcart pressure change pumpcart panel update trigger")
    private volatile double pPCChangeStateUpdate = 0.5;

    @ConfigurationParameter(category="Vacuum", isFinal=false, units="Torr", description="floor value for the allowable difference btwn the foreline and the PC inlet pressure")
    private volatile double pcPdiffFloor = 5.0e-5;
    @ConfigurationParameter(category="Vacuum", isFinal=false, units="unitless", description="scale value for the allowable difference btwn the foreline and the PC inlet pressure")
    private volatile double pcPdiffScaleFactor = 0.026;
    
    @ConfigurationParameter(category="Vacuum", isFinal=true, range = ("0.0..1.0e-4"), units = "Torr", description="floor value for the allowed cryostat-foreline pressure difference")
    private volatile double cryoPdiffFloor = 1.0e-4;
    @ConfigurationParameter(category="Vacuum", isFinal=true, range = ("0.95..1.0"), units="unitless", description="scale for the allowed cryostat-foreline pressure difference")
    private volatile double cryoPScale = 0.970;
    @ConfigurationParameter(category="Vacuum", isFinal=true, range = ("0.0..1.0e-4"), units = "Torr", description="floor value for the allowed HX-foreline pressure difference")
    private volatile double hxPdiffFloor = 1.0e-4;
    @ConfigurationParameter(category="Vacuum", isFinal=true, range = ("0.95..1.0"), units="unitless", description="scale for the allowed HX-foreline pressure difference")
    private volatile double hxPScale = 0.970;
    @ConfigurationParameter(category="Vacuum", isFinal=false, range = ("0.0..0.5"), units = "Torr", description="max turbo pressure at which turbo pump is enabled")
    private volatile double pressTurboLow; // 0.5 max
    @ConfigurationParameter(category="Vacuum", isFinal=false, range = ("0.0..0.5"), units = "Torr", description="max foreline pressure at which turbo pump is enabled")
    private volatile double pressForelineLow; // 0.5 max
    @ConfigurationParameter(category="Vacuum", isFinal=false, range = ("0.00..0.09"), units = "Torr", description="max allowed press difference across gate valve")
    private volatile double pressDiffLow; // 0.09 max, turbo spinning limit
    @ConfigurationParameter(category="Vacuum", isFinal=true, range = ("0.1..5.0"), units = "Torr", description="max allowed press difference across gate valve when turbo not spinning")
    private volatile double pressDiffHighMin; // 2.0 nom, turbo stopped min of scaled limit
    @ConfigurationParameter(category="Vacuum", isFinal=false, range = ("1.0e-6..1.1e-5"), units = "Torr", description="pressure above which the ion pumps are turned off")
    private volatile double pressIonOff; // 1.1e-5 max
    @ConfigurationParameter(category="Vacuum", isFinal=true, units="Torr", description="pressure below which the ion pumps are enabled")
    private volatile double pressIonEnable; // 1.0e-6
    @ConfigurationParameter(category="Vacuum", isFinal=true, units="Torr", description="pressure below which the system considered to be under vacuum and for correspondingly setting the state")
    private volatile double pressVacuum; // 1.0e-7
    @ConfigurationParameter(category="Vacuum", isFinal=true, units="Torr", description="max pressure for allowing refrigeration to operate")
    private volatile double pressRefrigOk; // 9.0e-3
    @ConfigurationParameter(category="Vacuum", isFinal=false, range = ("-0.01..0.1"), units="unitless", description="fraction of max turbo speed below which the turbo is considered to be not spinning") // allowing a slightly negative value for testing purposes
    private volatile double speedFractTurboLow; // 0.1
    @ConfigurationParameter(category="Vacuum", isFinal=false, range = ("6.0e-5..2.0e-4"), units = "Torr",description="max pressure allowed for enabling the vacuum gauge cold cathode")
    private volatile double pressCCEnable; // 2.0e-4
    @ConfigurationParameter(category="Vacuum", isFinal=false, range = ("1.2e-5..1.2e-4"), units = "Torr",description="max pressure allowed for enabling the vacuum gauge cold cathode")
    private volatile double pressCCOff; // 2e-5
    @ConfigurationParameter(category="Vacuum", isFinal=false, range = ("0..300000"), units = "ms", description="vacuum gauge cold cathode off delay")
    private volatile double delayCCOff; // ~20000 ops, 120000 pump-down
    @ConfigurationParameter(category="Vacuum", isFinal=true, units = "ms", description="max cryo gate transit time between limit switches before declaring a problem")
    private volatile long tranTimeCryoGate; // 2000
    @ConfigurationParameter(category="Vacuum", isFinal=true, units = "ms", description="max cryo gate transit time between limit switches before declaring a problem")
    private volatile long tranTimeHxGate; // 2000

    // General
    private static final Logger LOG = Logger.getLogger(VacuumMain.class.getName());
    private double speedCryoTurboLow, speedHxTurboLow;
    private MonitorTaskControl monitorControl;
    private final SwitchDevice[] switchDevices = new SwitchDevice[Devices.NUM_DEVICES];
    private VacPlutoDevice vacPlutoDevc;
    private IonPumpDevice ionPumpDevc;
    private VacTurboDevice cryoTurboDevc, hxTurboDevc;
    private final VacSysState vacState = new VacSysState();
    private final Map<String, Boolean> activeAlarmMap = new HashMap<>();
    private long cryoIonOverStartTime = 0, hxIonOverStartTime = 0;
    private long cryoVacCCOverStartTime = 0, hxVacCCOverStartTime = 0;
    private long cryoTurboCCOverStartTime = 0, hxTurboCCOverStartTime = 0;
    private long vacBadTime = 0;
    private SensorData cryoFlinePrs, cryoMainPrs, cryoTurboPrs, cryoTurboSpd, cryoTurboTmp, cryoTurboPwr, cryoTurboStat,
        hxFlinePrs, hxMainPrs, hxTurboPrs, hxTurboSpd, hxTurboTmp, hxTurboPwr, hxTurboStat, cmpAirPrs,
        pcCyclingStatus, pcVentingStatus, cryoFlineValveStatus, hxFlineValveStatus, pcInletPressure;
    private double oldCryoFlinePrs, oldCryoMainPrs, oldCryoTurboPrs,
        oldHxFlinePrs, oldHxMainPrs, oldHxTurboPrs, oldCmpAirPrs, oldPCPrs;
    private Map<Integer, Long> gateTransitStart = new HashMap<>();
    private boolean gotCommand = false;

    private int oldPcCycStat, oldPcVentStat, oldCryoFlValveStat, oldhxFlValveStat;

    //    private final SwitchState[] pdu24vcSwitchState = new SwitchState[];


    /**
     *  Constructor
     */
    public VacuumMain() {
        super("vacuum", AgentInfo.AgentType.WORKER);
    }



    /**
     *  Build phase
     */
    @Override
    public void build() {
        // Create the monitor task control object and node
        monitorControl = MonitorTaskControl.createNode(this, MonitorControl.NODE_NAME);

        // Create and schedule an AgentPeriodicTask to update the vacuum state
        AgentPeriodicTask pt;
        pt = new AgentPeriodicTask("vacuum-state", () -> updateVacuumState()).withPeriod(Duration.ofMillis(1000));
        periodicTaskService.scheduleAgentPeriodicTask(pt);

        // Register vacuum states
        stateService.registerState(CryoVacuumState.class, "Cryo vacuum state", this);
        stateService.updateAgentState(CryoVacuumState.UNKNOWN);
        stateService.registerState(HxVacuumState.class, "HX vacuum state", this);
        stateService.updateAgentState(HxVacuumState.UNKNOWN);

        // Initialize active alert map
        for (VacuumAlert alert : VacuumAlert.values()) {
            activeAlarmMap.put(alert.getId(), false);
        }
    }

    /**
     * Init phase.
     * We register here all the Alerts raised by this Subsystem.
     */
    @Override
    public void init() {
        for ( VacuumAlert alert : VacuumAlert.values() ) {
            alertService.registerAlert(alert.getAlert());
        }
    }
    
    public void connected(AgentInfo... agents) {
        for (AgentInfo agent : agents) {
            LOG.info("agent = "+agent);

            //Select the quadbox subsystem from the QUAD_BOX_AGENT property
            //to keep this name independent.
            if ( agent.hasAgentProperty(PowerAgentProperties.QUAD_BOX_AGENT) ) {
                String agentName = agent.getName();
                LOG.info("adding listening for agent = "+agentName);
                subsys.getAgentService(MessagingService.class).getMessagingAccess()
                    .addStatusMessageListener(this, (msg) -> msg.getOriginAgentInfo().getName().equals(agentName));

                //Schedule the fetching of the QuadboxState
                //Ideally this is done only when quadbox is already running and vacuum connects.
                //Unfortunately this situation also occurrs when there is a cluster
                //split, so there isn't an easy way to do it.
                getScheduler().schedule(() -> {requestQuadboxState(agentName);}, 0, TimeUnit.MILLISECONDS);
            }
        }
    }
    
    private void requestQuadboxState(String quadboxName) {
        CommandRequest cmd = new CommandRequest(quadboxName, "publishFullState");
        ConcurrentMessagingUtils cmu = new ConcurrentMessagingUtils(getMessagingAccess());
        try {
            cmu.sendSynchronousCommand(cmd);
        } catch (Exception ex) {
            LOG.log(Level.SEVERE, "Failed to execute command to acquire QuadBoxState",ex);
        }
    }
        
    /*
    @Override
    public void onCommandReply(Object reply, String path, String command, Object[] args) {
        qbs=(QuadBoxState)reply;
    }
    */
    
    /**
     *  Listens for the departure of the companion refrigeration subsystem.
     *
     *  @param  agents  Agents going absent
     */
    @Override
    public void disconnected(AgentInfo... agents) {
        for (AgentInfo agent : agents) {
            if (agent.hasAgentProperty(PowerAgentProperties.QUAD_BOX_AGENT) ) {
                subsys.getAgentService(MessagingService.class).getMessagingAccess().removeStatusMessageListener(this);
            }
        }
    }


    /**
     *  Post initialization phase
     */
    @Override
    public void postInit() {

        //Set a property to define that this Agent is a vacuum subsystem.
        propertiesService.setAgentProperty(VacuumAgentProperties.VACUUM_TYPE, VacuumMain.class.getCanonicalName());

        // Set a property to define that this Agent is a Quad Box controller.
        aps.setAgentProperty(VacuumAgentProperties.VACUUM_AGENT, getClass().getCanonicalName());


        // Add an agent present listener
        subsys.getAgentService(MessagingService.class).getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(this);


        // Add alert listener
        alertService.addListener(this);

        for (SwitchDevice device : switchDevcs) {
            int id = device.getSwitchDevice();
            if (id < Devices.DEVC_NO_SWITCH || id >= Devices.NUM_DEVICES) {
                throw new RuntimeException("Device of class " + device.getClass().getSimpleName() + " has invalid id: " + id);
            }
            if (id != Devices.DEVC_NO_SWITCH) {
                switchDevices[id] = device;
            }
        }
        for (int id = 0; id < Devices.NUM_DEVICES; id++) {
            if (switchDevices[id] == null) {
                throw new RuntimeException(Devices.getDescription(id) + " device not defined");
            }
        }
        vacPlutoDevc = (VacPlutoDevice)switchDevices[Devices.DEVC_PLUTO];
        ionPumpDevc = (IonPumpDevice)switchDevices[Devices.DEVC_ION_PUMP];
        cryoTurboDevc = (VacTurboDevice)switchDevices[Devices.DEVC_CRYO_TURBO_PUMP];
        hxTurboDevc = (VacTurboDevice)switchDevices[Devices.DEVC_HX_TURBO_PUMP];


        if (pcInletPressureChan == null) {
            throw new RuntimeException("Pump Cart Inlet Pressure channel (pcInletPressureChan) not specified");
        }
        pcInletPressure = new SensorData(pcInletPressureChan, MAX_PRESS_ERRORS);

        if (pcCyclingStatusChan == null) {
            throw new RuntimeException("Pump Cart Cycling Status channel (pcCyclingStatusChan) not specified");
        }
        pcCyclingStatus = new SensorData(pcCyclingStatusChan, MAX_PRESS_ERRORS);

        if (pcVentingStatusChan == null) {
            throw new RuntimeException("Pump Cart Venting Status channel (pcVentingStatusChan) not specified");
        }
        pcVentingStatus = new SensorData(pcVentingStatusChan, MAX_PRESS_ERRORS);

        if (cryoFlineValveStatusChan == null) {
            throw new RuntimeException("Cryo Foreline Valve Limit Switches Status channel (cryoFlineValveStatusChan) not specified");
        }
        cryoFlineValveStatus = new SensorData(cryoFlineValveStatusChan, MAX_PRESS_ERRORS);

        if (hxFlineValveStatusChan == null) {
            throw new RuntimeException("HX Foreline Valve Limit Switches Status channel (hxFlineValveStatusChan) not specified");
        }
        hxFlineValveStatus = new SensorData(hxFlineValveStatusChan, MAX_PRESS_ERRORS);


        if (cryoFlinePressure == null) {
            throw new RuntimeException("Cryo foreline pressure channel (cryoFlinePressure) not specified");
        }
        cryoFlinePrs = new SensorData(cryoFlinePressure, MAX_PRESS_ERRORS);
        if (cryoVacPressure == null) {
            throw new RuntimeException("Cryo vacuum pressure channel (cryoVacPressure) not specified");
        }
        cryoMainPrs = new SensorData(cryoVacPressure, MAX_PRESS_ERRORS);
        ionPumpDevc.setCryoPressureChannel(cryoVacPressure);
        if (cryoTurboPressure == null) {
            throw new RuntimeException("Cryo turbo pump pressure channel (cryoTurboPressure) not specified");
        }
        cryoTurboPrs = new SensorData(cryoTurboPressure, MAX_PRESS_ERRORS);
        if (cryoTurboSpeed == null) {
            throw new RuntimeException("Cryo turbo pump speed channel (cryoTurboSpeed) not specified");
        }
        cryoTurboSpd = new SensorData(cryoTurboSpeed, MAX_SPEED_ERRORS);
        if (cryoTurboTemp == null) {
            throw new RuntimeException("Cryo turbo pump temperature channel (cryoTurboTemp) not specified");
        }
        cryoTurboTmp = new SensorData(cryoTurboTemp, MAX_TEMP_ERRORS, true);
        if (cryoTurboPower == null) {
            throw new RuntimeException("Cryo turbo pump power channel (cryoTurboPower) not specified");
        }
        cryoTurboPwr = new SensorData(cryoTurboPower, MAX_POWER_ERRORS, true);
        if (cryoTurboStatus == null) {
            throw new RuntimeException("Cryo turbo pump status channel (cryoTurboStatus) not specified");
        }
        cryoTurboStat = new SensorData(cryoTurboStatus, MAX_POWER_ERRORS, true);
        if (hxFlinePressure == null) {
            throw new RuntimeException("HX foreline pressure channel (hxFlinePressure) not specified");
        }
        hxFlinePrs = new SensorData(hxFlinePressure, MAX_PRESS_ERRORS);
        if (hxVacPressure == null) {
            throw new RuntimeException("HX vacuum pressure channel (hxVacPressure) not specified");
        }
        hxMainPrs = new SensorData(hxVacPressure, MAX_PRESS_ERRORS);
        ionPumpDevc.setHxPressureChannel(hxVacPressure);
        if (hxTurboPressure == null) {
            throw new RuntimeException("HX turbo pump pressure channel (hxTurboPressure) not specified");
        }
        hxTurboPrs = new SensorData(hxTurboPressure, MAX_PRESS_ERRORS);
        if (hxTurboSpeed == null) {
            throw new RuntimeException("HX turbo pump speed channel (hxTurboSpeed) not specified");
        }
        hxTurboSpd = new SensorData(hxTurboSpeed, MAX_SPEED_ERRORS);
        if (hxTurboTemp == null) {
            throw new RuntimeException("HX turbo pump temperature channel (hxTurboTemp) not specified");
        }
        hxTurboTmp = new SensorData(hxTurboTemp, MAX_TEMP_ERRORS, true);
        if (hxTurboPower == null) {
            throw new RuntimeException("HX turbo pump power channel (hxTurboPower) not specified");
        }
        hxTurboPwr = new SensorData(hxTurboPower, MAX_POWER_ERRORS, true);
        if (hxTurboStatus == null) {
            throw new RuntimeException("HX turbo pump status channel (hxTurboStatus) not specified");
        }
        hxTurboStat = new SensorData(hxTurboStatus, MAX_POWER_ERRORS, true);
        if (airPressure == null) {
            throw new RuntimeException("Compressed air pressure channel (airPressure) not specified");
        }
        cmpAirPrs = new SensorData(airPressure, MAX_AIR_ERRORS, true);
        speedCryoTurboLow = speedFractTurboLow * SPEED_CRYO_TURBO_MAX;
        speedHxTurboLow = speedFractTurboLow * SPEED_HX_TURBO_MAX;
    }


    /**
     *  Post start
     */
    @Override
    public void postStart() {
        LOG.info("Vacuum subsystem started");
    }


    /**
     *  Gets the state of the Vacuum system.
     *
     *  @return  The vacuum state
     */
    @Command(type=Command.CommandType.QUERY, description="Get the vacuum system state", level=0)
    public VacSysState getVacuumState() {
        vacState.setTickMillis(monitorControl.getFastPeriod());
        return vacState;
    }    


    /**
     *  Gets the list of switch names.
     *
     *  @return  The switch names.
     *  @throws  VacuumException
     */
    @Command(type=Command.CommandType.QUERY, description="Get switch names", level=0)
    public List<String> getSwitchNames() throws VacuumException
    {
        return Switches.getNames();
    }


    /**
     *  Turns a switch on or off.
     *
     *  @param  name  The switch name.
     *  @param  on    Whether to turn on or off
     *  @throws  VacuumException
     */
    @Command(type=Command.CommandType.ACTION, description="Turn on/off a named switch")
    public void setSwitchOn(@Argument(description="The switch name") String name,
                            @Argument(description="Whether to turn on") boolean on) throws VacuumException
    {
        gotCommand = true;
        setSwitch(Switches.getId(name), on);
    }


    /**
     *  Returns enable and state of switch
     *
     *  @param  name  The switch name.
     *  @throws  VacuumException
     *  @return  The enabled and switch states, as a string
     */
    @Command(type=Command.CommandType.QUERY, description="Print enable and state for switch")
    public String checkSwitch(@Argument(description="The switch name") String name) throws VacuumException
    {
        int sw = Switches.getId(name);
        SwitchEnable enable = vacState.getSwitchEnable(sw);
        SwitchState state = vacState.getSwitchState(sw);
        return "enable="+ enable.name() +", state="+state.name();
    }


    /**
     *  Turns a switch on or off.
     *
     *  @param  sw  The switch number.
     *  @param  on  Whether to turn on or off
     *  @throws  VacuumException
     */
    private void setSwitch(int sw, boolean on) throws VacuumException
    {
        SwitchState state = vacState.getSwitchState(sw);
        if (state == SwitchState.OFFLINE) return;
        SwitchEnable enable = vacState.getSwitchEnable(sw);
        if (on && enable != SwitchEnable.ON) {
            throw new VacuumException("Cannot turn switch on, the switch is disabled.");
        }
        try {
            LOG.log(Level.INFO, String.format("VacuumMain.setSwitch(%d, %s)", sw, on));
            LOG.log(Level.INFO, String.format("VacuumMain.setSwitch(%s, %s)", Switches.getName(sw), on));
            switchDevices[Switches.getDevice(sw)].setSwitch(SwitchInfo.getSwitch(sw), on);
        }
        catch (DriverException e) {
            throw new VacuumException(String.format("Error setting switch for " + Devices.getDescription(Switches.getDevice(sw)) + " device: " + e));
        }
    }


    /**
     *  Gets whether a switch is on.
     *
     *  @param  sw  The switch number.
     *  @return  Whether the switch is on
     *  @throws  VacuumException
     */
    private Boolean isSwitchOn(int sw)
    {
        return switchDevices[Switches.getDevice(sw)].isSwitchOn(SwitchInfo.getSwitch(sw));
    }


    /**
     *  Clears a latched condition.
     *
     *  @param  cond  The condition number.
     *  @throws  VacuumException
     */
    @Command(type=Command.CommandType.ACTION, description="Clear a condition")
    public void clearLatch(@Argument(description="The condition number") int cond) throws VacuumException
    {
        gotCommand = true;
        vacPlutoDevc.clearLatch(cond);
    }


    /**
     *  Updates the vacuum system state periodically.
     *
     *  The vacuum state consists mainly of the state of the switches (lines) being
     *  controlled, along with whether they can be turned on.
     *
     *  Pressures and pump speeds are read to determine which switches are enabled
     *  and/or need to be turned off.  If the state changes it is published.
     */
    private void updateVacuumState()
    {
        boolean changed = monitorControl.hasPeriodChanged();
        changed |= updatePlcState();
        changed |= updateLatchState();
        changed |= updateCondState();
        // !!!!!!!!!!!!!!!!!!!!!!!!!
        // This does not seem safe because any of the 13 device reads of readSensors could cause these updates along with there checks to not be executed. - HN
        // !!!!!!!!!!!!!!!!!!!!!!!!!
        // I agree.  If readSensors returns false then one or more of the values populated from monitoring is invalid.
        // It does not look like any action is taken here.
        // But failure of monitoring should raise other alerts elsewhere -- needs review
        // It would seem this should at a minimum raise and alert and take the system out of normal mode -- SM
        if (readSensors()) {
            changed |= updateSwitchState();
            changed |= updateCryoState();
            changed |= updateHxState();
            checkRefrigVacuum();
        }
        if (gotCommand) {
            gotCommand = false;
            changed = true;
        }


        changed |= updateForelineState();


        if (changed) {
            publishState();
        }
    }

    /**
     *  Updates the foreline valve and pumping states and any significant pressure changes
     * 
     */
    private boolean updateForelineState() {
        boolean changed = false;

        readSensor(pcInletPressure);

        readSensor(pcCyclingStatus);
        readSensor(pcVentingStatus);

        vacState.setPumpCartPumpingSwitchState(pumpCartDevc.getPumpSwitchState());
        vacState.setPumpCartVentingSwitchState(pumpCartDevc.getVentSwitchState());

        LOG.log(Level.FINE,"Pump switch state - "+pumpCartDevc.getPumpSwitchState());

        int pcCycStat = (int)pcCyclingStatus.value;
        if (pcCycStat != oldPcCycStat) {
            changed = true;
        }
        oldPcCycStat = pcCycStat;
        vacState.setCyclingStatus(pcCycStat);

        int pcVentStat = (int)pcVentingStatus.value;
        if (pcVentStat != oldPcVentStat) {
            changed = true;
        }
        oldPcVentStat = pcVentStat;
        vacState.setVentingStatus(pcVentStat);


        // set turbo pump states
        // this needs review, the switch can be read directly, a false here does not imply STOPPED
        vacState.setCryoTurboPumpSwitchState(cryoTurboStat.value==TwisTorr.PumpStatus.NORMAL.ordinal());
        vacState.setHXTurboPumpSwitchState(hxTurboStat.value==TwisTorr.PumpStatus.NORMAL.ordinal());

        // why isn't the SwitchState array the definitive place to get a switch state after it is set in updateSwitch()
        // we appear to have redundant places to put states?
        //
        // vacState.setCryoTurboPumpSwitchState(vacState.getSwitchState(Switches.SW_CRYO_TURBO_PUMP) == SwitchState.ON)
        // vacState.setHxTurboPumpSwitchState(vacState.getSwitchState(Switches.SW_HX_TURBO_PUMP) == SwitchState.ON)
        vacState.setCryoTurboVentingSwitchState(vacState.getSwitchState(Switches.SW_CRYO_TURBO_VENT_VALVE) == SwitchState.ON);
        vacState.setHXTurboVentingSwitchState(vacState.getSwitchState(Switches.SW_HX_TURBO_VENT_VALVE) == SwitchState.ON);


        // ----------------

        readSensor(cryoFlineValveStatus);
        readSensor(hxFlineValveStatus);

        boolean limitSwitchesPowered = false;
        if (qbs != null) {
            limitSwitchesPowered = qbs.getSwitchState(QuadBoxSwitches.SW_INT_VALVES)==org.lsst.ccs.subsystem.power.constants.SwitchState.ON;
        }

        //      int cryoFlValveStat = limitSwitchesPowered ? (int)cryoFlineValveStatus.value : DeviceState.NOPWR.ordinal();
        int cryoFlValveStat = limitSwitchesPowered ? (int)cryoFlineValveStatus.value : DeviceState.OFFLINE.ordinal();
        if (cryoFlValveStat != oldCryoFlValveStat) {
            changed = true;
        }
        oldCryoFlValveStat = cryoFlValveStat;
        vacState.setCryoForelineValveStatus( (DeviceState.values()[cryoFlValveStat]).name() );


        //      int hxFlValveStat = limitSwitchesPowered ? (int)hxFlineValveStatus.value : DeviceState.NOPWR.ordinal() ;
        int hxFlValveStat = limitSwitchesPowered ? (int)hxFlineValveStatus.value : DeviceState.OFFLINE.ordinal() ;
        if (hxFlValveStat != oldhxFlValveStat) {
            changed = true;
        }
        oldhxFlValveStat = hxFlValveStat;
        vacState.setHXForelineValveStatus( (DeviceState.values()[hxFlValveStat]).name() ); // set from monitoring

        // for pump cart pressure difference checks
        //      pumpCartDevc.updateCryoFlinePressure(cryoFlinePrs.value);
        //      pumpCartDevc.updateHXFlinePressure(hxFlinePrs.value);

        double pUpdateScale = 0.5;

        vacState.setCryoPress(cryoMainPrs.value); // always update the state vector but only publish if there has been a significant change
        if (Math.abs(oldCryoMainPrs - cryoMainPrs.value) > Math.min(pChangeStateUpdate,pUpdateScale*oldCryoMainPrs)) {
            changed = true;
            oldCryoMainPrs = cryoMainPrs.value;
        }

        vacState.setCryoTurboPress(cryoTurboPrs.value);
        if (Math.abs(oldCryoTurboPrs - cryoTurboPrs.value) > Math.min(pChangeStateUpdate,pUpdateScale*oldCryoTurboPrs)) {
            changed = true;
            oldCryoTurboPrs = cryoTurboPrs.value;
        }
        vacState.setCryoFlinePress(cryoFlinePrs.value);
        if (Math.abs(oldCryoFlinePrs - cryoFlinePrs.value) > Math.min(pFLChangeStateUpdate,pUpdateScale*oldCryoFlinePrs)) {
            changed = true;
            oldCryoFlinePrs = cryoFlinePrs.value;
        }

        vacState.setHXPress(hxMainPrs.value);
        if (Math.abs(oldHxMainPrs - hxMainPrs.value) > Math.min(pChangeStateUpdate,pUpdateScale*oldHxMainPrs)) {
            changed = true;
            oldHxMainPrs = hxMainPrs.value;
        }
        vacState.setHXTurboPress(hxTurboPrs.value);
        if (Math.abs(oldHxTurboPrs - hxTurboPrs.value) > Math.min(pChangeStateUpdate,pUpdateScale*oldHxTurboPrs)) {
            changed = true;
            oldHxTurboPrs = hxTurboPrs.value;
        }
            //, oldCmpAirPrs;
        vacState.setHXFlinePress(hxFlinePrs.value);
        if (Math.abs(oldHxFlinePrs - hxFlinePrs.value) > Math.min(pFLChangeStateUpdate,pUpdateScale*oldHxFlinePrs)) {
            changed = true;
            oldHxFlinePrs = hxFlinePrs.value;
        }

        vacState.setPCPress(pcInletPressure.value);
        if (Math.abs(oldPCPrs - pcInletPressure.value) > Math.min(pPCChangeStateUpdate,oldPCPrs)) {
            changed = true;
            oldPCPrs = pcInletPressure.value;
        }

        vacState.setPCTargetPressureSrc(pumpCartDevc.getTargetPressureSrc());
        try {
            vacState.setPCTargetPressure(pumpCartDevc.getReferencePressure());
        } catch (DriverException ex) {
            LOG.log(Level.SEVERE, "Failed to set reference pressure value in VacSysState.",ex);
        }
        pumpCartDevc.updateVacSysState(vacState);

        return changed;
    }

    /**
     *  Updates the PLC state
     *
     *  @return  Whether the state changed
     */
    private boolean updatePlcState() {
        boolean changed = false;
        Boolean plcActive = vacPlutoDevc.isPLCActive();
        if (plcActive == null) {
            if (vacState.getPlcState() != PLCState.OFFLINE) {
                raiseAlarm(VacuumAlert.VACUUM_PLC_NOT_ALIVE, "Vacuum PLC is offline", null);
                vacState.setPlcState(PLCState.OFFLINE);
                changed = true;
            }
        }
        else if (!plcActive) {
            if (vacState.getPlcState() != PLCState.DEAD) {
                raiseAlarm(VacuumAlert.VACUUM_PLC_NOT_ALIVE, "Vacuum PLC has died", null);
                vacState.setPlcState(PLCState.DEAD);
                changed = true;
            }
        }
        else {
            if (vacState.getPlcState() != PLCState.ALIVE) {
                lowerAlarm(VacuumAlert.VACUUM_PLC_NOT_ALIVE, "Vacuum PLC is alive", null);
                vacState.setPlcState(PLCState.ALIVE);
                changed = true;
            }
        }
        return changed;
    }


    /**
     *  Updates the state of the PLC latches.
     * 
     *  @return  Whether the state changed
     */
    private boolean updateLatchState() {
        boolean changed = false;
        for (int cond = 0; cond < Latches.NUM_LATCHES; cond++) {
            Boolean active = vacPlutoDevc.isLatchActive(cond);
            Boolean latched = vacPlutoDevc.isLatchLatched(cond);
            LatchState state = active == null || latched == null ? LatchState.OFFLINE :
                               latched ? LatchState.LATCHED :
                               active ? LatchState.ACTIVE : LatchState.CLEAR;
            LatchState oldState = vacState.getLatch(cond); 
            if (state != oldState) {
                vacState.setLatch(cond, state);
                VacuumAlert alert = latchAlertMap.get(cond);
                if (state == LatchState.ACTIVE) {
                    raiseAlarm(alert, "Vacuum PLC error condition set", null);
                }
                else if (state != LatchState.OFFLINE && oldState != LatchState.OFFLINE) {
                    if (state == LatchState.LATCHED && oldState == LatchState.CLEAR) {
                        raiseAlarm(alert, "Vacuum PLC error condition set", null);
                    }
                    if (state == LatchState.LATCHED || oldState == LatchState.ACTIVE) {
                        lowerAlarm(alert, "Vacuum PLC error condition cleared", null);
                    }
                }
                changed = true;
            }
        }
        return changed;
    }


    /**
     *  Updates the state of the PLC conditions.
     * 
     *  @return  Whether the state changed
     */
    private boolean updateCondState() {
        boolean changed = false;
        for (int cond = 0; cond < Conditions.NUM_CONDITIONS; cond++) {
            Boolean active = vacPlutoDevc.isConditionActive(cond);
            ConditionState state = active == null ? ConditionState.OFF :
                                   active ? ConditionState.YES : ConditionState.NO;
            if (state != vacState.getCondition(cond)) {
                vacState.setCondition(cond, state);
                changed = true;
            }
        }
        return changed;
    }


    /**
     *  Reads the gauges and the state of the turbo pumps.
     *
     *  @return  Whether valid values were available for all devices
     */
    private boolean readSensors() {
        boolean valid = true;
        valid &= readSensor(cryoFlinePrs);
        valid &= readSensor(cryoMainPrs);
        valid &= readSensor(cryoTurboPrs);
        valid &= readSensor(cryoTurboSpd);
        valid &= readSensor(cryoTurboTmp);
        valid &= readSensor(cryoTurboPwr);
        valid &= readSensor(cryoTurboStat);
        valid &= readSensor(hxFlinePrs);
        valid &= readSensor(hxMainPrs);
        valid &= readSensor(hxTurboPrs);
        valid &= readSensor(hxTurboSpd);
        valid &= readSensor(hxTurboTmp);
        valid &= readSensor(hxTurboPwr);
        valid &= readSensor(hxTurboStat);
        valid &= readSensor(cmpAirPrs);
        return valid;
    }


    /**
     *  Reads a sensor, allowing for possible temporary errors.
     * 
     *  @param  data  The sensor data object
     *  @return  Whether valid value was available 
     */
    private boolean readSensor(SensorData data) {
        Measurement meas = data.channel.getLastMeasurement();
        if (meas.getCCSTimestamp() == null) return false;
        data.valueCCSTimestamp = meas.getCCSTimestamp();
        data.value = meas.getValue();
        if (!Double.isNaN(data.value)) {
            data.goodValue = data.value;
            data.goodValueCCSTimestamp = data.valueCCSTimestamp; 
            data.numErrors = 0;
        }
        else if (data.numErrors++ < data.maxErrors) {
            data.value = data.goodValue;
        }
        //      data.previousValue = data.value;
        if (data.getLimits) {
            data.lowLimit = data.channel.getLimitLo();
            data.highLimit = data.channel.getLimitHi();
        }
        return true;
    }


    /**
     *  Updates the state of the switches.
     * 
     *  @return  Whether the state changed
     */
    private boolean updateSwitchState() {
        boolean changed = false;
        changed |= updateCryoGateValve();
        changed |= updateCryoTurboPump();
        changed |= updateCryoTurboPumpVentValve();
        changed |= updateCryoIonPumps();
        changed |= updateHxGateValve();
        changed |= updateHxTurboPump();
        changed |= updateHxTurboPumpVentValve();
        changed |= updateHxIonPumps();
        changed |= updateUtSwitches();
        changed |= updateCryoVacCC();
        changed |= updateCryoTurboCC();
        changed |= updateHxVacCC();
        changed |= updateHxTurboCC();
        return changed;
    }



    /**
     *  Updates the state of the cryo gate valve by determining
     *  values and calling updateSwitch(<switch>, enable, turnOff,...)
     * 
     *  @return  Whether the state changed
     */
    private boolean updateCryoGateValve()
    {
        boolean enable = false;  // enable gate valve to open
        boolean turnOff = true;  // force gate valve closed by default

        boolean cryoTurboAvail = !Double.isNaN(cryoMainPrs.value) &&
                                !Double.isNaN(cryoTurboPrs.value) &&
                                !Double.isNaN(cryoTurboSpd.value);

        boolean airCompressorOK = cmpAirPrs.value > cmpAirPrs.lowLimit &&
                                  cmpAirPrs.value < cmpAirPrs.highLimit;

        //test turbo failure action

        if (airCompressorOK && testingModeOn) { // global minimal prerequisite to open
            turnOff = false;
            turnOff |= (int)cryoTurboStat.value == turboFailureCode;
            if (turnOff) {
                LOG.log(Level.FINE,"Turbo failure check would have caused gate valve to close.");
            }
        }
        if (cryoTurboAvail && airCompressorOK) { // global minimal prerequisite to open
            enable = true;
            // pressure difference between turbo->cryo
            double prDiff = Math.abs(cryoTurboPrs.value - cryoMainPrs.value);  // can't be NaN
            
            /* pressDiffLim scales from 20 at high pressure to smaller value at low pressure */
            if (cryoTurboSpd.value <= speedCryoTurboLow) {  // turbo not-spinning
                double pressDiffLim = pressDiffHighMin + Math.sqrt(cryoMainPrs.value / 780.0) * (20.0 - pressDiffHighMin);
                enable &=  prDiff <= pressDiffLim ; // low speed, high pressure limit (max 20 Torr)
            } else { 
                enable &=  prDiff <= pressDiffLow ; // high speed, low pressure limit (max 0.09 Torr)
            }
            // ensure inside >~ outside or both at very low pressure
            enable &= ((cryoMainPrs.value + cryoPdiffFloor) >  cryoPScale * cryoTurboPrs.value);

            turnOff = false;
            /* check foreline high */
            turnOff |= (cryoFlinePrs.value > pressForelineLow) && (cryoTurboSpd.value > speedCryoTurboLow);
            /* are Turbo operational limits exceeded */
            turnOff |= (cryoTurboTmp.value > cryoTurboTmp.highLimit);
            turnOff |= (cryoTurboPwr.value > cryoTurboPwr.highLimit);
            turnOff |= (int)cryoTurboStat.value == turboFailureCode;
            // reference note for later monitorControl.getPublishedPeriod()
            if (pumpCartDevc.getPumpSwitchState() == true && 
                pcCyclingStatus.valueCCSTimestamp != null) { // pumping intended on, channel is valid
                boolean turnOffFromPC = false;
               
                turnOffFromPC |= ((pcCyclingStatus.value < 1.0) || Double.isNaN(pcCyclingStatus.value)) &&
                    pcCyclingStatus.valueCCSTimestamp.getUTCDouble() > pumpCartDevc.getPumpSwitchStateTime().getUTCDouble();

                turnOffFromPC |= pcCyclingStatus.numErrors > MAX_PCPUMP_ERRORS;

                turnOff |= turnOffFromPC; 
                if (turnOffFromPC) {
                    LOG.fine("Pump cart caused gate valve to close");
                    if (vacPlutoDevc.getSwitchState(SwitchInfo.getSwitch(Switches.SW_CRYO_GATE_VALVE))==DeviceState.OPEN ) {
                        LOG.info("Pump cart causing gate valve to close with the following conditions: pumping = " +
                             (pcCyclingStatus.value > 0.0) + 
                             " #errors = " + pcCyclingStatus.numErrors + 
                             " pcCyclingStatus.valueCCSTimestamp > pumpCartDevc.getPumpSwitchStateTime: (" +
                             (pcCyclingStatus.valueCCSTimestamp.getUTCDouble() > 
                              pumpCartDevc.getPumpSwitchStateTime().getUTCDouble()) +")");
                    }
                }
            }
        }
        enable &= !turnOff;  // if forcing shut, also disable

        LOG.log(Level.FINE,
                "Gate valve decision inputs: " +
                " airCompressorOK "+airCompressorOK +
                ", cryoTurboAvail "+cryoTurboAvail +
                ", cryoFlinePrs.value "+cryoFlinePrs.value +
                ", pressForelineLow "+pressForelineLow +
                ", cryoTurboSpd.value "+cryoTurboSpd.value +
                ", speedCryoTurboLow "+speedCryoTurboLow +
                ", cryoTurboTmp.value "+cryoTurboTmp.value +
                ", cryoTurboTmp.highLimit "+cryoTurboTmp.highLimit +
                ", cryoTurboPwr.value "+cryoTurboPwr.value +
                ", cryoTurboPwr.highLimit "+cryoTurboPwr.highLimit +
                ", 1st or condition: cryoFlinePrs.value < pressForelineLow "+(cryoFlinePrs.value < pressForelineLow) +
                ", 2nd or condition: cryoTurboSpd.value <= speedCryoTurboLow "+(cryoTurboSpd.value <= speedCryoTurboLow) +
                 ", turnOff "+turnOff );
                // ", p_in > p_out with margin "+pDiffInOut +

        // update enable and off states of the switch and return
        return updateSwitch(Switches.SW_CRYO_GATE_VALVE, enable, turnOff,
                            vacPlutoDevc.getSwitchState(SwitchInfo.getSwitch(Switches.SW_CRYO_GATE_VALVE)), VacuumAlert.CRYO_GATE_CLOSED);
    }


    /**
     *  Updates the state of the cryo turbo pump vent valve.
          Enable|Disable conditions for venting:
          1.) to enable (AND of items): # could use more discussion
          - turbo is !(WAIT_INTLK|STARTING|NORMAL)
          - gate valve is shut
          - foreline valve is shut | pump-cart in-standby
          2.) Disable/turnOff if pump-cart offline
     * 
     *  @return  Whether the state changed
     */
    private boolean updateCryoTurboPumpVentValve()
    {

        boolean enable = true;    // enable gate valve to open
        boolean turnOff = false;  

        // NOT in active pumping state  
        enable &= !(
            cryoTurboStat.value==TwisTorr.PumpStatus.WAIT_INTLK.ordinal() ||
            cryoTurboStat.value==TwisTorr.PumpStatus.STARTING.ordinal() ||
            cryoTurboStat.value==TwisTorr.PumpStatus.AUTO_TUNING.ordinal() ||
            cryoTurboStat.value==TwisTorr.PumpStatus.NORMAL.ordinal());

        enable &= vacPlutoDevc.getSwitchState(SwitchInfo.getSwitch(Switches.SW_CRYO_GATE_VALVE)) == DeviceState.SHUT;
        
        // need isolation from hex or hex needs to be otherwise safe
        //
	Integer hx_turbo_stat = null;
	try {
	    if (isSwitchOn(Switches.SW_HX_TURBO_PUMP) != null) {
		hx_turbo_stat = (int)hxTurboStat.value;
	    }
	} catch (NullPointerException e) {
	    LOG.log(Level.SEVERE, "Error: Failed to retrieve HX turbo status {0}", e);
        }

	    
        enable &= ((DeviceState.valueOf(vacState.getHXForelineValveStatus()) == DeviceState.SHUT) || 
                        (vacState.getDeviceState(Switches.SW_HX_GATE_VALVE) == DeviceState.SHUT &&
			 (hx_turbo_stat != null && hx_turbo_stat == TwisTorr.PumpStatus.STOP.ordinal())));
        turnOff |= !enable;
        
        // pump cart offline
        turnOff |= Double.isNaN(pcCyclingStatus.value);  // isn't there another way to probe this?

        enable &= !turnOff;  // if forcing shut, also disable

        // if (false) {  // syntax for conditional string allocation
        //     LOG.log(Level.FINE, ()-> {return String.format("%s%s%s%s%s%s%s%s%s%s%s%s",
        //             String.format("cryoTurbo: Vent  valve decision inputs:\n"),....}}

        Boolean state = cryoTurboDevc.getVentValveOpenState();
        DeviceState ventDevState = state == null ? DeviceState.OFFLINE : state ? DeviceState.OPEN : DeviceState.SHUT;

        return updateSwitch(Switches.SW_CRYO_TURBO_VENT_VALVE, enable, turnOff, ventDevState, VacuumAlert.CRYO_TURBO_PUMP_VENT_CLOSED);
    }


    /**
     *  Updates the state of the cryo turbo pump.
     * 
     *  @return  Whether the state changed
     */
    private boolean updateCryoTurboPump()
    {
        boolean enable = false;
        if (!Double.isNaN(cryoTurboPrs.value) && !Double.isNaN(cryoFlinePrs.value) && !Double.isNaN(cryoTurboSpd.value)) {
            enable = cryoTurboPrs.value < pressTurboLow
                       && (cryoFlinePrs.value < pressForelineLow || cryoTurboSpd.value < speedCryoTurboLow);
        }

        LOG.log(Level.FINE,
                "turbo pump enable decision inputs: " +
                ", cryoTurboPrs.value "+cryoTurboPrs.value +
                ", cryoFlinePrs.value "+cryoFlinePrs.value +
                ", cryoTurboSpd.value "+cryoTurboSpd.value +
                ", pressForelineLow "+pressForelineLow +
                ", speedCryoTurboLow "+speedCryoTurboLow +
                ", cryoTurboPwr.highLimit "+cryoTurboPwr.highLimit +
                ", 1st or condition: cryoFlinePrs.value < pressForelineLow "+(cryoFlinePrs.value < pressForelineLow) +
                ", 2nd or condition: cryoTurboSpd.value <= speedCryoTurboLow "+(cryoTurboSpd.value <= speedCryoTurboLow) +
                 ", enable "+enable );
        
        return updateSwitch(Switches.SW_CRYO_TURBO_PUMP, enable, !enable, cryoTurboDevc.getDeviceState(), VacuumAlert.CRYO_TURBO_PUMP_STOPPED);
    }


    /**
     *  Updates the state of the cryo ion pumps.
     * 
     *  @return  Whether the state changed
     */
    private boolean updateCryoIonPumps()
    {
        boolean changed = false, enable = false, turnOff = false;
        if (!Double.isNaN(cryoMainPrs.value)) {
            enable = cryoMainPrs.value < pressIonEnable;
            if (cryoMainPrs.value >= pressIonOff) {
                if (cryoIonOverStartTime == 0) {
                    cryoIonOverStartTime = System.currentTimeMillis();
                }
                else {
                    turnOff = System.currentTimeMillis() - cryoIonOverStartTime >= DELAY_ION_OFF;
                }
            }
            else {
                cryoIonOverStartTime = 0;
            }
        }
        else {
            turnOff = true;
        }
        for (int sw : cryoIonPumps) {
            changed |= updateSwitch(sw, enable, turnOff, null, VacuumAlert.CRYO_ION_PUMPS_STOPPED);
        }
        return changed;
    }


    /**
     *  Updates the state of the hex turbo pump vent valve.
          Enable|Disable conditions for venting:
          1.) to enable (AND of items): # could use more discussion
          - turbo is !(WAIT_INTLK|STARTING|NORMAL)
          - gate valve is shut
          - foreline valve is shut | pump-cart in-standby
          2.) Disable/turnOff if pump-cart offline
     * 
     *  @return  Whether the state changed
     */
    private boolean updateHxTurboPumpVentValve()
    {

        boolean enable = true;    // enable gate valve to open
        boolean turnOff = false;  
        DeviceState ventDevState = DeviceState.OFFLINE;
        int sw = Switches.SW_HX_TURBO_VENT_VALVE;

        // NOT in active pumping state  
        enable &= !(
            hxTurboStat.value==TwisTorr.PumpStatus.WAIT_INTLK.ordinal() ||
            hxTurboStat.value==TwisTorr.PumpStatus.STARTING.ordinal() ||
            hxTurboStat.value==TwisTorr.PumpStatus.AUTO_TUNING.ordinal() ||
            hxTurboStat.value==TwisTorr.PumpStatus.NORMAL.ordinal());

        enable &= vacPlutoDevc.getSwitchState(SwitchInfo.getSwitch(Switches.SW_HX_GATE_VALVE)) == DeviceState.SHUT;
        
        // we need isolation from cryo or cryo needs to be otherwise safe
        // 
	Integer cryo_turbo_stat = null;
	try {
	    if (isSwitchOn(Switches.SW_CRYO_TURBO_PUMP) != null) {
		cryo_turbo_stat = (int)cryoTurboStat.value;
	    }
	} catch (NullPointerException e) {
	    LOG.log(Level.SEVERE, "Error: Failed to retrieve CRYO turbo status {0}", e);
        }


	enable &= ((DeviceState.valueOf(vacState.getCryoForelineValveStatus()) == DeviceState.SHUT) || 
                        (vacState.getDeviceState(Switches.SW_CRYO_GATE_VALVE) == DeviceState.SHUT &&
			 (cryo_turbo_stat != null && cryo_turbo_stat == TwisTorr.PumpStatus.STOP.ordinal())));

        turnOff |= !enable;
        
        // pump cart offline
        turnOff |= Double.isNaN(pcCyclingStatus.value);  // isn't there another way to probe this?

        enable &= !turnOff;  // if forcing shut, also disable

        // if (false) {  // syntax for conditional string allocation
        //     LOG.log(Level.FINE, ()-> {return String.format("%s%s%s%s%s%s%s%s%s%s%s%s",
        //             String.format("hxTurbo: Vent  valve decision inputs:\n"),....}}

        Boolean state = hxTurboDevc.getVentValveOpenState();
        ventDevState = state == null ? DeviceState.OFFLINE : state ? DeviceState.OPEN : DeviceState.SHUT;

        return updateSwitch(Switches.SW_HX_TURBO_VENT_VALVE, enable, turnOff, ventDevState, VacuumAlert.HX_TURBO_PUMP_VENT_CLOSED);
    }

    
    /**
     *  Updates the state of the hx gate valve by determining
     *  values and calling updateSwitch(<switch>, enable, turnOff,...)
     * 
     *  @return  Whether the state changed
     */
    private boolean updateHxGateValve()
    {
        boolean enable = false;  // enable gate valve to open
        boolean turnOff = true;  // force gate valve closed by default

        boolean hxTurboAvail = !Double.isNaN(hxMainPrs.value) &&
                                !Double.isNaN(hxTurboPrs.value) &&
                                !Double.isNaN(hxTurboSpd.value);

        boolean airCompressorOK = cmpAirPrs.value > cmpAirPrs.lowLimit &&
                                  cmpAirPrs.value < cmpAirPrs.highLimit;

        // ---- test turbo failure code
        if (airCompressorOK && testingModeOn) { // global minimal prerequisite to open
            turnOff = false;
            /* Turbo operational limits exceeded */
            turnOff |= (hxTurboTmp.value > hxTurboTmp.highLimit) || (hxTurboPwr.value > hxTurboPwr.highLimit) ;
            LOG.log(Level.FINE,"hxTurboStat = " + (int)hxTurboStat.value);
            /* Turbo operational limits exceeded */
            turnOff |= (hxTurboTmp.value > hxTurboTmp.highLimit);
            turnOff |= (hxTurboPwr.value > hxTurboPwr.highLimit);
            turnOff |= (int)hxTurboStat.value == turboFailureCode;
            if (turnOff) {
                LOG.log(Level.FINE,"Turbo failure check would have caused gate valve to close.");
            }
        }                

        if (hxTurboAvail && airCompressorOK) { // global minimal prerequisite to open
            enable = true;
            // pressure difference between turbo->hx
            double prDiff = Math.abs(hxTurboPrs.value - hxMainPrs.value);  // can't be NaN
            
            /* pressDiffLim scales from 20 at high pressure to smaller value at low pressure */
            if (hxTurboSpd.value <= speedHxTurboLow) {  // turbo not-spinning
                double pressDiffLim = pressDiffHighMin + Math.sqrt(hxMainPrs.value / 780.0) * (20.0 - pressDiffHighMin);
                enable &=  prDiff <= pressDiffLim ; // low speed, high pressure limit (max 20 Torr)
            } else { 
                enable &=  prDiff <= pressDiffLow ; // high speed, low pressure limit (max 0.09 Torr)
            }
            // ensure inside >~ outside or both at very low pressure
            enable &= ((hxMainPrs.value + hxPdiffFloor) >  hxPScale * hxTurboPrs.value);

            turnOff = false;
            /* foreline high */
            turnOff |= (hxFlinePrs.value > pressForelineLow) && (hxTurboSpd.value > speedHxTurboLow);
            /* Turbo operational limits exceeded */
            turnOff |= (hxTurboTmp.value > hxTurboTmp.highLimit);
            turnOff |= (hxTurboPwr.value > hxTurboPwr.highLimit);

            turnOff |= (int)hxTurboStat.value == turboFailureCode;

            if (pumpCartDevc.getPumpSwitchState() == true && 
                pcCyclingStatus.valueCCSTimestamp != null) { // pumping intended on, channel is valid
                boolean turnOffFromPC = false;
               
                turnOffFromPC |= ((pcCyclingStatus.value < 1.0) || Double.isNaN(pcCyclingStatus.value)) &&
                    pcCyclingStatus.valueCCSTimestamp.getUTCDouble() > pumpCartDevc.getPumpSwitchStateTime().getUTCDouble();

                turnOffFromPC |= pcCyclingStatus.numErrors > MAX_PCPUMP_ERRORS;

                turnOff |= turnOffFromPC; 
                if (turnOffFromPC) {
                    LOG.fine("Pump cart caused gate valve to close");
                    if (vacPlutoDevc.getSwitchState(SwitchInfo.getSwitch(Switches.SW_HX_GATE_VALVE))==DeviceState.OPEN ) {
                        LOG.info("Pump cart causing gate valve to close with the following conditions: pumping = " +
                             (pcCyclingStatus.value > 0.0) + 
                             " #errors = " + pcCyclingStatus.numErrors + 
                             " pcCyclingStatus.valueCCSTimestamp > pumpCartDevc.getPumpSwitchStateTime: (" +
                             (pcCyclingStatus.valueCCSTimestamp.getUTCDouble() > 
                              pumpCartDevc.getPumpSwitchStateTime().getUTCDouble()) +")");
                    }
                }
            }
        }
        enable &= !turnOff;  // if forcing shut, also disable

        LOG.log(Level.FINE,
                "Gate valve decision inputs: " +
                " airCompressorOK "+airCompressorOK +
                ", hxTurboAvail "+hxTurboAvail +
                ", hxFlinePrs.value "+hxFlinePrs.value +
                ", pressForelineLow "+pressForelineLow +
                ", hxTurboSpd.value "+hxTurboSpd.value +
                ", speedHxTurboLow "+speedHxTurboLow +
                ", hxTurboTmp.value "+hxTurboTmp.value +
                ", hxTurboTmp.highLimit "+hxTurboTmp.highLimit +
                ", hxTurboPwr.value "+hxTurboPwr.value +
                ", hxTurboPwr.highLimit "+hxTurboPwr.highLimit +
                ", 1st or condition: hxFlinePrs.value < pressForelineLow "+(hxFlinePrs.value < pressForelineLow) +
                ", 2nd or condition: hxTurboSpd.value <= speedHxTurboLow "+(hxTurboSpd.value <= speedHxTurboLow) +
                ", turnOff "+turnOff );
                // ", p_in > p_out with margin "+pDiffInOut +

        // update enable and off states of the switch and return
        return updateSwitch(Switches.SW_HX_GATE_VALVE, enable, turnOff,
                            vacPlutoDevc.getSwitchState(SwitchInfo.getSwitch(Switches.SW_HX_GATE_VALVE)), VacuumAlert.HX_GATE_CLOSED);
    }


    /**
     *  Updates the state of the HX turbo pump.
     * 
     *  enabled if: hxTurboPrs < 0.5 Torr && (hxFlinePrs < 1.0 Torr || hxTurboSpd < 10% full-speed)
     *  @return  Whether the state changed
     */
    private boolean updateHxTurboPump()
    {
        boolean enable = false;
        int sw = Switches.SW_HX_TURBO_PUMP;

        if (!Double.isNaN(hxTurboPrs.value) && !Double.isNaN(hxFlinePrs.value) && !Double.isNaN(hxTurboSpd.value)) {
            enable = hxTurboPrs.value < pressTurboLow
                       && (hxFlinePrs.value < pressForelineLow || hxTurboSpd.value < speedHxTurboLow);
        }


        LOG.log(Level.FINE,
                "turbo pump enable decision inputs: " +
                ", hxTurboPrs.value "+hxTurboPrs.value +
                ", hxFlinePrs.value "+hxFlinePrs.value +
                ", hxTurboSpd.value "+hxTurboSpd.value +
                ", pressForelineLow "+pressForelineLow +
                ", speedHxTurboLow "+speedHxTurboLow +
                ", hxTurboPwr.highLimit "+hxTurboPwr.highLimit +
                ", 1st or condition: hxFlinePrs.value < pressForelineLow "+(hxFlinePrs.value < pressForelineLow) +
                ", 2nd or condition: hxTurboSpd.value <= speedHxTurboLow "+(hxTurboSpd.value <= speedHxTurboLow) +
                 ", enable "+enable );

        
        return updateSwitch(sw, enable, !enable, hxTurboDevc.getDeviceState(), VacuumAlert.HX_TURBO_PUMP_STOPPED);
    }


    /**
     *  Updates the state of the HX ion pumps.
     * 
     *  @return  Whether the state changed
     */
    private boolean updateHxIonPumps()
    {
        boolean changed = false, enable = false, turnOff = false;
        if (!Double.isNaN(hxMainPrs.value)) {
            enable = hxMainPrs.value < pressIonEnable;
            if (hxMainPrs.value >= pressIonOff) {
                if (hxIonOverStartTime == 0) {
                    hxIonOverStartTime = System.currentTimeMillis();
                }
                else {
                    turnOff = System.currentTimeMillis() - hxIonOverStartTime >= DELAY_ION_OFF;
                }
            }
            else {
                hxIonOverStartTime = 0;
            }
        }
        else {
            turnOff = true;
        }
        for (int sw : hxIonPumps) {
            changed |= updateSwitch(sw, enable, turnOff, null, VacuumAlert.HX_ION_PUMPS_STOPPED);
        }
        return changed;
    }


    /*
     *  Updates the state of the CryoVacGauge Cold Cathode
     * 
     *  @return  Whether the state changed
    */
    private boolean updateCryoVacCC()
    {
        boolean changed = false, enable = false, turnOff = false;
        if (!Double.isNaN(cryoMainPrs.value)) {
            enable = cryoMainPrs.value <= pressCCEnable;
            if (cryoMainPrs.value >= pressCCOff) {
                if (cryoVacCCOverStartTime == 0) {
                    cryoVacCCOverStartTime = System.currentTimeMillis();
                }
                else {
                    turnOff = System.currentTimeMillis() - cryoVacCCOverStartTime >= delayCCOff;
                }
            }
            else {
                cryoVacCCOverStartTime = 0;
            }
        }
        else {
            turnOff = true;
        }
        changed |= updateSwitch(Switches.SW_CRYO_VAC_CC, enable, turnOff, null, VacuumAlert.CRYO_VAC_CC_STOPPED);
        return changed;
    }


    /*
     *  Updates the state of the TurboVacGauge Cold Cathode
     * 
     *  @return  Whether the state changed
    */
    private boolean updateCryoTurboCC()
    {
        boolean changed = false, enable = false, turnOff = false;
        if (!Double.isNaN(cryoTurboPrs.value)) {
            enable = cryoTurboPrs.value < pressCCEnable;
            if (cryoTurboPrs.value >= pressCCOff) {
                if (cryoTurboCCOverStartTime == 0) {
                    cryoTurboCCOverStartTime = System.currentTimeMillis();
                }
                else {
                    turnOff = System.currentTimeMillis() - cryoTurboCCOverStartTime >= delayCCOff;
                }
            }
            else {
                cryoTurboCCOverStartTime = 0;
            }
        }
        else {
            turnOff = true;
        }
        changed |= updateSwitch(Switches.SW_CRYO_TURBO_CC, enable, turnOff, null, VacuumAlert.CRYO_TURBO_CC_STOPPED);
        return changed;
    }


    /*
     *  Updates the state of the HxVacGauge Cold Cathode
     * 
     *  @return  Whether the state changed
    */
    private boolean updateHxVacCC()
    {
        boolean changed = false, enable = false, turnOff = false;
        if (!Double.isNaN(hxMainPrs.value)) {
            enable = hxMainPrs.value < pressCCEnable;
            if (hxMainPrs.value >= pressCCOff) {
                if (hxVacCCOverStartTime == 0) {
                    hxVacCCOverStartTime = System.currentTimeMillis();
                }
                else {
                    turnOff = System.currentTimeMillis() - hxVacCCOverStartTime >= delayCCOff;
                }
            }
            else {
                hxVacCCOverStartTime = 0;
            }
        }
        else {
            turnOff = true;
        }
        changed |= updateSwitch(Switches.SW_HX_VAC_CC, enable, turnOff, null, VacuumAlert.HX_VAC_CC_STOPPED);
        return changed;
    }


    /*
     *  Updates the state of the TurboVacGauge Cold Cathode
     * 
     *  @return  Whether the state changed
    */
    private boolean updateHxTurboCC()
    {
        boolean changed = false, enable = false, turnOff = false;
        if (!Double.isNaN(hxTurboPrs.value)) {
            enable = hxTurboPrs.value < pressCCEnable;
            if (hxTurboPrs.value >= pressCCOff) {
                if (hxTurboCCOverStartTime == 0) {
                    hxTurboCCOverStartTime = System.currentTimeMillis();
                }
                else {
                    turnOff = System.currentTimeMillis() - hxTurboCCOverStartTime >= delayCCOff;
                }
            }
            else {
                hxTurboCCOverStartTime = 0;
            }
        }
        else {
            turnOff = true;
        }
        changed |= updateSwitch(Switches.SW_HX_TURBO_CC, enable, turnOff, null, VacuumAlert.HX_TURBO_CC_STOPPED);
        return changed;
    }


    /**
     *  Updates the state of the UT switches (scroll pumps & interstitial valves).
     * 
     *  @return  Whether the state changed
     */
    private boolean updateUtSwitches()
    {

        boolean changed = false;

        // the following two lines are only needed while testing actions
        speedCryoTurboLow = speedFractTurboLow * SPEED_CRYO_TURBO_MAX;
        speedHxTurboLow = speedFractTurboLow * SPEED_HX_TURBO_MAX;

        for (int sw : utSwitches) {
            boolean enable = true;
            boolean turnOff = false;
            DeviceState devState = null;
            VacuumAlert al = null;

            if ( sw == Switches.SW_CRYO_FLINE_VALVE) {
                devState = DeviceState.OFFLINE;
                try {
                    devState = DeviceState.valueOf(vacState.getCryoForelineValveStatus());
                } catch(Exception e) {
                    LOG.log(Level.SEVERE, "Error getting cryo foreline valve state: ", e);
                }

                if (pcInletPressure.value < 750.0) {
                    enable = Math.abs(pcInletPressure.value - cryoFlinePrs.value)<maxInletFlinePressDiff() ;
                } else {
                    enable = cryoFlinePrs.value > (750.0-maxInletFlinePressDiff());
                }


                if ((int)cryoTurboStat.value == turboFailureCode) { // known to be redundant with alert action
                    al = VacuumAlert.CRYO_FLINE_FORCED_CLOSED_BY_TURBO;
                    turnOff = true;
                }

                if (isAlarmRaised(VacuumAlert.CRYO_GATE_POSN_ERROR)) { // known to be redundant with alert action
                    al = VacuumAlert.CRYO_FLINE_FORCED_CLOSED_BY_GV;
                    turnOff = true;
                }

                if ((cryoFlinePrs.value > pressForelineLow) && (cryoTurboSpd.value > speedCryoTurboLow || testingModeOn)) {
                    al = VacuumAlert.CRYO_FLINE_FORCED_CLOSED_BY_FLPRS;
                    turnOff = true;
                }



            } else if (sw == Switches.SW_HX_FLINE_VALVE) {
                devState = DeviceState.OFFLINE;
                try { // fix, this does not throw -- is it wrong?
                    devState = DeviceState.valueOf(vacState.getHXForelineValveStatus());
                } catch(Exception e){
                    LOG.log(Level.SEVERE, "Error getting HX foreline valve state: ", e);
                }
                if (pcInletPressure.value < 750.0) {
                    enable = Math.abs(pcInletPressure.value - hxFlinePrs.value)<maxInletFlinePressDiff() ;
                } else {
                    enable = cryoFlinePrs.value > (750.0-maxInletFlinePressDiff());
                }

                if ((int)hxTurboStat.value == turboFailureCode) {
                    al = VacuumAlert.HX_FLINE_FORCED_CLOSED_BY_TURBO;
                    turnOff = true;
                }

                if (isAlarmRaised(VacuumAlert.HX_GATE_POSN_ERROR)) {
                    al = VacuumAlert.HX_FLINE_FORCED_CLOSED_BY_GV;
                    turnOff = true;
                }

                if ((hxFlinePrs.value > pressForelineLow) && (hxTurboSpd.value > speedHxTurboLow || testingModeOn)) {
                    al = VacuumAlert.HX_FLINE_FORCED_CLOSED_BY_FLPRS;
                    turnOff = true;
                }

            }
            changed |= updateSwitch(sw, enable, turnOff, devState, al);
        }
        return changed;
    }

    @Command(type=Command.CommandType.QUERY, description = "determines what is the allowable pressure difference btwn the foreline and the PC inlet pressure")
    public double maxInletFlinePressDiff() {
        // disable the opening of the valve if the pressure on the foreline side is significantly different from the pump cart inlet pressure
        return (pcPdiffFloor + pcPdiffScaleFactor * pcInletPressure.value) ;
    }


    /**
     *  Updates the state of a switch.
     * 
     *  @param sw        The switch ID
     *  @param enable    Whether to enable it
     *  @param turnOff   Whether to turn it off
     *  @param devState  The state of the switch's device (turbo pump only)
     *  @param alert     The alert to raise/lower if on/off state change was forced
     *  @return  Whether the state changed
     */
    private boolean updateSwitch(int sw, boolean enable, boolean turnOff, DeviceState devState, VacuumAlert alert)
    {
        boolean changed = false;
        SwitchState oldState = vacState.getSwitchState(sw);
        if (turnOff && oldState == SwitchState.ON) {
            try {
                setSwitch(sw, false);
                raiseAlarm(alert, "Switch was forced to open", null);
            }
            catch (VacuumException e) {
                LOG.log(Level.SEVERE, "Error setting switch: {0}", e);
            }
        }
        Boolean isOn = isSwitchOn(sw);  // reads device for turbo main and vent valve switches
        SwitchState state = isOn != null ? isOn ? SwitchState.ON : SwitchState.OFF : SwitchState.OFFLINE;
        SwitchEnable enabled = enable ? SwitchEnable.ON : SwitchEnable.OFF;
        if (state != oldState || enabled != vacState.getSwitchEnable(sw)) {
            vacState.setSwitchState(sw, state);
            vacState.setSwitchEnable(sw, enabled);
            changed = true;
            if (enable && alert != null && isAlarmRaised(alert)) {
                lowerAlarm(alert, "Switch has become re-enabled", null);
            }
        }
        DeviceState oldDevState = vacState.getDeviceState(sw);
        if (devState != oldDevState) {

            // ------ Notifications/alarms for gate valve transit and error states ---------
            if (sw == Switches.SW_CRYO_GATE_VALVE || sw == Switches.SW_HX_GATE_VALVE) {
                if (oldDevState == DeviceState.ERROR) {
                    lowerAlarm(sw == Switches.SW_CRYO_GATE_VALVE ? VacuumAlert.CRYO_GATE_POSN_ERROR : VacuumAlert.HX_GATE_POSN_ERROR,
                               "Gate valve no longer reading both open and closed", null);
                }
                else if (oldDevState == DeviceState.TRANSIT) {
                    if (gateTransitStart.get(sw) == null) {
                        lowerAlarm(sw == Switches.SW_CRYO_GATE_VALVE ? VacuumAlert.CRYO_GATE_IN_TRANSIT : VacuumAlert.HX_GATE_IN_TRANSIT,
                                   "Gate valve no longer in too-long transition", null);
                    }
                }
                if (devState == DeviceState.ERROR) {
                    raiseAlarm(sw == Switches.SW_CRYO_GATE_VALVE ? VacuumAlert.CRYO_GATE_POSN_ERROR : VacuumAlert.HX_GATE_POSN_ERROR,
                               "Gate valve reading both open and closed", null);
                }
                else if (devState == DeviceState.TRANSIT) {
                    gateTransitStart.put(sw, System.currentTimeMillis());
                }
            }
            // -----------------------------------------------------------------------------

            vacState.setDeviceState(sw, devState);
            changed = true;
        }
        if (devState == DeviceState.TRANSIT) {

            // -----------------------------------------------------------------------------
            if (sw == Switches.SW_CRYO_GATE_VALVE || sw == Switches.SW_HX_GATE_VALVE) {
                Long transitTime = gateTransitStart.get(sw);
                if (transitTime != null
                    && (System.currentTimeMillis() - transitTime) > (sw == Switches.SW_CRYO_GATE_VALVE ? tranTimeCryoGate : tranTimeHxGate)) {
                    raiseAlarm(sw == Switches.SW_CRYO_GATE_VALVE ? VacuumAlert.CRYO_GATE_IN_TRANSIT : VacuumAlert.HX_GATE_IN_TRANSIT,
                               "Gate valve in transition for too long", null);
                    gateTransitStart.remove(sw);
                }
            }
            // -----------------------------------------------------------------------------

        }
        return changed;
    }


    /**
     *  Checks whether the vacuum is good enough to allow refrigeration.
     */
    private void checkRefrigVacuum() {
        boolean haveAlarm = isAlarmRaised(VacuumAlert.REFRIG_NOT_PERMITTED);
        if (cryoMainPrs.value < pressRefrigOk && hxMainPrs.value < pressRefrigOk) {  // Fails if either is NaN
            if (haveAlarm) {
                lowerAlarm(VacuumAlert.REFRIG_NOT_PERMITTED, "Cryo & HX vacuums are good (< " + pressRefrigOk + ")",
                           RefrigAction.Action.VACUUM_INSUFFICIENT);
            }
            vacBadTime = 0;
        }
        else {
            long time = System.currentTimeMillis();
            if (vacBadTime == 0) {
                vacBadTime = time;
            }
            else {
                if (time - vacBadTime >= DELAY_VACUUM_BAD) {
                    if (!haveAlarm) {
                        raiseAlarm(VacuumAlert.REFRIG_NOT_PERMITTED, "Cryo or HX vacuum is bad (>= " + pressRefrigOk + ")",
                                   RefrigAction.Action.VACUUM_INSUFFICIENT);
                    }
                }
            }
        }
    }


    /**
     *  Updates the summary cryo vacuum state.
     * 
     *  @return  Whether the state changed
     */
    private boolean updateCryoState() {
        boolean changed = false;
        CryoVacuumState cvState;
        if (Double.isNaN(cryoMainPrs.value)) {
            cvState = CryoVacuumState.UNKNOWN;
        }
        else if (cryoMainPrs.value <= pressVacuum) {
            cvState = CryoVacuumState.VACUUM;
        }
        else if (vacState.getSwitchState(Switches.SW_CRYO_ION_PUMP1) == SwitchState.ON) {
            cvState = CryoVacuumState.ION_ON;
        }
        else if (vacState.getSwitchEnable(Switches.SW_CRYO_ION_PUMP1) == SwitchEnable.ON) {
            cvState = CryoVacuumState.ION_OFF;
        }
        else if (vacState.getSwitchState(Switches.SW_CRYO_TURBO_PUMP) == SwitchState.ON) {
            cvState = CryoVacuumState.TURBO_ON;
        }
        else if (vacState.getSwitchEnable(Switches.SW_CRYO_TURBO_PUMP) == SwitchEnable.ON) {
            cvState = CryoVacuumState.TURBO_OFF;
        }
        else if (cryoMainPrs.value < PRESS_ATMOS) {
            cvState = CryoVacuumState.FORELINE;
        }
        else {
            cvState = CryoVacuumState.OFF;
        }
        if (cvState != vacState.getCryoVacuumState()) {
            vacState.setCryoVacuumState(cvState);
            stateService.updateAgentState(cvState);
            changed = true;
        }
        return changed;
    }


    /**
     *  Updates the summary HX vacuum state.
     * 
     *  @return  Whether the state changed
     */
    private boolean updateHxState() {
        boolean changed = false;
        HxVacuumState hvState;
        if (Double.isNaN(hxMainPrs.value)) {
            hvState = HxVacuumState.UNKNOWN;
        }
        else if (hxMainPrs.value <= pressVacuum) {
            hvState = HxVacuumState.VACUUM;
        }
        else if (vacState.getSwitchState(Switches.SW_HX_ION_PUMP1) == SwitchState.ON) {
            hvState = HxVacuumState.ION_ON;
        }
        else if (vacState.getSwitchEnable(Switches.SW_HX_ION_PUMP1) == SwitchEnable.ON) {
            hvState = HxVacuumState.ION_OFF;
        }
        else if (vacState.getSwitchState(Switches.SW_HX_TURBO_PUMP) == SwitchState.ON) {
            hvState = HxVacuumState.TURBO_ON;
        }
        else if (vacState.getSwitchEnable(Switches.SW_HX_TURBO_PUMP) == SwitchEnable.ON) {
            hvState = HxVacuumState.TURBO_OFF;
        }
        else if (hxMainPrs.value < PRESS_ATMOS) {
            hvState = HxVacuumState.FORELINE;
        }
        else {
            hvState = HxVacuumState.OFF;
        }
        if (hvState != vacState.getHxVacuumState()) {
            vacState.setHxVacuumState(hvState);
            stateService.updateAgentState(hvState);
            changed = true;
        }
        return changed;
    }


    /**
     *  Raises an alarm.
     *
     *  @param  alert  The vacuum alert to raise to alarm state
     *  @param  cond   The alert condition
     *  @param  action  The refrig action to perform, or null if none
     */
    private void raiseAlarm(VacuumAlert alert, String cond, RefrigAction.Action action)
    {
        try {
            Alert al = alert.getAlert();
            if (action != null) {
                RefrigAction.addData(al, action);
            }
            alertService.raiseAlert(al, AlertState.ALARM, cond);
            activeAlarmMap.put(alert.getId(), true);
        } catch (NullPointerException e) {
                LOG.log(Level.SEVERE, "Error: Raise alert failure {0}", e);
        }
    }


    /**
     *  Lowers an alarm.
     *
     *  @param  alert  The vacuum alert to lower to normal state
     *  @param  cond   The alert condition
     *  @param  action  The refrig action to perform, or null if none
     */
    private void lowerAlarm(VacuumAlert alert, String cond, RefrigAction.Action action)
    {
        Alert al = alert.getAlert();
        if (action != null) {
            RefrigAction.addData(al, action);
        }
        alertService.raiseAlert(al, AlertState.NOMINAL, cond);
        activeAlarmMap.put(alert.getId(), false);
    }


     /**
     *  Checks whether an alert is in alarm state.
     *
     *  @param  alert  The vacuum alert to check
     *  @return  Whether it has been raised
     */
    private boolean isAlarmRaised(VacuumAlert alert)
    {
        return activeAlarmMap.get(alert.getId()) == Boolean.TRUE;
    }


    /**
     *  Alert event handler.
     *
     *  Resets PLC latch when corresponding alert is cleared.
     *
     *  @param  event  The alert event
     */
    @Override
    public void onAlert(AlertEvent event)
    {
        if ( event.getType() == AlertEvent.AlertEventType.ALERT_RAISED) {
            Alert raisedAlert = event.getAlert();
            String alertId = raisedAlert.getAlertId();

            // Turbo power alert actions
            
            if ( alertId.equals(VacuumAlert.CRYO_TURBO_PUMP_POWER_HIGH.getAlert().getAlertId()) ) {
                if ( event.getLevel() == AlertState.ALARM ) {
                    updateSwitch(Switches.SW_CRYO_TURBO_PUMP, false, true, cryoTurboDevc.getDeviceState(), VacuumAlert.CRYO_TURBO_PUMP_STOPPED);
                    LOG.info("Stopping the cryo turbo pump due to high turbo power alert from monitoring!");
                }
            }
            if ( alertId.equals(VacuumAlert.HX_TURBO_PUMP_POWER_HIGH.getAlert().getAlertId()) ) {
                if ( event.getLevel() == AlertState.ALARM ) {
                    updateSwitch(Switches.SW_HX_TURBO_PUMP, false, true, hxTurboDevc.getDeviceState(), VacuumAlert.HX_TURBO_PUMP_STOPPED);
                    LOG.info("Stopping the hx turbo pump due to high turbo power alert from monitoring!");
                }
            }

            // Turbo temp alert actions

            if ( alertId.equals(VacuumAlert.CRYO_TURBO_PUMP_TEMP_HIGH.getAlert().getAlertId()) ) {
                if ( event.getLevel() == AlertState.ALARM ) {
                    updateSwitch(Switches.SW_CRYO_TURBO_PUMP, false, true, cryoTurboDevc.getDeviceState(), VacuumAlert.CRYO_TURBO_PUMP_STOPPED);
                    LOG.info("Stopping the cryo turbo pump due to high turbo temp alert from monitoring!");
                }
            }
            if ( alertId.equals(VacuumAlert.HX_TURBO_PUMP_TEMP_HIGH.getAlert().getAlertId()) ) {
                if ( event.getLevel() == AlertState.ALARM ) {
                    updateSwitch(Switches.SW_HX_TURBO_PUMP, false, true, hxTurboDevc.getDeviceState(), VacuumAlert.HX_TURBO_PUMP_STOPPED);
                    LOG.info("Stopping the hx turbo pump due to high turbo temp alert from monitoring!");
                }
            }

            // Turbo pump failure alert actions

            if ( alertId.equals(VacuumAlert.CRYO_TURBO_PUMP_FAIL.getAlert().getAlertId()) ) {
                if ( event.getLevel() == AlertState.ALARM ) {
                    // Shut the gate valve
                    updateSwitch(SwitchInfo.getSwitch(Switches.SW_CRYO_GATE_VALVE), false, true, vacPlutoDevc.getSwitchState(SwitchInfo.getSwitch(Switches.SW_CRYO_GATE_VALVE)), null);
                    // Shut the foreline valve
                    updateSwitch(SwitchInfo.getSwitch(Switches.SW_CRYO_FLINE_VALVE), false, true, DeviceState.valueOf(vacState.getCryoForelineValveStatus()), null);
                    LOG.info("Shutting the cryo vacuum valve due to bad turbo alert from monitoring!");
                }
            }
            if ( alertId.equals(VacuumAlert.HX_TURBO_PUMP_FAIL.getAlert().getAlertId()) ) {
                if ( event.getLevel() == AlertState.ALARM ) {
                    // Shut the gate valve
                    updateSwitch(SwitchInfo.getSwitch(Switches.SW_HX_GATE_VALVE), false, true, vacPlutoDevc.getSwitchState(SwitchInfo.getSwitch(Switches.SW_HX_GATE_VALVE)), null);
                    // Shut the foreline valve
                    updateSwitch(SwitchInfo.getSwitch(Switches.SW_HX_FLINE_VALVE), false, true, DeviceState.valueOf(vacState.getHXForelineValveStatus()), null);
                    LOG.info("Shutting the hex vacuum valves due to bad turbo alert from monitoring!");                    
                }
            }
            
        }
        if (event.getType() != AlertEvent.AlertEventType.ALERT_CLEARED) return;
        for (String id : event.getClearedIds()) {
            Integer cond = revLatchAlertMap.get(id);
            if (cond != null) {
                try {
                    clearLatch(cond);
                }
                catch (VacuumException e) {
                    LOG.log(Level.SEVERE, "Error clearing latched PLC condition ({0}): {1}", new Object[]{cond, e});
                }
            }
        }
    }


    /**
     *  Publishes the state of the vacuum system.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    private void publishState()
    {
        publishSubsystemDataOnStatusBus(new KeyValueData(VacSysState.KEY, getVacuumState()));
    }

    @Command(type=Command.CommandType.ACTION, description="test cryo gate valve position")
    public void test_cryo_gate_posn_error() throws VacuumException
    {
        raiseAlarm(VacuumAlert.CRYO_GATE_POSN_ERROR, "TESTing cryo gate posn error", null);
    }

    @Command(type=Command.CommandType.ACTION, description="test hex gate valve position")
    public void test_hex_gate_posn_error() throws VacuumException
    {
        raiseAlarm(VacuumAlert.HX_GATE_POSN_ERROR, "TESTing hex gate posn error", null);
    }

    @Command(type=Command.CommandType.ACTION, description="test turbo failure arg=turbo code")
    public void test_turbo_failure(int code)
    {
        turboFailureCode = code;
    }

    @Command(type=Command.CommandType.ACTION, description="set the vacuum subsystem testing mode on flag")
        public void set_testing_mode_on(boolean on)
    {
        testingModeOn = on;
    }

    class MyCommandOriginator implements CommandOriginator {

        @Override
        public void processAck(CommandAck ack) {
            System.out.println("Ack!!!");
        }

        @Override
        public void processNack(CommandNack nack) {
            System.out.println("Nack!!!");
        }

        @Override
        public void processResult(CommandResult result) {
            System.out.println("Got result!!!" + result);
        }
    }
}
