package org.lsst.ccs.subsystem.vacuum;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.drivers.auxelex.Pdu24VD;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.vacuum.constants.Devices;

/**
 *  Interface to a SLAC PDU.
 *
 *  This class is designed only to be extended, not instantiated itself
 *
 *  @author Owen Saxton
 */
public class VacPduDevice extends Device implements SwitchDevice {

    /**
     *  Constants.
     */
    public static final int
        CHAN_CRYO_SCROLL = Pdu24VD.CHAN_CRYO_SCROLL,
        CHAN_HEX_SCROLL = Pdu24VD.CHAN_HEX_SCROLL,
        CHAN_INST_SCROLL = Pdu24VD.CHAN_INST_SCROLL;

    protected static final int
        MON_TYPE_VOLTAGE = 0,
        MON_TYPE_CURRENT = 1;

    private static final Map<String, Integer> mTypeMap = new HashMap<>();
    static {
        mTypeMap.put("VOLTAGE", MON_TYPE_VOLTAGE);
        mTypeMap.put("CURRENT", MON_TYPE_CURRENT);
    }
    private static final Set<Integer> validChans = new HashSet<>();
    static {
        validChans.add(CHAN_CRYO_SCROLL);
        validChans.add(CHAN_HEX_SCROLL);
        validChans.add(CHAN_INST_SCROLL);
    }

    protected static final Logger LOG = Logger.getLogger(VacPduDevice.class.getName());

    /**
     *  Data fields.
     */
    @ConfigurationParameter(category="Vacuum", isFinal=true)
    protected volatile Integer node;

    private final Pdu24VD pdu = new Pdu24VD();
    private boolean initError = false;


    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice() {
        fullName = "24V dirty PDU (" + node + ")";
    }


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        try {
            pdu.open(node, Pdu24VD.PORT_1);
            setOnline(true);
            initSensors();
            LOG.log(Level.INFO, "Connected to {0}", fullName);
            initError = false;
        }
        catch (DriverException e) {
            if (!initError) {
                LOG.log(Level.SEVERE, "Error connecting to {0}: {1}", new Object[]{fullName, e});
                initError = true;
            }
        }
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        try {
            pdu.close();
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error disconnecting from {0}:{1}", new Object[]{fullName, e});
        }
    }


    /**
     *  Checks a monitoring channel's parameters for validity.
     *
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string
     *  @return  Two-element array containing the encoded type [0] and subtype [1] 
     *  @throws  Exception  If parameters are invalid
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        Integer mType = mTypeMap.get(type.toUpperCase());
        if (mType == null) {
            ErrorUtils.reportChannelError(LOG, name, "type", type);
        }
        if (!validChans.contains(hwChan)) {
            ErrorUtils.reportChannelError(LOG, name, "hwChan", hwChan);
        }
        return new int[]{mType, 0};
    }


    /**
     *  Reads a monitoring channel.
     *
     *  @param  hwChan  The hardware channel number
     *  @param  type    The encoded channel type
     *  @return  The read value
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = Double.NaN;
        String item = null;
        if (isOnline()) {
            try {
                switch (type) {
                case MON_TYPE_VOLTAGE:
                    item = "voltage";
                    value = pdu.readVoltage(hwChan);
                    break;

                case MON_TYPE_CURRENT:
                    item = "current";
                    value = pdu.readCurrent(hwChan);
                    break;
                }
            }
            catch (DriverException e) {
                LOG.log(Level.SEVERE, "Error reading {0} {1}: {2}", new Object[]{fullName, item, e});
                setOnline(false);
            }
        }
        return value;
    }


    /**
     *  Gets the switch device type
     * 
     *  @return  The switch device type
     */
    @Override
    public int getSwitchDevice()
    {
        return Devices.DEVC_PDU;
    }


    /**
     *  Turns a PDU channel on or off.
     *
     *  @param  chan  The channel number
     *  @param  on  Whether to turn on or off
     *  @throws  DriverException
     */
    @Override
    public void setSwitch(int chan, boolean on) throws DriverException
    {
        if (!validChans.contains(chan)) {
            throw new RuntimeException("Programming error!!  Invalid channel number: " + chan);
        }
        if (on) {
            pdu.powerOn(chan);
        }
        else {
            pdu.powerOff(chan);
        }
    }


    /**
     *  Gets the state of a PDU channel.
     *
     *  @param  chan  The channel number
     *  @return  Whether the channel is on or off
     */
    @Override
    public Boolean isSwitchOn(int chan)
    {
        if (!isOnline() || !validChans.contains(chan)) return null;
        try {
            return pdu.isPowered(chan);
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error getting state for {0} channel {1}: {2}", new Object[]{path, chan, e});
            setOnline(false);
            return null;
        }
    }

}
