package org.lsst.ccs.subsystem.vacuum;

import org.lsst.ccs.subsystem.common.devices.pluto.PlutoDevice;

/**
 *  Handles limited access to the protection system Pluto PLC.
 *
 *  @author Owen Saxton
 */
public class MpmPlutoDevice extends PlutoDevice {

    /**
     *  Constants.
     */
    public static final int
        SW_BLOCK_COLD_REFG  = 0,
        SW_BLOCK_CRYO_REFG  = 1,
        NUM_SWITCHES        = 2;

    private static final int
        NUM_AREAS = 30,
        SWDI_ON_BIT = 0,
        SWDI_OFF_BIT = 1,
        SWDI_READ_AREA = 2,
        SWDI_READ_BIT = 3;

    /**
     *  Private lookup maps, etc.
     */
    private static final int[][] switches = new int[NUM_SWITCHES][];
    static {                              // ON bit, OFF bit, read area, read bit 
        switches[SW_BLOCK_COLD_REFG] = new int[]{20, 21, 17, 6};
        switches[SW_BLOCK_CRYO_REFG] = new int[]{36, 37, 27, 6};
    }


    /**
     *   Constructor.
     */
    public MpmPlutoDevice()
    {
        super(NUM_AREAS);
    }


    /**
     *  Sets a switch on or off.
     *
     *  For the MPM Pluto, this is implemented as a pair of push buttons,
     *  one for on, one for off.
     *
     *  @param  sw  The switch number.
     *  @param  on  The on state to set: true or false
     */
    public void setSwitch(int sw, boolean on)
    {
        if (isOnline()) {
            int bitNum = switches[sw][on ? SWDI_ON_BIT : SWDI_OFF_BIT];
            toggleBit(bitNum / 16, bitNum & 0x0f);
        }
    }


    /**
     *  Gets the on state of a switch.
     *
     *  The state is not the state of the bit that was toggled, but is read back
     *  either directly from the controlled hardware, or from the PLC output line.
     *
     *  @param  sw  The switch number.
     *  @return  Whether the switch is on
     */
    public Boolean isSwitchOn(int sw)
    {
        int[] swData = switches[sw];
        Integer value = readAddBit(swData[SWDI_READ_AREA], swData[SWDI_READ_BIT] + 16);
        return value != null ? value != 0 : null;
    }

}
