package org.lsst.ccs.subsystem.vacuum;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.mks.Model9XX;
import org.lsst.ccs.drivers.mks.Model9XX.Direction;
import org.lsst.ccs.drivers.mks.Model9XX.Enable;
import org.lsst.ccs.drivers.mks.Model9XX.Gas;
import org.lsst.ccs.drivers.mks.Model9XX.Sensor;
import org.lsst.ccs.drivers.mks.Model9XX.Unit;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.vacuum.config.VacuumConfig;
import org.lsst.ccs.subsystem.vacuum.constants.RelayAlert;

/**
 *  Interface to an MKS 972/974/925 vacuum gauge.
 *
 *  @author Owen Saxton
 */
public class Mks9xxDevice extends Device implements ClearAlertHandler {

    /**
     *  Data fields.
     */
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;

    @ConfigurationParameter(name="devcId", category=VacuumConfig.CRYO, isFinal=true)
    private volatile String devcId;
    @ConfigurationParameter(name="busAddr", category=VacuumConfig.CRYO, isFinal=true)
    private volatile int busAddr = Model9XX.DEFAULT_ADDRESS;
    @ConfigurationParameter(name="relayTrip", category=VacuumConfig.CRYO, maxLength=Model9XX.RELAY_MAX, isFinal=true)
    private volatile double[] relayTrip;

    private final Model9XX.ConnType connType = Model9XX.ConnType.SERIAL;
    private final int baudRate = Model9XX.DEFAULT_BAUDRATE;

    private static final Logger LOG = Logger.getLogger(Mks9xxDevice.class.getName());
    private final Model9XX mks = new Model9XX();
    private int errorCount = 0;
    private boolean initError = false;
    private final Map<String, Boolean> activeAlerts = new HashMap<>();


    /**
     *  Performs configuration.
     */
    @Override
    public void initDevice() {
        if (devcId == null) {
            ErrorUtils.reportConfigError(LOG, name, "devcId", "is missing");
        }
        fullName = "MKS model 9XX (" + devcId + ":" + busAddr + ")";
    }


    /**
     *  Performs full initialization.
     */
    @Override
    public void initialize()
    {
        try {
            mks.setRetryLimit(3);
            mks.open(connType, devcId, baudRate, busAddr);
            mks.setLock(false);
            mks.setPressureUnit(Model9XX.Unit.TORR);
            mks.setLock(true);
            errorCount = 0;
            setOnline(true);
            initSensors();
            checkRelays();
            LOG.log(Level.INFO, "Connected to {0}: model = {1}, serial no. = {2}",
                    new Object[]{fullName, mks.getModel(), mks.getSerialNumber()});
            initError = false;
        }
        catch (DriverException e) {
            if (!initError) {
                LOG.log(Level.SEVERE, "Error connecting to {0}: {1}", new Object[]{fullName, e});
                initError = true;
            }
            try {
                mks.close();
            }
            catch (DriverException ce) {
                // Here if device wasn't opened
            }
        }
    }


    /**
     *  Closes the connection.
     */
    @Override
    public void close()
    {
        try {
            mks.close();
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error disconnecting from {0}: {1}", new Object[]{fullName, e});
        }
    }


    /**
     *  Reads a monitoring channel.
     *
     *  @param  hwChan  The hardware channel number
     *  @param  type    The encoded channel type
     *  @return  The read value
     */
    @Override
    public double readChannel(int hwChan, int type)
    {
        double value = Double.NaN;
        if (isOnline()) {
            try {
                value = mks.readPressure(Model9XX.Sensor.COMB4);
                errorCount = 0;
            }
            catch (DriverException e) {
                LOG.log(Level.SEVERE, "Error reading pressure from {0}: {1}", new Object[]{fullName, e});
                if (++errorCount >= 5) {
                    setOnline(false);
                }
            }
        }
        return value;
    }


    /**
     *  Checks that relays are set correctly
     */
    private void checkRelays() throws DriverException
    {
        for (int rNum = 1; rNum <= relayTrip.length; rNum++) {
            Model9XX.Direction dirn = mks.getRelayDirection(rNum);
            if (dirn != Model9XX.Direction.BELOW) {
                raiseAlarm(RelayAlert.RELAY_DIRN_BAD.getAlert(name, rNum),
                           String.format("%s relay %s direction is %s", name, rNum, dirn));
            }
            else {
                lowerAlarm(RelayAlert.RELAY_DIRN_BAD.getAlert(name, rNum),
                           String.format("%s relay %s direction is %s", name, rNum, dirn));
            }
            Model9XX.Enable enable = mks.getRelayEnable(rNum);
            if (enable == Model9XX.Enable.OFF) {
                raiseAlarm(RelayAlert.RELAY_NOT_ENAB.getAlert(name, rNum),
                           String.format("%s relay %s enable is %s", name, rNum, enable));
            }
            else {
                lowerAlarm(RelayAlert.RELAY_NOT_ENAB.getAlert(name, rNum),
                           String.format("%s relay %s enable is %s", name, rNum, enable));
            }
            double trip = mks.getRelayTrip(rNum);
            if (trip != relayTrip[rNum - 1]) {
                raiseAlarm(RelayAlert.RELAY_TRIP_BAD.getAlert(name, rNum),
                           String.format("%s relay %s trip point (%.5g) should be %.5g", name, rNum, trip, relayTrip[rNum - 1]));
            }
            else {
                lowerAlarm(RelayAlert.RELAY_TRIP_BAD.getAlert(name, rNum),
                           String.format("%s relay %s trip point is %.5g", name, rNum, trip));
            }
        }
    }


    /**
     *  Raise a relay configuration alarm
     * 
     *  @param  alert  The alarm to lower
     *  @param  cause  The alarm cause
     */
    private void raiseAlarm(Alert alert, String cause)
    {
        Boolean active = activeAlerts.get(alert.getAlertId());
        if (active != Boolean.TRUE) {
            alertService.raiseAlert(alert, AlertState.ALARM, cause);
            activeAlerts.put(alert.getAlertId(), true);
        }
    }


    /**
     *  Lower a relay configuration alarm
     * 
     *  @param  alert  The alarm to lower
     *  @param  cause  The alarm cause
     */
    private void lowerAlarm(Alert alert, String cause)
    {
        Boolean active = activeAlerts.get(alert.getAlertId());
        if (active == Boolean.TRUE) {
            alertService.raiseAlert(alert, AlertState.NOMINAL, cause);
            activeAlerts.put(alert.getAlertId(), false);
        }
    }


    /**
     * Callback to clear an {@code Alert} instance.
     * 
     * @param alert The Alert instance to clear.
     * @param alertState The AlertState for the provided Alert.
     * @return A ClearAlertCode to indicate which action is to be taken
     *         by the framework.
     */
    @Override
    public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState)
    {
        Boolean active = activeAlerts.get(alert.getAlertId());
        return active == null ? ClearAlertHandler.ClearAlertCode.UNKNOWN_ALERT
                              : active ? ClearAlertHandler.ClearAlertCode.DONT_CLEAR_ALERT : ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
    }


    /**
     *  Reads the pressure.
     *
     *  @param  sensor  Which pressure sensor to read
     *  @return  The pressure
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Read the pressure")
    public double readPressure(@Argument(description="Sensor name") Sensor sensor) throws DriverException
    {
        return mks.readPressure(sensor);
    }    


    /**
     *  Reads the MicroPirani temperature.
     *
     *  @return  The temperature (C)
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Read the Micropirani temperature")
    public double readTemperature() throws DriverException
    {
        return mks.readTemperature();
    }


    /**
     *  Gets a relay trip point.
     *
     *  @param  relay  The relay number
     *  @return  The trip point
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get a relay trip point")
    public double getRelayTrip(@Argument(description="Relay number") int relay) throws DriverException
    {
        return mks.getRelayTrip(relay);
    }    


    /**
     *  Gets a relay hysteresis value.
     *
     *  @param  relay  The relay number (1 - 3)
     *  @return  The hysteresis value
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get a relay hysteresis value")
    public double getRelayHyst(int relay) throws DriverException
    {
        return mks.getRelayHyst(relay);
    }


    /**
     *  Gets a relay trip direction.
     *
     *  @param  relay  The relay number
     *  @return  The trip direction enum
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get a relay trip direction")
    public Direction getRelayDirection(@Argument(description="Relay number") int relay) throws DriverException
    {
        return mks.getRelayDirection(relay);
    }    


    /**
     *  Gets a relay enabled state.
     *
     *  @param  relay  The relay number
     *  @return  The enabled state enum
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get a relay enabled state")
    public Enable getRelayEnable(@Argument(description="Relay number") int relay) throws DriverException
    {
        return mks.getRelayEnable(relay);
    }    


    /**
     *  Gets whether a relay is active.
     *
     *  @param  relay  The relay number (1 - 3)
     *  @return  Whether active
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get whether a relay is active")
    public boolean isRelayActive(@Argument(description="Relay number") int relay) throws DriverException
    {
        return mks.isRelayActive(relay);
    }


    /**
     *  Gets an analog output setting.
     *
     *  @param  chan  The analog channel number
     *  @return  The setting string
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get an analog output setting")
    public String getAnalogOut(@Argument(description="Channel number") int chan) throws DriverException
    {
        return mks.getAnalogOut(chan);
    }    


    /**
     *  Gets the calibration gas.
     *
     *  @return  The calibration gas enum
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get the calibration gas")
    public Gas getCalibrationGas() throws DriverException
    {
        return mks.getCalibrationGas();
    }


    /**
     *  Gets the transducer device type.
     *
     *  @return  The device type name
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get the device type")
    public String getDeviceType() throws DriverException
    {
        return mks.getDeviceType();
    }


    /**
     *  Gets the firmware version.
     *
     *  @return  The firmware version
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get the firmware version")
    public String getFirmwareVersion() throws DriverException
    {
        return mks.getFirmwareVersion();
    }


    /**
     *  Gets the hardware version.
     *
     *  @return  The hardware version
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get the hardware version")
    public String getHardwareVersion() throws DriverException
    {
        return mks.getHardwareVersion();
    }


    /**
     *  Gets the manufacturer.
     *
     *  @return  The manufacturer
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get the manufacturer")
    public String getManufacturer() throws DriverException
    {
        return mks.getManufacturer();
    }


    /**
     *  Gets the model.
     *
     *  @return  The model
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get the model")
    public String getModel() throws DriverException
    {
        return mks.getModel();
    }


    /**
     *  Gets the part number.
     *
     *  @return  The part number
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get the part number")
    public String getPartNumber() throws DriverException
    {
        return mks.getPartNumber();
    }


    /**
     *  Gets the serial number.
     *
     *  @return  The serial number
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get the serial number")
    public String getSerialNumber() throws DriverException
    {
        return mks.getSerialNumber();
    }


    /**
     *  Gets the transducer time on.
     *
     *  @return  The transducer time on (hours)
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get the transducer time on (hours)")
    public int getTimeOn() throws DriverException
    {
        return mks.getTimeOn();
    }

    /**
     *  Gets the sensor time on.
     *
     *  @return  The sensor time on (hours)
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get the sensor time on (hours)")
    public int getSensorTimeOn() throws DriverException
    {
        return mks.getSensorTimeOn();
    }


    /**
     *  Gets the sensor pressure dose.
     *
     *  @return  The sensor pressure dose
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get the sensor pressure dose")
    public double getSensorDose() throws DriverException
    {
        return mks.getSensorDose();
    }


    /**
     *  Gets the transducer status.
     *
     *  @return  The status string
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get the transducer status")
    public String getTransStatus() throws DriverException
    {
        return mks.getTransStatus();
    }


    /**
     *  Gets the pressure unit.
     *
     *  @return  The pressure unit
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get the pressure unit")
    public Unit getPressureUnit() throws DriverException
    {
        return mks.getPressureUnit();
    }


    /**
     *  Gets the baud rate.
     *
     *  @return  The baud rate: 4800, 9600, 19200, 38400, 57600, 115200 or 230400
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get the baud rate")
    public int getBaudRate() throws DriverException
    {
        return mks.getBaudRate();
    }


    /**
     *  Gets the user tag.
     *
     *  @return  The user tag string
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get the user tag")
    public String getUserTag() throws DriverException
    {
        return mks.getUserTag();
    }


    /**
     *  Gets the switch enabled state.
     *
     *  @return  Whether the switch is enabled
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get the switch enabled state")
    public boolean isSwitchEnabled() throws DriverException
    {
        return mks.isSwitchEnabled();
    }


    /**
     *  Gets the RS-485 delayed response state.
     *
     *  @return  Whether delayed response enabled
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get the RS-485 delayed response state")
    public boolean isResponseDelayed() throws DriverException
    {
        return mks.isResponseDelayed();
    }


    /**
     *  Gets the relay trip delayed state.
     *
     *  @return  Whether relay trip delay is enabled
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get the relay trip delayed state")
    public boolean isRelayDelayed() throws DriverException
    {
        return mks.isRelayDelayed();
    }


    /**
     *  Gets the identify state.
     *
     *  @return  Whether the identify light is flashing
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get the identify state")
    public boolean isIdentifying() throws DriverException
    {
        return mks.isIdentifying();
    }


    /**
     *  Gets the atmospheric pressure calibration factor.
     *
     *  @return  The atmospheric pressure factor
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get the atmospheric pressure calibration factor")
    public double getAtmospheric() throws DriverException
    {
        return mks.getAtmospheric();
    }


    /**
     *  Gets the MicroPirani auto zero limit.
     *
     *  @return  The auto zero limit pressure
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get the MicroPirani auto zero limit")
    public double getAutoZeroLimit() throws DriverException
    {
        return mks.getAutoZeroLimit();
    }


    /**
     *  Gets the cold cathode full-scale pressure calibration factor.
     *
     *  @return  The full-scale pressure factor
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get the cold cathode full-scale pressure calibration factor")
    public double getFullScale() throws DriverException
    {
        return mks.getFullScale();
    }


    /**
     *  Gets cold cathode turn-on pressure.
     *
     *  @return  The turn-on pressure
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get cold cathode turn-on pressure")
    public double getOnPressureCC() throws DriverException
    {
        return mks.getOnPressureCC();
    }


    /**
     *  Gets cold cathode turn-off pressure.
     *
     *  @return  The turn-off pressure
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get cold cathode turn-off pressure")
    public double getOffPressureCC() throws DriverException
    {
        return mks.getOffPressureCC();
    }


    /**
     *  Gets low integration pressure.
     *
     *  @return  The pressure
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get the low integration pressure")
    public double getLowIntPressure() throws DriverException
    {
        return mks.getLowIntPressure();
    }


    /**
     *  Gets high integration pressure.
     *
     *  @return  The pressure
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get the high integration pressure")
    public double getHighIntPressure() throws DriverException
    {
        return mks.getHighIntPressure();
    }


    /**
     *  Gets cold cathode auto power state.
     *
     *  @return  Whether auto power state is on
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get cold cathode auto power state")
    public boolean isCCAutoPowerOn() throws DriverException
    {
        return mks.isCCAutoPowerOn();
    }


    /**
     *  Gets cold cathode power state.
     *
     *  @return  Whether power state is on
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get cold cathode power state")
    public boolean isCCPowerOn() throws DriverException
    {
        return mks.isCCPowerOn();
    }


    /**
     *  Gets cold cathode protection delay.
     *
     *  @return  Protection delay time (-1 = OFF)
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get cold cathode protection delay")
    public int getCCProtDelay() throws DriverException
    {
        return mks.getCCProtDelay();
    }


    /**
     *  Gets cold cathode pressure dose limit.
     *
     *  @return  The pressure dose
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get cold cathode pressure dose limit")
    public double getSensorDoseLimit() throws DriverException
    {
        return mks.getSensorDoseLimit();
    }


    /**
     *  Gets read buffer contents.
     *
     *  @return  The buffer contents
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get read buffer contents")
    public String getReadBuffer() throws DriverException
    {
        return mks.getReadBuffer();
    }


    /**
     *  Formats read buffer contents.
     *
     *  @return  The formatted buffer contents
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Format read buffer contents")
    public String formatReadBuffer() throws DriverException
    {
        return mks.formatReadBuffer();
    }


    /**
     *  Sets read timeout.
     * 
     *  @param  timeout  The timeout (secs)
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.ACTION, level=50, description="Set read timeout")
    public void setTimeout(@Argument(description="The timeout (secs)") double timeout) throws DriverException
    {
        mks.setTimeout(timeout);
    }


    /**
     *  Gets command retry limit.
     *
     *  @return  The retry limit
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get command retry limit")
    public int getRetryLimit()
    {
        return mks.getRetryLimit();
    }


    /**
     *  Sets command retry limit.
     * 
     *  @param  limit  The limit
     */
    @Command(type=Command.CommandType.ACTION, level=50, description="Set command retry limit")
    public void setRetryLimit(@Argument(description="The retry limit") int limit)
    {
        mks.setRetryLimit(limit);
    }


    /**
     *  Gets command retry count.
     *
     *  @return  The retry count
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get command retry count")
    public int getRetryCount()
    {
        return mks.getRetryCount();
    }


    /**
     *  Clears command retry count.
     */
    @Command(type=Command.CommandType.ACTION, level=50, description="Clear command retry count")
    public void clearRetryCount()
    {
        mks.clearRetryCount();
    }

}
