package org.lsst.ccs.subsystem.vacuum;

import java.util.HashMap;
import java.util.Map;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.pfeiffer.ASM380;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.Control;
import org.lsst.ccs.monitor.MonitorUpdateTask;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.vacuum.config.VacuumConfig;
import org.lsst.ccs.subsystem.vacuum.constants.VacuumAlert;
import org.lsst.ccs.subsystem.vacuum.constants.DeviceState;
import org.lsst.ccs.subsystem.vacuum.data.VacSysState;

import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 *  Interface to the pump cart device (ASM380 leak detector)
 *
 *  @author CCS team
 */
public class ASM380Device extends Device {

    boolean TEST_MODE=false; // this will be changed to a standard simulation setup soon

    public static final int
        PUMP_STOPPED = 0,
        PUMP_PUMPING = 1,
        VENT_CLOSED = 0,
        VENT_VENTING = 1;

    CCSTimeStamp lastChannelUpdateStamp;
    
    static class PumpSwitchState {
        boolean state;
        CCSTimeStamp time;
     
        private PumpSwitchState(boolean state, CCSTimeStamp time) {
            this.state = state;
            this.time = time;
        }
    }   
    PumpSwitchState pumpSwitchState = new PumpSwitchState(false, CCSTimeStamp.currentTime());

    static class VentSwitchState {
        boolean state;
        CCSTimeStamp time;
     
        private VentSwitchState(boolean state, CCSTimeStamp time) {
            this.state = state;
            this.time = time;
        }
    }   

    VentSwitchState ventSwitchState = new VentSwitchState(false, CCSTimeStamp.currentTime());

    private double cryoForelinePressure = -1.0;
    private CCSTimeStamp cryoForelineTimeStamp = null;
    private double hxForelinePressure = -1.0;
    private CCSTimeStamp hxForelineTimeStamp = null;

    private double lastInletPressure = Double.NaN;

    //    private final VacSysState vs = new VacSysState();

    private String targetPressureSrc = null;

    private VacSysState vs;


    private List<MonitorUpdateTask> tasksForDevice = null;

    /**
     *  Data fields.
     */
    @ConfigurationParameter(name="devcId", category=VacuumConfig.CRYO, isFinal=true)
    private volatile String devcId;

    @ConfigurationParameter(category=VacuumConfig.CRYO, isFinal=false, range = ("0.00..800.0"), units = "Torr")
    private volatile double maxVentPressure = 765; // maximum pressure while venting

    @ConfigurationParameter(category=VacuumConfig.CRYO, isFinal=false, range = ("-1.00..800.0"), units = "Torr")
    private volatile double minPumpPressure = 0.001; // minimum pressure while venting

    private boolean abortVent = false; // abort any current vent operation when high
    private boolean abortPump = false; // abort any current pump operation when high

    private final ASM380.ConnType connType = ASM380.ConnType.SERIAL;
    private final int baudRate = 9600;

    private static final Logger LOG = Logger.getLogger(ASM380Device.class.getName());
    private final ASM380 asm380Dev = new ASM380();
    private int errorCount = 0;
    private boolean initError = false;
    private final Map<String, Boolean> activeAlerts = new HashMap<>();

    //    private boolean pumpSwitch = false; // the pumping intent
    private boolean pulseVentingActive = false;

    private String lastVentStatus = null;
    private String reportedPumpStatus = null;
    private boolean lastPumpStatus = false;
    private boolean pumpTransActive = false;
    private int lastStatus;

    /**
     *  Performs configuration.
     */
    @Override
    public void initDevice() {
       
        if (devcId == null) {
            ErrorUtils.reportConfigError(LOG, name, "devcId", "is missing");
        }
        
        fullName = "ASM380";

        tasksForDevice = getDeviceMonitorUpdateTasks();


        //Register the Alerts raised by this Device.                                                                                                                                                       
        // ex:    alertService.registerAlert(ASM380Alert.XXXBAD.getAlert(name, rNum));
    }

    /**
     *  Performs full initialization.
     */
    @Override
    public void initialize()
    {
        try {
            //            asm380Dev.setRetryLimit(3);
            asm380Dev.open(connType, devcId, baudRate, 0);
            errorCount = 0;

            LOG.log(Level.INFO, "Connected to {0}",
                    new Object[]{fullName});
            initError = false;
            setOnline(true);
        }
        catch (DriverException e) {
            if (!initError) {
                LOG.log(Level.SEVERE, "Error connecting to {0}: {1}", new Object[]{fullName, e});
                initError = true;
            }
            try {
                asm380Dev.close();
            }
            catch (DriverException ce) {
                // Here if device wasn't opened
            }
            if (TEST_MODE) {
                setOnline(true); // set it online anyway if in test mode
            }
        }


        try {
            int status = asm380Dev.getStatus();

            pumpSwitchState.state = (status & ASM380.ST_IN_CYCLE) != 0;
            pumpSwitchState.time = CCSTimeStamp.currentTime();
            // initialize these so getPumpStatus() handles first call correctly
            lastPumpStatus = pumpSwitchState.state;
            reportedPumpStatus = lastPumpStatus ? "on" : "off";
            lastStatus     = status;
            
            ventSwitchState.state = (status & ASM380.ST_INLET_VENT) != 0;
            ventSwitchState.time = CCSTimeStamp.currentTime();
        } catch (DriverException ex) {
            LOG.log(Level.SEVERE, "Setting {0} offline due to error retrieving initial status {0}: {1}", new Object[]{fullName, ex});       
            setOnline(false);
        }


    }

    /**
     *  Closes the connection.
     */
    @Override
    public void close()
    {
        try {
            asm380Dev.close();
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error disconnecting from {0}: {1}", new Object[]{fullName, e});
        }
    }



    /*
     *  Reads a monitoring channel.
     *
     *  @param  ch  The channel to read
     *  @return  The read value
     */
    @Override
         public double readChannel(Channel ch)
    {
        double value = Double.NaN;
        if (isOnline() && !TEST_MODE) {
            switch(ch.getHwChan()) {
            case 0:
                try {
                    value = readInletPressure();
                    lastInletPressure = value;
                    errorCount = 0;
                }
                catch (DriverException e) {
                    LOG.log(Level.SEVERE, "Error reading pressure from {0}: {1}", new Object[]{fullName, e});
                    if (++errorCount >= 5) {
                        setOnline(false);
                    }
                }
                break;
            case 1:
                try {
                    String tmpValue = getPumpStatus();
                    if (tmpValue.equals("on")) {
                        value = PUMP_PUMPING;
                    } else if (tmpValue.equals("off"))  {
                        value = PUMP_STOPPED;
                    } else {
                        LOG.log(Level.WARNING, String.format("getPumpStatus() returned: %s, skipping", tmpValue));
                    }
                }
                catch (DriverException e) {
                    LOG.log(Level.SEVERE, "Error reading pump cycle enabled status from {0}: {1}", new Object[]{fullName, e});
                    if (++errorCount >= 5) {
                        setOnline(false);
                    }
                }
                break;
            case 2:
                try {
                    value = asm380Dev.readTemperature();
                    errorCount = 0;
                }
                catch (DriverException e) {
                    LOG.log(Level.SEVERE, "Error reading temperature from {0}: {1}", new Object[]{fullName, e});
                    if (++errorCount >= 5) {
                        setOnline(false);
                    }
                }
                break;
            case 4:
                try {
                    value = getVentValveStatus().contains("open") ? VENT_VENTING : VENT_CLOSED ;
                    errorCount = 0;
                }
                catch (DriverException e) {
                    LOG.log(Level.SEVERE, "Error reading venting status from {0}: {1}", new Object[]{fullName, e});
                    if (++errorCount >= 5) {
                        setOnline(false);
                    }
                }
                break;
            case 5:
                value = lastInletPressure;
                break;
            }

        }
        return value;
    }


    @Command(description = "Show the version string")
    public String showVersion() throws DriverException {
        return asm380Dev.getVersion();
    }

    @Command(description = "Show the instrument status")
    public String showStatus() throws DriverException {
        return String.format("0x%04x", asm380Dev.getStatus());
    }

    @Command(description = "Show the decoded instrument status")
    public String showDecodedStatus() throws DriverException {
        String[] status = asm380Dev.decodeStatus();
        StringBuilder text = new StringBuilder();
        for (int j = 0; j < status.length; j++) {
            if (j != 0) {
                text.append("\n");
            }
            text.append(status[j]);
        }
        return text.toString();
    }

    @Command(description = "Return the reference pressure")
    public double getReferencePressure() throws DriverException {
        double pressure = -1 ;

        if (isOnline()) {
            if (vs != null && !TEST_MODE) {
                LOG.fine("HXForelineValveStatus = " + vs.getHXForelineValveStatus());

                if (vs.getHXForelineValveStatus().contains("OPEN") && !vs.getCryoForelineValveStatus().contains("OPEN")) {
                    pressure = vs.getHXFlinePress();
                    targetPressureSrc = "HX";
                    LOG.fine("Using HX foreline pressure as the reference.");
                } else      if (vs.getCryoForelineValveStatus().contains("OPEN") && !vs.getHXForelineValveStatus().contains("OPEN")) {
                    pressure = vs.getCryoFlinePress();
                    targetPressureSrc = "Cryo";
                    LOG.fine("Using Cryo foreline pressure as the reference.");
                } else      if (vs.getHXForelineValveStatus().contains("OPEN") && vs.getCryoForelineValveStatus().contains("OPEN")) {
                    pressure = (vs.getHXFlinePress() + vs.getCryoFlinePress())/2.0 ;
                    targetPressureSrc = "HX and Cryo";
                    LOG.fine("Using the average of the cryo and foreline pressures as the reference.");
                } else {
                    pressure = readInletPressure();
                    targetPressureSrc = "Inlet";
                    LOG.fine("Using inlet pressure because both foreline valves are closed.");
                }
            } else {
                pressure = readInletPressure();
                targetPressureSrc = "Inlet";
                LOG.fine("Using inlet pressure because there is no VacSysState or Pump Cart not online.");
            }
        }

        return pressure ;
        //        return (useInletPressure ? readInletPressure() : cryoForelinePressure);
    }

    @Command(description = "Read the inlet pressure")
    public double readInletPressure() throws DriverException {
        return (TEST_MODE ? -1 : asm380Dev.readInletPressure());
    }

    @Command(description = "Read the temperature")
    public int readTemperature() throws DriverException {
        return asm380Dev.readTemperature();
    }

    @Command(description = "Show the primary pump operating hours")
    public int showPrimaryHours() throws DriverException {
        return asm380Dev.getPrimaryHours();
    }

    @Command(description = "Show the high vacuum pump operating hours")
    public int showHighVacHours() throws DriverException {
        return asm380Dev.getHighVacHours();
    }

    @Command(description = "Show target pressure src")
    public String getTargetPressureSrc() {
        try {
            if (isOnline()) {
                double press = getReferencePressure(); // needed to get the src setting updated
            }
        } catch (DriverException ex) {
            LOG.log(Level.INFO, "Exception while getting reference pressure: "+ex);
        }

        return targetPressureSrc;
    }

    @Command(description = "Show the stored warning codes")
    public String showWarningCodes() throws DriverException {
        return showCodes(asm380Dev.getWarningCodes());
    }

    @Command(description = "Show the stored fault codes")
    public String showFaultCodes() throws DriverException {
        return showCodes(asm380Dev.getFaultCodes());
    }

    private static String showCodes(int[] codes) {
        if (codes.length == 0) {
            return "No codes stored";
        }
        else {
            StringBuilder text = new StringBuilder();
            for (int code : codes) {
                text.append(code).append(" ");
            }
            return text.toString();
        }
    }

    @Command(description = "Clear the stored warning codes")
    public void clearWarnings() throws DriverException {
        asm380Dev.clearWarnings();
    }

    @Command(description = "Clear the stored fault codes")
    public void clearFaults() throws DriverException {
        asm380Dev.clearFaults();
    }

    @Command(description = "set pump on (true) or off (false)")
    public void pumpOn(boolean on) throws DriverException {
        if (!getVentValveStatus().contains("closed") && on == true) {
            //      raiseAlarm(VacuumAlert.BOTH_PUMPING_VENTING,
            LOG.log(Level.SEVERE,"Rejected attempt to turn pumping on with the valve not in the closed state.");
        } else {
            asm380Dev.pumpOn(on);
            pumpSwitchState.time = CCSTimeStamp.currentTime();

            int maxcount = 10;  // arbitarily chosen, needs calibration
            int count = 0;
            String pStatus = null;
            while (count < maxcount) {
                try {
                    Thread.sleep(50);
                } catch (InterruptedException ex) {
                    LOG.log(Level.SEVERE, "sleep interrupted while waiting for pumping state change {0}");
                }
                if (getPumpStatus().equals(on ? "on" : "off")) {
                    pumpSwitchState.state = on;
                    LOG.log(Level.INFO, "Pump is {0} by command", (lastPumpStatus ? "on" : "off"));
                    forceDataPublication();
                    return;
                } else {
                    count++;
                }
            }
            asm380Dev.pumpOn(false);  // force off on timeout
            LOG.log(Level.SEVERE,"pumpOn({0}) failed with timeout, forcing off", on);
            forceDataPublication();
        }
    }

    @Command(description = "get the pumping status")
    public String getPumpStatus() throws DriverException {

        int status = asm380Dev.getStatus();
        boolean pumpStatus = (status & ASM380.ST_IN_CYCLE) != 0;

        if (pumpStatus == lastPumpStatus) {
            reportedPumpStatus = pumpStatus ? "on" : "off";
            pumpTransActive = false;
        } else {  // change of status
            LOG.log(Level.INFO,String.format("pumpStatus changed: %s -> %s", lastPumpStatus, pumpStatus));
            LOG.log(Level.INFO,String.format("pumpStatus detailed: 0x%04x -> 0x%04x", lastStatus, status));
            if (!pumpTransActive) {  // start of transition
                reportedPumpStatus = "TransStarted";
                pumpTransActive = true;
            } else { // two change in sequence ==> glitch
                reportedPumpStatus = pumpStatus ? "on" : "off";
                pumpTransActive = false;
                LOG.log(Level.WARNING,String.format("Glitch detected in pump status, ignoring", lastStatus, status));
            }
        }
        lastPumpStatus = pumpStatus;
        lastStatus     = status;

        return reportedPumpStatus;
    }

    @Command(description = "get the pumping switch state")
    public boolean getPumpSwitchState() {
        return pumpSwitchState.state;
    }

    @Command(description = "get the pumping switch state update time")
    public CCSTimeStamp getPumpSwitchStateTime() {
        return pumpSwitchState.time;
    }

    @Command(description = "get the pumping switch state")
    public boolean getVentSwitchState() {
        return ventSwitchState.state;
    }
    @Command(description = "get the pumping switch state update time")
    public CCSTimeStamp getVentSwitchStateTime() {
        return ventSwitchState.time;
    }

    @Command(description = "execute open vent sequence")
    public void openVentValve(boolean open) throws DriverException {
        if (!getPumpStatus().equals("off") && open==true) {
            //      raiseAlarm(VacuumAlert.BOTH_PUMPING_VENTING,
            LOG.log(Level.SEVERE,"Rejected attempt to open vent while pump not in off state.");
        } else {
            if (open) {
                abortVent=false;
            }
            if (!TEST_MODE) {
                asm380Dev.setVentValveNominal();
                asm380Dev.openVentValve(open);
            }

            ventSwitchState.state = open;
            ventSwitchState.time = CCSTimeStamp.currentTime();

            try {
                Thread.sleep(200);
            } catch (InterruptedException ex) {
                LOG.log(Level.SEVERE, "Interrupted!! Prematurely interrupted while waiting for venting state change " + ex);
            }

            forceDataPublication();
            LOG.log(Level.INFO, "VentStatusAfterSleep: {0}", lastVentStatus);
        }
    }

    @Command(description = "get the venting status")
    public String getVentValveStatus() throws DriverException {
        return lastVentStatus = (TEST_MODE ? "closed" : asm380Dev.getVentValveStatus());
    }

    @Command(description = "get the venting valve actuation parameters")
    public String getVentValveParms() throws DriverException {
        return asm380Dev.getVentValveParms();
    }

    @Command(description = "setup venting sequence")
    public void setVentValveParms(
                 @Argument(description="delay before actuation (0->2 secs)") int delay,
                 @Argument(description="time (secs) in open state") int period) throws DriverException {
        asm380Dev.setVentValveParms(delay,period);
    }

    @Command(description = "execute pulse vent sequence")
    public void pulseVentValve(@Argument(description="time (secs) in open state") int seconds) throws DriverException {

        if (pulseVentingActive) {
            LOG.log(Level.SEVERE, "Pulse venting already active");
            return;
        }

        Thread vThread = new Thread(new Runnable() {
                @Override
                public void run()  {

                    try {
                        if (seconds>0) {
                            asm380Dev.setVentValveParms(0,seconds); // start immediately and vent for seconds seconds
                        }

                        long ventStart = System.currentTimeMillis();

                        openVentValve(true); // this command action runs asynchronously
                        try {
                            double last_press=0.0;
                            while(((System.currentTimeMillis() - ventStart)/1000 < (seconds+1) || seconds<0) && abortVent==false) {
                                double press = getReferencePressure();

                                if (  (press>=750.0 && last_press<maxVentPressure)   ||     press<maxVentPressure  ) {
                                    Thread.sleep(50); // check every 1/20 of a second
                                } else {
                                    LOG.log(Level.SEVERE, "Interrupted!! current inlet pressure " + press + "Torr exceeds max setting = " + maxVentPressure);
                                    break;
                                }

                                last_press = press;
                            }
                        } catch (InterruptedException ex) {
                            LOG.log(Level.SEVERE, "Interrupted!! Prematurely stopping the venting" + ex);
                        }

                    } catch (DriverException ex) {
                            LOG.log(Level.SEVERE, "Exception while pulse venting: "+ex);
                    }
                    try {
                        openVentValve(false);
                        pulseVentingActive = false;
                    } catch (DriverException ex) {
                        LOG.log(Level.SEVERE, "Exception while trying to close valve after pulse venting: "+ex);
                    }
                    forceDataPublication();
                }
            });
        vThread.start();
    }


    @Command(description = "execute pulse cycling/pumping")
    public void pulsePump(@Argument(description="time (secs) in pumping state") int seconds) throws DriverException {
        if (pumpSwitchState.state) {
            LOG.log(Level.SEVERE, "Pulse pumping already active");
            return;
        }
        try {
            pumpOn(true);
        } catch (DriverException ex) {
            LOG.log(Level.SEVERE, "Exception while pulse pumping: " + ex);
            return;
        }
        if (seconds < 0 && minPumpPressure <= 0.0) { // just return with pump on 
            return;
        }

        long pumpStart = (long)(1000.0 * pumpSwitchState.time.getUTCDouble());
        Thread pThread = new Thread(new Runnable() {
                @Override
                public void run() {
                    try {
                        while((seconds < 0 || (System.currentTimeMillis() - pumpStart)/1000 < seconds) && abortPump==false) {
                            double press = getReferencePressure();
                            if (press < minPumpPressure) {
                                LOG.log(Level.INFO, "pumping stopped by crossing min pressure:{0}", minPumpPressure);
                                break;
                            }
                            try {
                                Thread.sleep(50);
                            } catch (InterruptedException ex) {
                                LOG.log(Level.SEVERE, "sleep interrupted in pulse pumping");
                            }
                        }
                    } catch (DriverException ex) {
                        LOG.log(Level.SEVERE, "Exception in pulse pumping" + ex);
                    }
                    try {
                        pumpOn(false);
                    } catch (DriverException ex) {
                        LOG.log(Level.SEVERE, "Exception while trying to stop pulse pumping: " + ex);
                    }
                }
            });
        pThread.start();
    }



    @Command(description = "set the maximum inlet pressure to vent to")
    public void setMaxVentPressure(double value) throws DriverException {
        maxVentPressure = value;
    }

    @Command(type=Command.CommandType.QUERY, description = "get the maximum inlet pressure to vent to")
    public double getMaxVentPressure() throws DriverException {
        return maxVentPressure;
    }
    

    @Command(description = "set the minimum inlet pressure to pump down to")
    public void setMinPumpPressure(double value) throws DriverException {
        minPumpPressure = value;
    }
    
    @Command(type=Command.CommandType.QUERY, description = "get the maximum inlet pressure to vent to")
    public double getMinPumpPressure() throws DriverException {
        return minPumpPressure;
    }
    
    @Command(description = "enable/disable pump cart audible alarm")
    public void enableAudibleAlarm(boolean enable) throws DriverException {
        asm380Dev.enableAlarm(enable);
    }

    @Command(description = "enable/disable pump cart triode filament")
    public void enableFilament(boolean enable) throws DriverException {
        asm380Dev.enableFilament(enable);
    }

    @Command(description = "reset pump cart triode safety")
    public void resetTriodeSafety() throws DriverException {
        asm380Dev.resetTriodeSafety();
    }

    @Command(description = "set venting sequence nominal")
    public void setVentValveNominal() throws DriverException {
        asm380Dev.setVentValveNominal();
    }


    @Command(description = "Write a command")
    public String writeCommand(@Argument(description="The commmand to send") String command) throws DriverException {
         return asm380Dev.writeCommand(command);
    }

    @Command(description = "set abort flag")
    public void abort() throws DriverException {
        abortVent = true;
        abortPump = true;
    }

    @Command(description = "set abort venting flag and close vent valve")
    public void abortVent() throws DriverException {
        abortVent = true;
        asm380Dev.setVentValveNominal();
        openVentValve(false);
    }

    @Command(description = "set abort pumping flag and stop pumping")
    public void abortPump() throws DriverException {
        abortPump = true;
        pumpOn(false);
    }

    @Command(description = "show abort flag")
    public boolean showAbort() throws DriverException {
         return abortVent || abortPump ;
    }

    @Command(description = "show venting abort flag")
    public boolean showVentAbortState() throws DriverException {
         return abortVent ;
    }

    @Command(description = "show pumping abort flag")
    public boolean showPumpAbortState() throws DriverException {
         return abortPump ;
    }

    @Command(description = "clear abort flag")
    public void clearAbort() throws DriverException {
        abortVent = false;
        abortPump = false;
    }


    /**
     *  Raise a relay configuration alarm
     *
     *  @param  alert  The alarm to lower
     *  @param  cause  The alarm cause
     */
    private void raiseAlarm(Alert alert, String cause)
    {
        Boolean active = activeAlerts.get(alert.getAlertId());
        if (active != Boolean.TRUE) {
            alertService.raiseAlert(alert, AlertState.ALARM, cause);
            activeAlerts.put(alert.getAlertId(), true);
        }
    }


    /**
     *  Lower a relay configuration alarm
     *
     *  @param  alert  The alarm to lower
     *  @param  cause  The alarm cause
     */
    private void lowerAlarm(Alert alert, String cause)
    {
        Boolean active = activeAlerts.get(alert.getAlertId());
        if (active == Boolean.TRUE) {
            alertService.raiseAlert(alert, AlertState.NOMINAL, cause);
            activeAlerts.put(alert.getAlertId(), false);
        }
    }

    /**
     *  Gets the device state.
     *
     *  @return  The device state
     **/
    public DeviceState getDeviceState()
    {

            if (!isOnline()) {
                return DeviceState.OFFLINE;
            } else if (reportedPumpStatus.contains("on")) {
                return DeviceState.NORMAL;
            } else if (reportedPumpStatus.contains("off")) {
                return DeviceState.STOPPED;
            } else {
                return DeviceState.TRANSIT;
            }

    }

    private void forceDataPublication() {
        for (MonitorUpdateTask task: tasksForDevice) {
            task.scheduleUpdateAndPublishNow();
        }
    }

    public void updateVacSysState(VacSysState vs) {
        LOG.fine("setting VacSysState in ASM380");
        this.vs = vs;
    }


}
