package org.lsst.ccs.subsystem.vacuum;

import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.subsystem.common.devices.dataforth.Maq20DiscControl;
import org.lsst.ccs.subsystem.vacuum.constants.Devices;

/**
 *  Handles the simulated vacuum system MAQ20 device with switches.
 *
 *  @author Owen Saxton
 */
public class VacSimMaq20Device extends org.lsst.ccs.subsystem.common.devices.dataforth.SimMaq20Device implements SwitchDevice {

    /**
     *  Constants.
     */
    public static final int
        SW_INST_FTPP_VALVE  = VacMaq20Device.SW_INST_FTPP_VALVE,
        SW_INST_FTH_VALVE   = VacMaq20Device.SW_INST_FTH_VALVE,
        SW_INST_L3H_VALVE   = VacMaq20Device.SW_INST_L3H_VALVE,
        SW_INST_L3LF_VALVE  = VacMaq20Device.SW_INST_L3LF_VALVE,
        SW_INST_FLINE_VALVE = VacMaq20Device.SW_INST_FLINE_VALVE,
        SW_CRYO_FLINE_VALVE = VacMaq20Device.SW_CRYO_FLINE_VALVE,
        SW_HX_FLINE_VALVE   = VacMaq20Device.SW_HX_FLINE_VALVE,
        NUM_SWITCHES        = VacMaq20Device.NUM_SWITCHES;

    @LookupField(strategy=LookupField.Strategy.CHILDREN)
    protected Maq20DiscControl maqCtrl;


    /**
     *  Device initialization.
     */
    @Override
    protected void initDevice()
    {
        super.initDevice();
        if (maqCtrl == null) {
            throw new RuntimeException(path + ": Child Maq20DiscControl node has not been defined");
        }
    }


    /**
     *  Gets the switch device ID.
     * 
     *  @return  The device ID
     */
    @Override
    public int getSwitchDevice()
    {
        return Devices.DEVC_MAQ20;
    }


    /**
     *  Sets a switch on or off.
     *
     *  For the vacuum Pluto, this is implemented as a pair of push buttons,
     *  one for on, one for off.
     *
     *  @param  sw  The switch number.
     *  @param  on  The on state to set: true or false
     *  @throws DriverException
     */
    @Override
    public void setSwitch(int sw, boolean on) throws DriverException
    {
         maqCtrl.setLineOn(sw, on);
    }


    /**
     *  Gets the on state of a switch.
     *
     *  @param  sw  The switch number.
     *  @return  Whether the switch is on
     */
    @Override
    public Boolean isSwitchOn(int sw)
    {
        return maqCtrl.isLineOn(sw);
    }

}
