package org.lsst.ccs.subsystem.vacuum;

import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.subsystem.common.devices.dataforth.Maq20Device;
import org.lsst.ccs.subsystem.common.devices.dataforth.Maq20DiscControl;
import org.lsst.ccs.subsystem.vacuum.constants.DeviceState;
import org.lsst.ccs.subsystem.vacuum.constants.Devices;
import org.lsst.ccs.command.annotations.Command;
import java.util.List;
import java.util.ArrayList;

/**
 *  Handles the vacuum system MAQ20 device with switches.
 *
 *  @author Owen Saxton
 */
public class VacMaq20Device extends Maq20Device implements SwitchDevice {

    /**
     *  Constants.
     */
    public static final int
        SW_INST_FTPP_VALVE  = 0,
        SW_INST_FTH_VALVE   = 1,
        SW_INST_L3H_VALVE   = 2,
        SW_INST_L3LF_VALVE  = 3,
        SW_INST_FLINE_VALVE = 4,
        SW_CRYO_FLINE_VALVE = 5,
        SW_HX_FLINE_VALVE   = 6,
        NUM_SWITCHES        = 7;

    public static final int
        CRYO_FLINE_VALVE_OPEN   = 0,
        CRYO_FLINE_VALVE_CLOSED = 1,
        HX_FLINE_VALVE_OPEN     = 3,
        HX_FLINE_VALVE_CLOSED   = 4;


    public static final int 
	RELAY_MODID = 0,
	VALVE_POSITIONS_MODID = 1;

    @LookupField(strategy=LookupField.Strategy.CHILDREN)
    protected List<Maq20DiscControl> maqCtrlDiscList = new ArrayList<>();
    //    @LookupField(strategy=LookupField.Strategy.CHILDREN)
    //   protected List<Maq20Device> maqDevList = new ArrayList<>();
    //    @LookupField(strategy=LookupField.Strategy.CHILDREN)
    //    protected Maq20DiscControl maqCtrl;
    protected Maq20DiscControl maqRly = null;
    protected Maq20DiscControl maqValves = null;
	

    /**
     *  Device initialization.
     */
    @Override
    protected void initDevice()
    {
        super.initDevice();
	for (Maq20DiscControl maq : maqCtrlDiscList) {
	    if (maq.getModIndex() == RELAY_MODID) {
		maqRly = maq;
	    } else if (maq.getModIndex() == VALVE_POSITIONS_MODID) {
		maqValves = maq;
	    }
	}

        if (maqRly == null) {
            throw new RuntimeException(path + ": Child Maq20DiscControl relay module node has not been defined");
        }
        if (maqValves == null) {
            throw new RuntimeException(path + ": Child Maq20Device valve limit switch module node has not been defined");
        }
    }


    /**
     *  Gets the switch device ID.
     * 
     *  @return  The device ID
     */
    @Override
    public int getSwitchDevice()
    {
        return Devices.DEVC_MAQ20;
    }


    /**
     *  Sets a switch on or off.
     *
     *  For the vacuum Pluto, this is implemented as a pair of push buttons,
     *  one for on, one for off.
     *
     *  @param  sw  The switch number.
     *  @param  on  The on state to set: true or false
     *  @throws DriverException
     */
    @Override
    public void setSwitch(int sw, boolean on) throws DriverException
    {
	if (sw < 0x100) {
	    maqRly.setLineOn(sw, on);
	} else {
	    throw new RuntimeException(path + ": ERROR! tried to set the state of a valve limit switch.");
	}
    }


    /**
     *  Gets the on state of a switch.
     *
     *  @param  sw  The switch number.
     *  @return  Whether the switch is on
     */
    @Override
    public Boolean isSwitchOn(int sw)
    {
	    return maqRly.isLineOn(sw);
    }

    /**
     *  Gets the on state of a limit switch.
     *
     *  @param  sw  The switch number.
     *  @return  Whether the switch is on
     */
    @Command(type=Command.CommandType.QUERY, description="Get the limit switch state", level=0)
    public Boolean isLimitSwitchOn(int sw)
    {
	// should no longer be necessary but leaving in as a safety until tested
	int isw = sw & 0x7;
	System.out.println("Querying valve state for isw = " + isw);
	return maqValves.isInputOn(isw);

    }


    /**
     *  Gets the detailed device state of a foreline cryo valve
     *
     *  Used only for pump cart valves
     *
     *  @return  The device state string, or null if not a valve                                                 
     */
    @Command(type=Command.CommandType.QUERY, description="Get the cryo foreline limit switch state as a string", level=0)
    public String getCryoLimitSwitchStateString()
    {
	return(getCryoLimitSwitchState().name());
    }


    /**
     *  Gets the detailed device state of a foreline cryo valve
     *
     *  Used only for pump cart valves
     *
     *  @return  The device state, or null if not a valve                                                 
     */
    public DeviceState getCryoLimitSwitchState()
    {
	//        int[] stateDesc = switchStates.get(sw);
	//        if (stateDesc == null) return null;
        Integer openValue = maqValves.isInputOn(CRYO_FLINE_VALVE_OPEN) ? 1 : 0 ;
        Integer shutValue = maqValves.isInputOn(CRYO_FLINE_VALVE_CLOSED) ? 1 : 0 ;
        if (openValue == null || shutValue == null) return DeviceState.OFFLINE;
        return openValue == 1 ? shutValue == 0 ? DeviceState.OPEN : DeviceState.ERROR : shutValue == 1 ? DeviceState.SHUT : DeviceState.TRANSIT;
    }
    
    /**
     *  Gets the detailed device state of a foreline hx valve
     *
     *  Used only for pump cart valves
     *
     *  @return  The device state string, or null if not a valve                                                 
     */
    @Command(type=Command.CommandType.QUERY, description="Get the hx foreline limit switch state as a string", level=0)
    public String getHxLimitSwitchStateString()
    {
	return(getHxLimitSwitchState().name());
    }


    /**
     *  Gets the detailed device state of a foreline hx valve
     *
     *  Used only for pump cart valves
     *
     *  @return  The device state, or null if not a valve                                                 
     */
    public DeviceState getHxLimitSwitchState()
    {
	//        int[] stateDesc = switchStates.get(sw);
	//        if (stateDesc == null) return null;
        Integer openValue = maqValves.isInputOn(HX_FLINE_VALVE_OPEN) ? 1 : 0 ;
        Integer shutValue = maqValves.isInputOn(HX_FLINE_VALVE_CLOSED) ? 1 : 0 ;
        if (openValue == null || shutValue == null) return DeviceState.OFFLINE;
        return openValue == 1 ? shutValue == 0 ? DeviceState.OPEN : DeviceState.ERROR : shutValue == 1 ? DeviceState.SHUT : DeviceState.TRANSIT;
    }
    
}
