package org.lsst.ccs.subsystem.vacuum;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.pfeiffer.ASM380;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.vacuum.config.VacuumConfig;
import org.lsst.ccs.subsystem.vacuum.constants.VacuumAlert;
import org.lsst.ccs.subsystem.vacuum.constants.DeviceState;


/**
 *  Interface to the pump cart device (ASM380 leak detector)
 *
 *  @author CCS team
 */
public class ASM380Device extends Device {

    /**                                                                                                                                                                                                     
     *  Data fields.                                                                                                                                                                                        
     */
    @ConfigurationParameter(name="devcId", category=VacuumConfig.CRYO, isFinal=true)
    private volatile String devcId;

    private final ASM380.ConnType connType = ASM380.ConnType.SERIAL;
    private final int baudRate = 9600;

    private static final Logger LOG = Logger.getLogger(ASM380Device.class.getName());
    private final ASM380 mydev = new ASM380();
    private int errorCount = 0;
    private boolean initError = false;
    private final Map<String, Boolean> activeAlerts = new HashMap<>();


    /**
     *  Performs configuration.
     */
    @Override
    public void initDevice() {
       
        if (devcId == null) {
            ErrorUtils.reportConfigError(LOG, name, "devcId", "is missing");
        }
	
        fullName = "ASM380";

        //Register the Alerts raised by this Device.                                                                                                                                                       
        // ex:    alertService.registerAlert(ASM380Alert.XXXBAD.getAlert(name, rNum));
    }

    /**
     *  Performs full initialization.
     */
    @Override
    public void initialize()
    {
        try {
	    //            mydev.setRetryLimit(3);
            mydev.open(connType, devcId, baudRate, 0);
            errorCount = 0;

            LOG.log(Level.INFO, "Connected to {0}",
                    new Object[]{fullName});
            initError = false;
            setOnline(true);
        }
        catch (DriverException e) {
            if (!initError) {
                LOG.log(Level.SEVERE, "Error connecting to {0}: {1}", new Object[]{fullName, e});
                initError = true;
            }
            try {
                mydev.close();
            }
            catch (DriverException ce) {
                // Here if device wasn't opened
            }
        }
    }

    /**
     *  Closes the connection.
     */
    @Override
    public void close()
    {
        try {
            mydev.close();
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error disconnecting from {0}: {1}", new Object[]{fullName, e});
        }
    }



    /*
     *  Reads a monitoring channel.
     *
     *  @param  ch  The channel to read
     *  @return  The read value
     */
    @Override
	 public double readChannel(Channel ch)
    {
        double value = Double.NaN;
        if (isOnline()) {
	    switch(ch.getHwChan()) {
	    case 0:
		try {
		    value = mydev.readInletPressure();
		    errorCount = 0;
		}
		catch (DriverException e) {
		    LOG.log(Level.SEVERE, "Error reading pressure from {0}: {1}", new Object[]{fullName, e});
		    if (++errorCount >= 5) {
			setOnline(false);
		    }
		}
		break;
	    case 1:
		try {
		    value = getPumpStatus().contains("on") ? 1.0 : 0.0 ;
		    errorCount = 0;
		}
		catch (DriverException e) {
		    LOG.log(Level.SEVERE, "Error reading pump cycle enabled status from {0}: {1}", new Object[]{fullName, e});
		    if (++errorCount >= 5) {
			setOnline(false);
		    }
		}
		break;
	    case 2:
		try {
		    value = mydev.readTemperature();
		    errorCount = 0;
		}
		catch (DriverException e) {
		    LOG.log(Level.SEVERE, "Error reading temperature from {0}: {1}", new Object[]{fullName, e});
		    if (++errorCount >= 5) {
			setOnline(false);
		    }
		}
		break;
	    case 4:
		try {
		    value = getVentValveStatus().contains("open") ? 1.0 : 0.0 ;
		    errorCount = 0;
		}
		catch (DriverException e) {
		    LOG.log(Level.SEVERE, "Error reading venting status from {0}: {1}", new Object[]{fullName, e});
		    if (++errorCount >= 5) {
			setOnline(false);
		    }
		}
		break;
	    }
        }
        return value;
    }


    @Command(description = "Show the version string")
    public String showVersion() throws DriverException {
        return mydev.getVersion();
    }

    @Command(description = "Show the instrument status")
    public String showStatus() throws DriverException {
        return String.format("0x%04x", mydev.getStatus());
    }

    @Command(description = "Show the decoded instrument status")
    public String showDecodedStatus() throws DriverException {
        String[] status = mydev.decodeStatus();
        StringBuilder text = new StringBuilder();
        for (int j = 0; j < status.length; j++) {
            if (j != 0) {
                text.append("\n");
            }
            text.append(status[j]);
        }
        return text.toString();
    }

    @Command(description = "Read the inlet pressure")
    public double readInletPressure() throws DriverException {
        return mydev.readInletPressure();
    }

    @Command(description = "Read the temperature")
    public int readTemperature() throws DriverException {
        return mydev.readTemperature();
    }

    @Command(description = "Show the primary pump operating hours")
    public int showPrimaryHours() throws DriverException {
        return mydev.getPrimaryHours();
    }

    @Command(description = "Show the high vacuum pump operating hours")
    public int showHighVacHours() throws DriverException {
        return mydev.getHighVacHours();
    }

    @Command(description = "Show the stored warning codes")
    public String showWarningCodes() throws DriverException {
        return showCodes(mydev.getWarningCodes());
    }

    @Command(description = "Show the stored fault codes")
    public String showFaultCodes() throws DriverException {
        return showCodes(mydev.getFaultCodes());
    }

    private static String showCodes(int[] codes) {
        if (codes.length == 0) {
            return "No codes stored";
        }
        else {
            StringBuilder text = new StringBuilder();
            for (int code : codes) {
                text.append(code).append(" ");
            }
            return text.toString();
        }
    }

    @Command(description = "Clear the stored warning codes")
    public void clearWarnings() throws DriverException {
        mydev.clearWarnings();
    }

    @Command(description = "Clear the stored fault codes")
    public void clearFaults() throws DriverException {
        mydev.clearFaults();
    }

    @Command(description = "set pump on (true) or off (false)")
    public void pumpOn(boolean on) throws DriverException {
	if (!getVentValveStatus().contains("closed")) {
	    //	    raiseAlarm(VacuumAlert.BOTH_PUMPING_VENTING,
	    LOG.log(Level.SEVERE,"Rejected attempt to turn pumping on with the valve open.");
	} else {
	    mydev.pumpOn(on);
	}
    }

    @Command(description = "get the pumping status")
    public String getPumpStatus() throws DriverException {
        return mydev.getPumpStatus();
    }

    @Command(description = "execute open vent sequence")
    public void openVentValve(boolean open) throws DriverException {
	mydev.setVentValveNominal();
        mydev.openVentValve(open);
    }

    @Command(description = "get the venting status")
    public String getVentValveStatus() throws DriverException {
        return mydev.getVentValveStatus();
    }

    @Command(description = "get the venting valve actuation parameters")
    public String getVentValveParms() throws DriverException {
        return mydev.getVentValveParms();
    }

    @Command(description = "setup venting sequence")
    public void setVentValveParms(
		 @Argument(description="delay before actuation (0->2 secs)") int delay,
		 @Argument(description="time (secs) in open state") int period) throws DriverException {
        mydev.setVentValveParms(delay,period);
    }

    @Command(description = "execute pulse vent sequence")
    public void pulseVentValve(@Argument(description="time (secs) in open state") int seconds) throws DriverException {

        mydev.setVentValveParms(0,seconds);
	//	try {
	mydev.openVentValve(true);
	try {
	    Thread.sleep(1000*(seconds+1));
	} catch (InterruptedException ex) {
            LOG.log(Level.SEVERE, "Interrupted!! Prematurely stopping the pumping" + ex);
	}
	//        } catch (InterruptedException ex) {
	//            LOG.log(Level.SEVERE, "Interrupted! Prematurely stopping the venting" + ex);
	//	}
	mydev.setVentValveNominal();
    }

    @Command(description = "execute pulse cycling/pumping")
    public void pulsePump(@Argument(description="time (secs) in pumping state") int seconds) throws DriverException {
        mydev.pumpOn(true);
	try {
	    Thread.sleep(1000*seconds);
        } catch (InterruptedException ex) {
            LOG.log(Level.SEVERE, "Interrupted!! Prematurely stopping the pumping" + ex);
	}
	mydev.pumpOn(false);
    }

    @Command(description = "set venting sequence nominal")
    public void setVentValveNominal() throws DriverException {
	mydev.setVentValveNominal();
    }


    @Command(description = "Write a command")
    public String writeCommand(@Argument(description="The commmand to send") String command) throws DriverException {
         return mydev.writeCommand(command);
    }

    /**
     *  Raise a relay configuration alarm
     *
     *  @param  alert  The alarm to lower
     *  @param  cause  The alarm cause
     */
    private void raiseAlarm(Alert alert, String cause)
    {
        Boolean active = activeAlerts.get(alert.getAlertId());
        if (active != Boolean.TRUE) {
            alertService.raiseAlert(alert, AlertState.ALARM, cause);
            activeAlerts.put(alert.getAlertId(), true);
        }
    }


    /**
     *  Lower a relay configuration alarm
     *
     *  @param  alert  The alarm to lower
     *  @param  cause  The alarm cause
     */
    private void lowerAlarm(Alert alert, String cause)
    {
        Boolean active = activeAlerts.get(alert.getAlertId());
        if (active == Boolean.TRUE) {
            alertService.raiseAlert(alert, AlertState.NOMINAL, cause);
            activeAlerts.put(alert.getAlertId(), false);
        }
    }

    /**
     *  Gets the device state.
     *
     *  @return  The device state
     **/
    public DeviceState getDeviceState()
    {

	try {
	    if (!isOnline()) {
		return DeviceState.OFFLINE;
	    } else if (getPumpStatus().contains("on")) {
		return DeviceState.NORMAL;
	    } else if (getPumpStatus().contains("off")) {
		return DeviceState.STOPPED;
	    } else {
		return DeviceState.TRANSIT;
	    }
	} catch (DriverException e) {
            return null;
        }

    }


}
