package org.lsst.ccs.subsystem.vacuum;

import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.subsystem.common.devices.dataforth.Maq20Device;
import org.lsst.ccs.subsystem.common.devices.dataforth.Maq20DiscControl;
import org.lsst.ccs.subsystem.vacuum.constants.Devices;

/**
 *  Handles the vacuum system MAQ20 device with switches.
 *
 *  @author Owen Saxton
 */
public class VacMaq20Device extends Maq20Device implements SwitchDevice {

    /**
     *  Constants.
     */
    public static final int
        SW_INST_FTPP_VALVE  = 0,
        SW_INST_FTH_VALVE   = 1,
        SW_INST_L3H_VALVE   = 2,
        SW_INST_L3LF_VALVE  = 3,
        SW_INST_FLINE_VALVE = 4,
        SW_CRYO_FLINE_VALVE = 5,
        SW_HX_FLINE_VALVE   = 6,
        NUM_SWITCHES        = 7;

    @LookupField(strategy=LookupField.Strategy.CHILDREN)
    protected Maq20DiscControl maqCtrl;


    /**
     *  Device initialization.
     */
    @Override
    protected void initDevice()
    {
        super.initDevice();
        if (maqCtrl == null) {
            throw new RuntimeException(path + ": Child Maq20DiscControl node has not been defined");
        }
    }


    /**
     *  Gets the switch device ID.
     * 
     *  @return  The device ID
     */
    @Override
    public int getSwitchDevice()
    {
        return Devices.DEVC_MAQ20;
    }


    /**
     *  Sets a switch on or off.
     *
     *  For the vacuum Pluto, this is implemented as a pair of push buttons,
     *  one for on, one for off.
     *
     *  @param  sw  The switch number.
     *  @param  on  The on state to set: true or false
     *  @throws DriverException
     */
    @Override
    public void setSwitch(int sw, boolean on) throws DriverException
    {
         maqCtrl.setLineOn(sw, on);
    }


    /**
     *  Gets the on state of a switch.
     *
     *  @param  sw  The switch number.
     *  @return  Whether the switch is on
     */
    @Override
    public Boolean isSwitchOn(int sw)
    {
        return maqCtrl.isLineOn(sw);
    }

}
