package org.lsst.ccs.subsystem.vacuum;

import java.util.ArrayList;
import java.time.Duration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.alert.AlertEvent;
import org.lsst.ccs.services.alert.AlertListener;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.common.MonitorTaskControl;
import org.lsst.ccs.subsystem.vacuum.constants.VacuumAlert;
import org.lsst.ccs.subsystem.vacuum.constants.ConditionState;
import org.lsst.ccs.subsystem.vacuum.constants.Conditions;
import org.lsst.ccs.subsystem.vacuum.constants.DeviceState;
import org.lsst.ccs.subsystem.vacuum.constants.Devices;
import org.lsst.ccs.subsystem.vacuum.constants.LatchState;
import org.lsst.ccs.subsystem.vacuum.constants.Latches;
import org.lsst.ccs.subsystem.vacuum.constants.MonitorControl;
import org.lsst.ccs.subsystem.vacuum.constants.PLCState;
import org.lsst.ccs.subsystem.vacuum.constants.SwitchEnable;
import org.lsst.ccs.subsystem.vacuum.constants.Switches;
import org.lsst.ccs.subsystem.vacuum.constants.SwitchState;
import org.lsst.ccs.subsystem.vacuum.constants.VacuumAgentProperties;
import org.lsst.ccs.subsystem.vacuum.constants.CryoVacuumState;
import org.lsst.ccs.subsystem.vacuum.constants.HxVacuumState;
import org.lsst.ccs.subsystem.vacuum.data.VacuumException;
import org.lsst.ccs.subsystem.vacuum.data.VacSysState;

/**
 * The camera vacuum system
 *
 * @author The LSST CCS Team
 */
public class VacuumMain extends Subsystem implements HasLifecycle, ClearAlertHandler, AlertListener {

    static class SensorData {
        Channel channel;
        int maxErrors;
        int numErrors = 0;
        double value = 0.0;
        double goodValue = Double.NaN;

        private SensorData(Channel channel, int maxErrors) {
            this.channel = channel;
            this.maxErrors = maxErrors;
        }
    }

    private static final double
        PRESS_ATMOS = 759.0,
        PRESS_TURBO_LOW = 5.0,
        PRESS_FORELINE_LOW = 5.0,
        PRESS_DIFF_LOW = 0.09,
        PRESS_DIFF_HIGH = 20.0,
        PRESS_ION_OFF = 1.1e-5,
        PRESS_ION_ENABLE = 1.0e-6,
        PRESS_VACUUM = 1.0e-7,
        PRESS_REFRIG_OK = 9.0e-3,
        PRESS_CMP_AIR_LOW = 80.0,
        SPEED_CRYO_TURBO_MAX = 60000,
        SPEED_CRYO_TURBO_LOW = 0.1 * SPEED_CRYO_TURBO_MAX,
        SPEED_HX_TURBO_MAX = 81000,
        SPEED_HX_TURBO_LOW = 0.1 * SPEED_HX_TURBO_MAX;
    private static final int
        MAX_PRESS_ERRORS = 2,
        MAX_SPEED_ERRORS = 0,
        MAX_AIR_ERRORS = 0,
        DELAY_ION_OFF = 10000,
        DELAY_VACUUM_BAD = 10000;
    private static final Map<Integer, VacuumAlert> latchAlertMap = new HashMap<>();
    static {
        latchAlertMap.put(Latches.LATCH_CR_VACUUM, VacuumAlert.CRYO_VACUUM_BAD);
        latchAlertMap.put(Latches.LATCH_CR_GATE_NFC, VacuumAlert.CRYO_GATE_FORCED_SHUT);
        latchAlertMap.put(Latches.LATCH_CR_GATE_AO, VacuumAlert.CRYO_GATE_CANNOT_OPEN);
        latchAlertMap.put(Latches.LATCH_CR_PUMP, VacuumAlert.CRYO_TURBO_PUMP_BAD);
        latchAlertMap.put(Latches.LATCH_HX_VACUUM, VacuumAlert.HX_VACUUM_BAD);
        latchAlertMap.put(Latches.LATCH_HX_GATE_NFC, VacuumAlert.HX_GATE_FORCED_SHUT);
        latchAlertMap.put(Latches.LATCH_HX_GATE_AO, VacuumAlert.HX_GATE_CANNOT_OPEN);
        latchAlertMap.put(Latches.LATCH_HX_PUMP, VacuumAlert.HX_TURBO_PUMP_BAD);
    }
    private static final Map<String, Integer> revLatchAlertMap = new HashMap<>();
    static {
        for (int cond : latchAlertMap.keySet()) {
            revLatchAlertMap.put(latchAlertMap.get(cond).getId(), cond);
        }
    }
    private static final List<Integer> cryoIonPumps = new ArrayList<>();
    static {
        cryoIonPumps.add(Switches.SW_CRYO_ION_PUMP1);
        cryoIonPumps.add(Switches.SW_CRYO_ION_PUMP2);
        cryoIonPumps.add(Switches.SW_CRYO_ION_PUMP3);
        cryoIonPumps.add(Switches.SW_CRYO_ION_PUMP4);
        cryoIonPumps.add(Switches.SW_CRYO_ION_PUMP5);
        cryoIonPumps.add(Switches.SW_CRYO_ION_PUMP6);
    }
    private static final List<Integer> hxIonPumps = new ArrayList<>();
    static {
        hxIonPumps.add(Switches.SW_HX_ION_PUMP1);
        hxIonPumps.add(Switches.SW_HX_ION_PUMP2);
    }
    private static final List<Integer> utSwitches = new ArrayList<>();
    static {
        utSwitches.add(Switches.SW_INST_FTH_VALVE);
        utSwitches.add(Switches.SW_INST_FTPP_VALVE);
        utSwitches.add(Switches.SW_INST_L3H_VALVE);
        utSwitches.add(Switches.SW_INST_L3LF_VALVE);
        utSwitches.add(Switches.SW_INST_SCROLL_PUMP);
        utSwitches.add(Switches.SW_CRYO_SCROLL_PUMP);
        utSwitches.add(Switches.SW_HX_SCROLL_PUMP);
    }

    @LookupName
    private String name;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService propertiesService;

    @LookupField(strategy=LookupField.Strategy.DESCENDANTS)
    private final List<SwitchDevice> switchDevcs = new ArrayList<>();
    @LookupField(strategy=LookupField.Strategy.DESCENDANTS)
    private MpmPlutoDevice mpmPlutoDevc;

    // From Groovy file
    private Channel cryoVacPressure, cryoTurboPressure, cryoFlinePressure, cryoTurboSpeed,
                    hxVacPressure, hxTurboPressure, hxFlinePressure, hxTurboSpeed, airPressure;

    // General
    private static final Logger LOG = Logger.getLogger(VacuumMain.class.getName());
    private MonitorTaskControl monitorControl;
    private final SwitchDevice[] switchDevices = new SwitchDevice[Devices.NUM_DEVICES];
    private VacPlutoDevice vacPlutoDevc;
    private IonPumpDevice ionPumpDevc;
    private CryoTurboDevice cryoTurboDevc;
    private HxTurboDevice hxTurboDevc;
    private final VacSysState vacState = new VacSysState();
    private boolean running = false;
    private final Map<String, Boolean> activeAlarmMap = new HashMap<>();
    private long cryoIonOverStartTime = 0, hxIonOverStartTime = 0;
    private long vacBadTime = 0;
    private SensorData cryoFlinePr, cryoMainPr, cryoTurboPr, cryoTurboSp, hxFlinePr, hxMainPr, hxTurboPr, hxTurboSp, cmpAirPr;


    /**
     *  Constructor
     */
    public VacuumMain() {
        super("vacuum", AgentInfo.AgentType.WORKER);
        getAgentInfo().getAgentProperties().setProperty("org.lsst.ccs.use.full.paths", "true");
    }


    /**
     *  Build phase
     */
    @Override
    public void build() {
        // Create the monitor task control object and node
        monitorControl = MonitorTaskControl.createNode(this, MonitorControl.NODE_NAME);

        // Create and schedule an AgentPeriodicTask to update the vacuum state
        AgentPeriodicTask pt;
        pt = new AgentPeriodicTask("vacuum-state", () -> updateVacuumState()).withPeriod(Duration.ofMillis(1000));
        periodicTaskService.scheduleAgentPeriodicTask(pt);

        // Register vacuum states
        stateService.registerState(CryoVacuumState.class, "Cryo vacuum state", this);
        stateService.updateAgentState(CryoVacuumState.UNKNOWN);
        stateService.registerState(HxVacuumState.class, "HX vacuum state", this);
        stateService.updateAgentState(HxVacuumState.UNKNOWN);

        // Initialize active alert map
        for (VacuumAlert alert : VacuumAlert.values()) {
            activeAlarmMap.put(alert.getId(), false);
        }
    }


    /**
     *  Post initialization phase
     */
    @Override
    public void postInit() {

        //Set a property to define that this Agent is a vacuum subsystem.
        propertiesService.setAgentProperty(VacuumAgentProperties.VACUUM_TYPE, VacuumMain.class.getCanonicalName());

        // Add alert listener
        alertService.addListener(this);

        for (SwitchDevice device : switchDevcs) {
            switchDevices[device.getSwitchDevice()] = device;
        }
        for (int id = 0; id < Devices.NUM_DEVICES; id++) {
            if (switchDevices[id] == null) {
                ErrorUtils.reportConfigError(LOG, name, Devices.getDescription(id) + " device", "not defined");
            }
        }
        vacPlutoDevc = (VacPlutoDevice)switchDevices[Devices.DEVC_PLUTO];
        ionPumpDevc = (IonPumpDevice)switchDevices[Devices.DEVC_ION_PUMP];
        cryoTurboDevc = (CryoTurboDevice)switchDevices[Devices.DEVC_CRYO_TURBO_PUMP];
        hxTurboDevc = (HxTurboDevice)switchDevices[Devices.DEVC_HX_TURBO_PUMP];
        if (mpmPlutoDevc == null) {
            ErrorUtils.reportConfigError(LOG, name, "MpmPlutoDevice", "not defined");
        }

        if (cryoFlinePressure == null) {
            ErrorUtils.reportConfigError(LOG, name, "Cryo foreline pressure channel", "not specified");
        }
        cryoFlinePr = new SensorData(cryoFlinePressure, MAX_PRESS_ERRORS);
        if (cryoVacPressure == null) {
            ErrorUtils.reportConfigError(LOG, name, "Cryo vacuum pressure channel", "not specified");
        }
        cryoMainPr = new SensorData(cryoVacPressure, MAX_PRESS_ERRORS);
        ionPumpDevc.setCryoPressureChannel(cryoVacPressure);
        if (cryoTurboPressure == null) {
            ErrorUtils.reportConfigError(LOG, name, "Cryo turbo pump pressure channel", "not specified");
        }
        cryoTurboPr = new SensorData(cryoTurboPressure, MAX_PRESS_ERRORS);
        if (cryoTurboSpeed == null) {
            ErrorUtils.reportConfigError(LOG, name, "Cryo turbo pump speed channel", "not specified");
        }
        cryoTurboSp = new SensorData(cryoTurboSpeed, MAX_SPEED_ERRORS);
        if (hxFlinePressure == null) {
            ErrorUtils.reportConfigError(LOG, name, "HX foreline pressure channel", "not specified");
        }
        hxFlinePr = new SensorData(hxFlinePressure, MAX_PRESS_ERRORS);
        if (hxVacPressure == null) {
            ErrorUtils.reportConfigError(LOG, name, "HX vacuum pressure channel", "not specified");
        }
        hxMainPr = new SensorData(hxVacPressure, MAX_PRESS_ERRORS);
        ionPumpDevc.setHxPressureChannel(hxVacPressure);
        if (hxTurboPressure == null) {
            ErrorUtils.reportConfigError(LOG, name, "HX turbo pump pressure channel", "not specified");
        }
        hxTurboPr = new SensorData(hxTurboPressure, MAX_PRESS_ERRORS);
        if (hxTurboSpeed == null) {
            ErrorUtils.reportConfigError(LOG, name, "HX turbo pump speed channel", "not specified");
        }
        hxTurboSp = new SensorData(hxTurboSpeed, MAX_SPEED_ERRORS);
        if (airPressure == null) {
            ErrorUtils.reportConfigError(LOG, name, "Compressed air pressure channel", "not specified");
        }
        cmpAirPr = new SensorData(airPressure, MAX_AIR_ERRORS);
    }


    /**
     *  Post start
     */
    @Override
    public void postStart() {
        LOG.info("Vacuum subsystem started");
        running = true;
    }


    /**
     *  Gets the state of the Vacuum system.
     *
     *  @return  The vacuum state
     */
    @Command(type=Command.CommandType.QUERY, description="Get the vacuum system state", level=0)
    public VacSysState getVacuumState() {
        vacState.setTickMillis(monitorControl.getPublishPeriod());
        return vacState;
    }    


    /**
     *  Gets the list of switch names.
     *
     *  @return  The switch names.
     *  @throws  VacuumException
     */
    @Command(type=Command.CommandType.QUERY, description="Get switch names", level=0)
    public List<String> getSwitchNames() throws VacuumException
    {
        return Switches.getNames();
    }


    /**
     *  Turns a switch on or off.
     *
     *  @param  name  The switch name.
     *  @param  on    Whether to turn on or off
     *  @throws  VacuumException
     */
    @Command(type=Command.CommandType.ACTION, description="Turn on/off a named switch")
    public void setSwitchOn(@Argument(description="The switch name") String name,
                            @Argument(description="Whether to turn on") boolean on) throws VacuumException
    {
        try {
            setSwitch(Switches.getId(name), on);
        }
        finally {
            publishState();
        }
    }


    /**
     *  Turns a switch on or off.
     *
     *  @param  sw  The switch number.
     *  @param  on  Whether to turn on or off
     *  @throws  VacuumException
     */
    private void setSwitch(int sw, boolean on) throws VacuumException
    {
        SwitchState state = vacState.getSwitchState(sw);
        if (state == SwitchState.OFFLINE) return;
        SwitchEnable enable = vacState.getSwitchEnable(sw);
        if (on && enable != SwitchEnable.ON) return;
        try {
            switchDevices[Switches.getDevice(sw)].setSwitch(SwitchInfo.getSwitch(sw), on);
        }
        catch (DriverException e) {
            throw new VacuumException("Error setting switch for " + Devices.getDescription(Switches.getDevice(sw)) + " device: " + e);
        }
    }


    /**
     *  Gets whether a switch is on.
     *
     *  @param  sw  The switch number.
     *  @return  Whether the switch is on
     *  @throws  VacuumException
     */
    private Boolean isSwitchOn(int sw)
    {
        return switchDevices[Switches.getDevice(sw)].isSwitchOn(SwitchInfo.getSwitch(sw));
    }


    /**
     *  Clears a latched condition.
     *
     *  @param  cond  The condition number.
     *  @throws  VacuumException
     */
    @Command(type=Command.CommandType.ACTION, description="Clear a condition")
    public void clearLatch(@Argument(description="The condition number") int cond) throws VacuumException
    {
        try {
            vacPlutoDevc.clearLatch(cond);
        }
        finally {
            publishState();
        }
    }


    /**
     *  Updates the vacuum system state periodically.
     *
     *  The vacuum state consists mainly of the state of the switches (lines) being
     *  controlled, along with whether they can be turned on.
     *
     *  Pressures and pump speeds are read to determine which switches are enabled
     *  and/or need to be turned off.  If the state changes it is published.
     */
    private void updateVacuumState()
    {
        if (!running) return;
        boolean changed = monitorControl.hasPeriodChanged();
        changed |= updatePlcState();
        changed |= updateLatchState();
        changed |= updateCondState();
        readSensors();
        changed |= updateSwitchState();
        changed |= updateCryoState();
        changed |= updateHxState();
        checkRefrigVacuum();
        if (changed) {
            publishState();
        }
    }


    /**
     *  Updates the PLC state
     * 
     *  @return  Whether the state changed
     */
    private boolean updatePlcState() {
        boolean changed = false;
        Boolean plcActive = vacPlutoDevc.isPLCActive();
        if (plcActive == null) {
            if (vacState.getPlcState() != PLCState.OFFLINE) {
                raiseAlarm(VacuumAlert.VACUUM_PLC_NOT_ALIVE, "Vacuum PLC is offline");
                vacState.setPlcState(PLCState.OFFLINE);
                changed = true;
            }
        }
        else if (!plcActive) {
            if (vacState.getPlcState() != PLCState.DEAD) {
                raiseAlarm(VacuumAlert.VACUUM_PLC_NOT_ALIVE, "Vacuum PLC has died");
                vacState.setPlcState(PLCState.DEAD);
                changed = true;
            }
        }
        else {
            if (vacState.getPlcState() != PLCState.ALIVE) {
                lowerAlarm(VacuumAlert.VACUUM_PLC_NOT_ALIVE, "Vacuum PLC is alive");
                vacState.setPlcState(PLCState.ALIVE);
                changed = true;
            }
        }
        return changed;
    }


    /**
     *  Updates the state of the PLC latches.
     * 
     *  @return  Whether the state changed
     */
    private boolean updateLatchState() {
        boolean changed = false;
        for (int cond = 0; cond < Latches.NUM_LATCHES; cond++) {
            Boolean active = vacPlutoDevc.isLatchActive(cond);
            Boolean latched = vacPlutoDevc.isLatchLatched(cond);
            LatchState state = active == null || latched == null ? LatchState.OFFLINE :
                               latched ? LatchState.LATCHED :
                               active ? LatchState.ACTIVE : LatchState.CLEAR;
            LatchState oldState = vacState.getLatch(cond); 
            if (state != oldState) {
                vacState.setLatch(cond, state);
                VacuumAlert alert = latchAlertMap.get(cond);
                if (state == LatchState.ACTIVE) {
                    raiseAlarm(alert, "Vacuum PLC error condition set");
                }
                else if (state != LatchState.OFFLINE && oldState != LatchState.OFFLINE) {
                    if (state == LatchState.LATCHED && oldState == LatchState.CLEAR) {
                        raiseAlarm(alert, "Vacuum PLC error condition set");
                    }
                    if (state == LatchState.LATCHED || oldState == LatchState.ACTIVE) {
                        lowerAlarm(alert, "Vacuum PLC error condition cleared");
                    }
                }
                changed = true;
            }
        }
        return changed;
    }


    /**
     *  Updates the state of the PLC conditions.
     * 
     *  @return  Whether the state changed
     */
    private boolean updateCondState() {
        boolean changed = false;
        for (int cond = 0; cond < Conditions.NUM_CONDITIONS; cond++) {
            Boolean active = vacPlutoDevc.isConditionActive(cond);
            ConditionState state = active == null ? ConditionState.OFF :
                                   active ? ConditionState.YES : ConditionState.NO;
            if (state != vacState.getCondition(cond)) {
                vacState.setCondition(cond, state);
                changed = true;
            }
        }
        return changed;
    }


    /**
     *  Reads the gauges and the state of the turbo pumps.
     */
    private void readSensors() {
        readSensor(cryoFlinePr);
        readSensor(cryoMainPr);
        readSensor(cryoTurboPr);
        readSensor(cryoTurboSp);
        readSensor(hxFlinePr);
        readSensor(hxMainPr);
        readSensor(hxTurboPr);
        readSensor(hxTurboSp);
        readSensor(cmpAirPr);
    }


    /**
     *  Reads a sensor, allowing for possible temporary errors.
     * 
     *  @param  data  The sensor data object
     */
    private void readSensor(SensorData data) {
        data.value = data.channel.getValue();
        if (!Double.isNaN(data.value)) {
            data.goodValue = data.value;
            data.numErrors = 0;
        }
        else if (data.numErrors++ < data.maxErrors) {
            data.value = data.goodValue;
        }
    }


    /**
     *  Updates the state of the switches.
     * 
     *  @return  Whether the state changed
     */
    private boolean updateSwitchState() {
        boolean changed = false;
        changed |= updateCryoGateValve();
        changed |= updateCryoTurboPump();
        changed |= updateCryoIonPumps();
        changed |= updateHxGateValve();
        changed |= updateHxTurboPump();
        changed |= updateHxIonPumps();
        changed |= updateUtSwitches();
        return changed;
    }


    /**
     *  Updates the state of the cryo gate valve.
     * 
     *  @return  Whether the state changed
     */
    private boolean updateCryoGateValve()
    {
        boolean enable = false;
        boolean cryoTurbAvail = !Double.isNaN(cryoMainPr.value) && !Double.isNaN(cryoTurboPr.value)
                                  && !Double.isNaN(cryoTurboSp.value);
        if (cryoTurbAvail) {
            double prDiff = Math.abs(cryoTurboPr.value - cryoMainPr.value);
            enable = (prDiff <= PRESS_DIFF_HIGH && cryoTurboSp.value <= SPEED_CRYO_TURBO_LOW)
                        || (prDiff <= PRESS_DIFF_LOW && cryoTurboSp.value > SPEED_CRYO_TURBO_LOW);
        }
        boolean enable2 = cryoTurbAvail && cmpAirPr.value > PRESS_CMP_AIR_LOW
                            && (cryoFlinePr.value < PRESS_FORELINE_LOW || cryoTurboSp.value <= SPEED_CRYO_TURBO_LOW);
        enable &= enable2;
        return updateSwitch(Switches.SW_CRYO_GATE_VALVE, enable, !enable2, null, VacuumAlert.CRYO_GATE_CLOSED);
    }


    /**
     *  Updates the state of the cryo turbo pump.
     * 
     *  @return  Whether the state changed
     */
    private boolean updateCryoTurboPump()
    {
        boolean enable = false;
        if (!Double.isNaN(cryoTurboPr.value) && !Double.isNaN(cryoFlinePr.value) && !Double.isNaN(cryoTurboSp.value)) {
            enable = cryoTurboPr.value < PRESS_TURBO_LOW
                       && (cryoFlinePr.value < PRESS_FORELINE_LOW || cryoTurboSp.value < SPEED_CRYO_TURBO_LOW);
        }
        return updateSwitch(Switches.SW_CRYO_TURBO_PUMP, enable, !enable, cryoTurboDevc.getDeviceState(), VacuumAlert.CRYO_TURBO_PUMP_STOPPED);
    }


    /**
     *  Updates the state of the cryo ion pumps.
     * 
     *  @return  Whether the state changed
     */
    private boolean updateCryoIonPumps()
    {
        boolean changed = false, enable = false, turnOff = false;
        if (!Double.isNaN(cryoMainPr.value)) {
            enable = cryoMainPr.value < PRESS_ION_ENABLE;
            if (cryoMainPr.value >= PRESS_ION_OFF) {
                if (cryoIonOverStartTime == 0) {
                    cryoIonOverStartTime = System.currentTimeMillis();
                }
                else {
                    turnOff = System.currentTimeMillis() - cryoIonOverStartTime >= DELAY_ION_OFF;
                }
            }
            else {
                cryoIonOverStartTime = 0;
            }
        }
        else {
            turnOff = true;
        }
        for (int sw : cryoIonPumps) {
            changed |= updateSwitch(sw, enable, turnOff, null, VacuumAlert.CRYO_ION_PUMPS_STOPPED);
        }
        return changed;
    }


    /**
     *  Updates the state of the HX gate valve.
     * 
     *  @return  Whether the state changed
     */
    private boolean updateHxGateValve()
    {
        boolean enable = false;
        boolean hxTurbAvail = !Double.isNaN(hxMainPr.value) && !Double.isNaN(hxTurboPr.value) && !Double.isNaN(hxTurboSp.value);
        if (hxTurbAvail) {
            double prDiff = Math.abs(hxTurboPr.value - hxMainPr.value);
            enable = (prDiff <= PRESS_DIFF_HIGH && hxTurboSp.value <= SPEED_HX_TURBO_LOW)
                        || (prDiff <= PRESS_DIFF_LOW && hxTurboSp.value > SPEED_HX_TURBO_LOW);
        }
        boolean enable2 = hxTurbAvail && cmpAirPr.value > PRESS_CMP_AIR_LOW
                            && (hxFlinePr.value < PRESS_FORELINE_LOW || hxTurboSp.value <= SPEED_HX_TURBO_LOW);
        enable &= enable2;
        return updateSwitch(Switches.SW_HX_GATE_VALVE, enable, !enable2, null, VacuumAlert.HX_GATE_CLOSED);
    }


    /**
     *  Updates the state of the HX turbo pump.
     * 
     *  @return  Whether the state changed
     */
    private boolean updateHxTurboPump()
    {
        boolean enable = false;
        if (!Double.isNaN(hxTurboPr.value) && !Double.isNaN(hxFlinePr.value) && !Double.isNaN(hxTurboSp.value)) {
            enable = hxTurboPr.value < PRESS_TURBO_LOW
                       && (hxFlinePr.value < PRESS_FORELINE_LOW || hxTurboSp.value < SPEED_HX_TURBO_LOW);
        }
        return updateSwitch(Switches.SW_HX_TURBO_PUMP, enable, !enable, hxTurboDevc.getDeviceState(), VacuumAlert.HX_TURBO_PUMP_STOPPED);
    }


    /**
     *  Updates the state of the HX ion pumps.
     * 
     *  @return  Whether the state changed
     */
    private boolean updateHxIonPumps()
    {
        boolean changed = false, enable = false, turnOff = false;
        if (!Double.isNaN(hxMainPr.value)) {
            enable = hxMainPr.value < PRESS_ION_ENABLE;
            if (hxMainPr.value >= PRESS_ION_OFF) {
                if (hxIonOverStartTime == 0) {
                    hxIonOverStartTime = System.currentTimeMillis();
                }
                else {
                    turnOff = System.currentTimeMillis() - hxIonOverStartTime >= DELAY_ION_OFF;
                }
            }
            else {
                hxIonOverStartTime = 0;
            }
        }
        else {
            turnOff = true;
        }
        for (int sw : hxIonPumps) {
            changed |= updateSwitch(sw, enable, turnOff, null, VacuumAlert.HX_ION_PUMPS_STOPPED);
        }
        return changed;
    }


    /**
     *  Updates the state of the UT switches (scroll pumps & interstitial valves).
     * 
     *  @return  Whether the state changed
     */
    private boolean updateUtSwitches()
    {
        boolean changed = false;
        for (int sw : utSwitches) {
            changed |= updateSwitch(sw, true, false, null, null);
        }
        return changed;
    }


    /**
     *  Updates the state of a switch.
     * 
     *  @param sw        The switch ID
     *  @param enable    Whether to enable it
     *  @param turnOff   Whether to turn it off
     *  @param devState  The state of the switch's device (turbo pump only)
     *  @param alert     The alert to raise/lower if on/off state change was forced
     *  @return  Whether the state changed
     */
    private boolean updateSwitch(int sw, boolean enable, boolean turnOff, DeviceState devState, VacuumAlert alert)
    {
        boolean changed = false;
        SwitchState oldState = vacState.getSwitchState(sw);
        if (turnOff && oldState == SwitchState.ON) {
            try {
                setSwitch(sw, false);
                raiseAlarm(alert, "Switch was forced to open");
            }
            catch (VacuumException e) {
                LOG.log(Level.SEVERE, "Error setting switch: {0}", e);
            }
        }
        Boolean isOn = isSwitchOn(sw);
        SwitchState state = isOn != null ? isOn ? SwitchState.ON : SwitchState.OFF : SwitchState.OFFLINE;
        SwitchEnable enabled = enable ? SwitchEnable.ON : SwitchEnable.OFF;
        if (state != oldState || enabled != vacState.getSwitchEnable(sw)) {
            vacState.setSwitchState(sw, state);
            vacState.setSwitchEnable(sw, enabled);
            changed = true;
            if (enable && alert != null && isAlarmRaised(alert)) {
                lowerAlarm(alert, "Switch has become re-enabled");
            }
        }
        if (devState != vacState.getDeviceState(sw)) {
            vacState.setDeviceState(sw, devState);
            changed = true;
        }
        return changed;
    }


    /**
     *  Checks whether the vacuum is good enough to allow refrigeration.
     */
    private void checkRefrigVacuum() {
        boolean haveAlarm = isAlarmRaised(VacuumAlert.REFRIG_NOT_PERMITTED);
        if (cryoMainPr.value < PRESS_REFRIG_OK && hxMainPr.value < PRESS_REFRIG_OK) {  // Fails if either is NaN
            if (haveAlarm) {
                lowerAlarm(VacuumAlert.REFRIG_NOT_PERMITTED, "Cryo & HX vacuums are good (< " + PRESS_REFRIG_OK + ")");
                mpmPlutoDevc.setSwitch(MpmPlutoDevice.SW_BLOCK_COLD_REFG, false);
                mpmPlutoDevc.setSwitch(MpmPlutoDevice.SW_BLOCK_CRYO_REFG, false);
            }
            vacBadTime = 0;
        }
        else {
            long time = System.currentTimeMillis();
            if (vacBadTime == 0) {
                vacBadTime = time;
            }
            else {
                if (time - vacBadTime >= DELAY_VACUUM_BAD) {
                    if (!haveAlarm) {
                        raiseAlarm(VacuumAlert.REFRIG_NOT_PERMITTED, "Cryo or HX vacuum is bad (>= " + PRESS_REFRIG_OK + ")");
                    }
                    mpmPlutoDevc.setSwitch(MpmPlutoDevice.SW_BLOCK_COLD_REFG, true);
                    mpmPlutoDevc.setSwitch(MpmPlutoDevice.SW_BLOCK_CRYO_REFG, true);
                }
            }
        }
    }


    /**
     *  Updates the summary cryo vacuum state.
     * 
     *  @return  Whether the state changed
     */
    private boolean updateCryoState() {
        boolean changed = false;
        CryoVacuumState cvState;
        if (Double.isNaN(cryoMainPr.value)) {
            cvState = CryoVacuumState.UNKNOWN;
        }
        else if (cryoMainPr.value <= PRESS_VACUUM) {
            cvState = CryoVacuumState.VACUUM;
        }
        else if (vacState.getSwitchState(Switches.SW_CRYO_ION_PUMP1) == SwitchState.ON) {
            cvState = CryoVacuumState.ION_ON;
        }
        else if (vacState.getSwitchEnable(Switches.SW_CRYO_ION_PUMP1) == SwitchEnable.ON) {
            cvState = CryoVacuumState.ION_OFF;
        }
        else if (vacState.getSwitchState(Switches.SW_CRYO_TURBO_PUMP) == SwitchState.ON) {
            cvState = CryoVacuumState.TURBO_ON;
        }
        else if (vacState.getSwitchEnable(Switches.SW_CRYO_TURBO_PUMP) == SwitchEnable.ON) {
            cvState = CryoVacuumState.TURBO_OFF;
        }
        else if (cryoMainPr.value < PRESS_ATMOS) {
            cvState = CryoVacuumState.FORELINE;
        }
        else {
            cvState = CryoVacuumState.OFF;
        }
        if (cvState != vacState.getCryoVacuumState()) {
            vacState.setCryoVacuumState(cvState);
            stateService.updateAgentState(cvState);
            changed = true;
        }
        return changed;
    }


    /**
     *  Updates the summary HX vacuum state.
     * 
     *  @return  Whether the state changed
     */
    private boolean updateHxState() {
        boolean changed = false;
        HxVacuumState hvState;
        if (Double.isNaN(hxMainPr.value)) {
            hvState = HxVacuumState.UNKNOWN;
        }
        else if (hxMainPr.value <= PRESS_VACUUM) {
            hvState = HxVacuumState.VACUUM;
        }
        else if (vacState.getSwitchState(Switches.SW_HX_ION_PUMP1) == SwitchState.ON) {
            hvState = HxVacuumState.ION_ON;
        }
        else if (vacState.getSwitchEnable(Switches.SW_HX_ION_PUMP1) == SwitchEnable.ON) {
            hvState = HxVacuumState.ION_OFF;
        }
        else if (vacState.getSwitchState(Switches.SW_HX_TURBO_PUMP) == SwitchState.ON) {
            hvState = HxVacuumState.TURBO_ON;
        }
        else if (vacState.getSwitchEnable(Switches.SW_HX_TURBO_PUMP) == SwitchEnable.ON) {
            hvState = HxVacuumState.TURBO_OFF;
        }
        else if (hxMainPr.value < PRESS_ATMOS) {
            hvState = HxVacuumState.FORELINE;
        }
        else {
            hvState = HxVacuumState.OFF;
        }
        if (hvState != vacState.getHxVacuumState()) {
            vacState.setHxVacuumState(hvState);
            stateService.updateAgentState(hvState);
            changed = true;
        }
        return changed;
    }


    /**
     *  Raises an alarm.
     *
     *  @param  alert  The vacuum alert to raise to alarm state
     *  @param  cond   The alert condition
     */
    private void raiseAlarm(VacuumAlert alert, String cond)
    {
        alertService.raiseAlert(alert.getAlert(), AlertState.ALARM, cond);
        activeAlarmMap.put(alert.getId(), true);
    }


    /**
     *  Lowers an alarm.
     *
     *  @param  alert  The vacuum alert to lower to normal state
     *  @param  cond   The alert condition
     */
    private void lowerAlarm(VacuumAlert alert, String cond)
    {
        alertService.raiseAlert(alert.getAlert(), AlertState.NOMINAL, cond);
        activeAlarmMap.put(alert.getId(), false);
    }


     /**
     *  Checks whether an alert is in alarm state.
     *
     *  @param  alert  The vacuum alert to check
     *  @return  Whether it has been raised
     */
    private boolean isAlarmRaised(VacuumAlert alert)
    {
        return activeAlarmMap.get(alert.getId()) == Boolean.TRUE;
    }


    /**
     * Callback to clear an {@code Alert} instance.
     * 
     * @param alert The Alert instance to clear.
     * @param alertState The AlertState for the provided Alert.
     * @return A ClearAlertCode to indicate which action is to be taken
     *         by the framework.
     */
    @Override
    public ClearAlertCode canClearAlert(Alert alert, AlertState alertState)
    {
        Boolean active = activeAlarmMap.get(alert.getAlertId());
        return active == null ? ClearAlertCode.UNKNOWN_ALERT
                              : active ? ClearAlertCode.DONT_CLEAR_ALERT : ClearAlertCode.CLEAR_ALERT;
    }


    /**
     *  Alert event handler.
     *
     *  Resets PLC latch when corresponding alert is cleared.
     *
     *  @param  event  The alert event
     */
    @Override
    public void onAlert(AlertEvent event)
    {
        if (event.getType() != AlertEvent.AlertEventType.ALERT_CLEARED) return;
        for (String id : event.getClearedIds()) {
            Integer cond = revLatchAlertMap.get(id);
            if (cond != null) {
                try {
                    clearLatch(cond);
                }
                catch (VacuumException e) {
                    LOG.log(Level.SEVERE, "Error clearing latched PLC condition ({0}): {1}", new Object[]{cond, e});
                }
            }
        }
    }


    /**
     *  Publishes the state of the vacuum system.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    private void publishState()
    {
        publishSubsystemDataOnStatusBus(new KeyValueData(VacSysState.KEY, getVacuumState()));
    }

}
