package org.lsst.ccs.subsystem.vacuum;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.auxelex.IonPump;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.localdb.statusdb.server.TrendingData;
import org.lsst.ccs.localdb.trendserver.TrendingClientService;
import org.lsst.ccs.localdb.utils.TrendingConnectionUtils;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.vacuum.config.VacuumConfig;
import org.lsst.ccs.subsystem.vacuum.constants.Devices;
import org.lsst.ccs.subsystem.vacuum.data.VacuumException;

/**
 *  Interface to a SLAC ion pump controller.
 *
 *  @author Owen Saxton
 */
public class IonPumpDevice extends Device implements SwitchDevice {

    /**
     *  Constants.
     */
    public static final int
        CHAN_HIP1 = 0,
        CHAN_HIP2 = 1,
        CHAN_OIP  = 2,
        CHAN_CIP1 = 3,
        CHAN_CIP2 = 4,
        CHAN_CIP3 = 5,
        CHAN_CIP4 = 6,
        CHAN_CIP5 = 7,
        CHAN_CIP6 = 8;

    private static final int
        MON_TYPE_VOLTAGE  = 0,
        MON_TYPE_CURRENT  = 1,
        MON_TYPE_POWER    = 2,
        MON_TYPE_USAGE    = 3,
        MON_TYPE_LIFETIME = 4;

    private static final String
        IP_ADDRESS   = "ipAddr";

    private static final Map<String, Integer> mTypeMap = new HashMap<>();
    static {
        mTypeMap.put("VOLTAGE", MON_TYPE_VOLTAGE);
        mTypeMap.put("CURRENT", MON_TYPE_CURRENT);
        mTypeMap.put("POWER", MON_TYPE_POWER);
        mTypeMap.put("USAGE", MON_TYPE_USAGE);
        mTypeMap.put("LIFETIME", MON_TYPE_LIFETIME);
    }

    //private static final int PERSIST_INTERVAL = 10000;
    private static final int CHECK_INTERVAL = 10000;
    private static final double MAX_USAGE = 0.0375;  // Torr hours

    /**
     *  Data fields.
     */
    @LookupField(strategy = LookupField.Strategy.TOP)
    private Subsystem subsys;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private TrendingClientService trendingService;
    @LookupField (strategy = LookupField.Strategy.CHILDREN)
    private final Map<String, IonPumpControl> ctrlChans = new LinkedHashMap<>();

    @ConfigurationParameter(name=IP_ADDRESS, category=VacuumConfig.CRYO, isFinal=true)
    private volatile String ipAddr;

    private static final Logger LOG = Logger.getLogger(IonPumpDevice.class.getName());
    private final IonPump ipc = new IonPump();
    private TrendingConnectionUtils trendingUtils;
    private boolean trendingError = false, usageInited = false;
    private final Map<String, Integer> usageChannelMap = new HashMap<>();
    private final double[] savedUsage = new double[IonPump.NUM_CHANNELS];
    private final double[] accumUsage = new double[IonPump.NUM_CHANNELS];
    private final long[] updateTime = new long[IonPump.NUM_CHANNELS];
    private Channel cryoPressure, hxPressure;
    private long checkTime = 0;
    private boolean initError = false;


    /**
     *  Constructor.
     */
    public IonPumpDevice()
    {
        super();
        System.setProperty("org.lsst.ccs.agent." + TrendingClientService.USES_TRENDING_SERVICE, "true");
        Arrays.fill(savedUsage, Double.NaN);
        Arrays.fill(accumUsage, 0.0);
    }


    /**
     *  Post start phase
     */
    @Override
    public void postStart() {
        StringBuilder missingChans = new StringBuilder();
        for (int chan = 0; chan < IonPump.NUM_CHANNELS; chan++) {
            if (!usageChannelMap.values().contains(chan)) {
                missingChans.append(" ").append(chan);
            }
        }
        if (missingChans.length() != 0) {
            LOG.log(Level.INFO, "Ion pump {0} has no usage monitoring item for the following channels:{1}",
                    new Object[]{name, missingChans});
        }
        initSavedTime();
    }


    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice() {
        if (ipAddr == null) {
            ErrorUtils.reportConfigError(LOG, name, IP_ADDRESS, "is missing");
        }
        fullName = "Ion pump controller (" + ipAddr + ")";
    }


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        try {
            ipc.open(ipAddr);
            for (IonPumpControl ctrl : ctrlChans.values()) {
                ctrl.writeVoltage();
                ctrl.writeCurrent();
                ctrl.writePower();
            }
            setOnline(true);
            initSensors();
            LOG.log(Level.INFO, "Connected to {0}", fullName);
            initError = false;
        }
        catch (DriverException | VacuumException e) {
            if (!initError) {
                LOG.log(Level.SEVERE, "Error connecting to {0}: {1}", new Object[]{fullName, e});
                initError = true;
            }
            try {
                ipc.close();
            }
            catch (DriverException ce) {
                // Here if device wasn't opened
            }
        }
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        try {
            ipc.close();
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error disconnecting from {0}: {1}", new Object[]{fullName, e});
        }
    }


    /**
     *  Checks a monitoring channel's parameters for validity.
     *
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string
     *  @return  Two-element array containing the encoded type [0] and subtype [1] 
     *  @throws  Exception  If parameters are invalid
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        Integer mType = mTypeMap.get(type.toUpperCase());
        if (mType == null) {
            ErrorUtils.reportChannelError(LOG, name, "type", type);
        }
        if (hwChan < 0 || hwChan >= IonPump.NUM_CHANNELS) {
            ErrorUtils.reportChannelError(LOG, name, "hw channel number", hwChan);
        }
        if (mType == MON_TYPE_USAGE) {
            usageChannelMap.put(mon.getChannel(name).getPath(), hwChan);
        }
        return new int[]{mType, 0};
    }


    /**
     *  Reads a monitoring channel.
     *
     *  @param  hwChan  The hardware channel number
     *  @param  type    The encoded channel type
     *  @return  The read value
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = super.readChannel(hwChan, type);
        String item = null;
        if (isOnline()) {
            try {
                switch (type) {
                case MON_TYPE_VOLTAGE:
                    item = "voltage";
                    value = ipc.readVoltage(hwChan);
                    break;

                case MON_TYPE_CURRENT:
                    item = "current";
                    value = ipc.readCurrent(hwChan);
                    break;

                case MON_TYPE_POWER:
                    item = "power";
                    value = ipc.readPower(hwChan);
                    break;

                case MON_TYPE_USAGE:
                    long time = System.currentTimeMillis();
                    if (time - checkTime >= CHECK_INTERVAL) {
                        initSavedTime();
                        checkTime = time;
                    }
                    synchronized (usageChannelMap) {
                        if (updateTime[hwChan] != 0) {
                            if (isChannelOn(hwChan) == Boolean.TRUE) {
                                double pressure = (hwChan == CHAN_HIP1 || hwChan == CHAN_HIP2) ? hxPressure.getValue()
                                                                                               : cryoPressure.getValue();
                                if (!Double.isNaN(pressure)) {
                                    accumUsage[hwChan] += (time - updateTime[hwChan]) * pressure / (1000 * 3600);  // Torr hours
                                }
                            }
                        }
                        updateTime[hwChan] = time;
                        value = accumUsage[hwChan] + savedUsage[hwChan];  // NaN if savedUsage is NaN
                    }
                    break;

                case MON_TYPE_LIFETIME:
                    if (isChannelOn(hwChan) == Boolean.TRUE) {
                        value = (MAX_USAGE - savedUsage[hwChan] - accumUsage[hwChan])
                                  / ((hwChan == CHAN_HIP1 || hwChan == CHAN_HIP2) ? hxPressure.getValue() : cryoPressure.getValue());
                    }
                    break;
                }
            }
            catch (DriverException e) {
                LOG.log(Level.SEVERE, "Error reading {0} {1}: {2}", new Object[]{fullName, item, e});
                setOnline(false);
            }
        }
        return value;
    }


    /**
     *  Initializes the saved usage times from the trending database
     */
    private void initSavedTime()
    {
        if (usageInited) return;
        trendingUtils = trendingService.getBusTrendingConnection();
        if (trendingUtils == null) {
            if (!trendingError) {
                LOG.log(Level.SEVERE, "Trending data cannot be accessed");
                trendingError = true;
            }
            return;
        }
        if (trendingError) {
            LOG.log(Level.INFO, "Trending data can now be accessed");
            trendingError = false;
        }
        synchronized(usageChannelMap) {
            for (Map.Entry e : usageChannelMap.entrySet()) {
                if (Double.isNaN(savedUsage[(Integer)e.getValue()])) {
                    String chName = subsys.getName() + "/" + (String)e.getKey();
                    TrendingData data = trendingUtils.getLatestData(chName);
                    double value = data == null ? Double.NaN : data.getValue("value");
                    if (Double.isNaN(value)) {
                        LOG.log(Level.WARNING, "No trending data present for ion pump channel {0}", chName);
                        value = 0.0;
                    }
                    else {
                        LOG.log(Level.INFO, "Got trending data for ion pump channel {0}", chName);
                    }
                    savedUsage[(Integer)e.getValue()] = value;
                }
            }
        }
        usageInited = true;
    }


    /**
     *  Command to get the set of usage monitoring channel names.
     * 
     *  @return The set of names
     */
    @Command(type = Command.CommandType.QUERY, level = 0, description = "Get the set of usage channel names")
    public Set<String> getUsageChannelNames()
    {
        return new HashSet(usageChannelMap.keySet());
    }


    /**
     *  Command to set the value for a usage monitoring channel.
     * 
     *  @param  chanName  The channel name
     *  @param  usage     The usage value
     *  @throws  VacuumException
     */
    @Command(description = "Set the value for a usage monitoring channel", level = 50)
    public void setUsageValue(@Argument(description = "Channel name") String chanName,
                              @Argument(description = "Usage value") double usage) throws VacuumException
    {
        Integer hwChan = usageChannelMap.get(chanName);
        if (hwChan == null) {
            throw new VacuumException("Invalid usage channel name: " + chanName);
        }
        synchronized(usageChannelMap) {
            savedUsage[hwChan] = usage - accumUsage[hwChan];
        }
    }


    /**
     *  Sets the cryo pressure channel.
     *
     *  @param  pressure  The channel
     */
    void setCryoPressureChannel(Channel pressure)
    {
        cryoPressure = pressure;
    }


    /**
     *  Sets the HX pressure channel.
     *
     *  @param  pressure  The channel
     */
    void setHxPressureChannel(Channel pressure)
    {
        hxPressure = pressure;
    }


    /**
     *  Turns a channel on or off
     *
     *  @param  chan  The hardware channel number
     *  @param  on    Whether to set on or off
     */
    protected void setChannelOn(int chan, boolean on)
    {
        try {
            if (on) {
                ipc.powerOn(chan);
            }
            else {
                ipc.powerOff(chan);
            }
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error setting channel {0} on {1}", new Object[]{chan, fullName});
        }
        
    }


    /**
     *  Gets the on state of a channel.
     *
     *  @param  chan  The hardware channel number
     *  @return  Whether the line is on or off
     */
    protected Boolean isChannelOn(int chan)
    {
        if (!isOnline()) return null;
        try {
            return ipc.isPowered(chan);
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error getting state for channel {0} on {1}", new Object[]{chan, fullName});
            return null;
        }
    }


    /**
     *  Gets the list of defined channel names.
     *
     *  @return  The channel names
     */
    public List<String> getChannelNames()
    {
        return new ArrayList<>(ctrlChans.keySet());
    }


    /**
     *  Gets the list of defined channel numbers.
     *
     *  @return  The channel numbers
     */
    public List<Integer> getChannelNumbers()
    {
        List<Integer> chans = new ArrayList<>();
        for (IonPumpControl ctrl : ctrlChans.values()) {
            chans.add(ctrl.getHwChan());
        }
        return chans;
    }


    /**
     *  Sets the voltage for a channel.
     *
     *  @param  chan   The channel number
     *  @param  value  The voltage
     *  @throws  VacuumException
     */
    void setVoltage(int chan, double value) throws VacuumException
    {
        try {
            ipc.setVoltage(chan, value);
        }
        catch (DriverException e) {
            throw new VacuumException(e);
        }
    }


    /**
     *  Sets the current for a channel.
     *
     *  @param  chan   The channel number
     *  @param  value  The current
     *  @throws  VacuumException
     */
    void setCurrent(int chan, double value) throws VacuumException
    {
        try {
            ipc.setCurrent(chan, value);
        }
        catch (DriverException e) {
            throw new VacuumException(e);
        }
    }


    /**
     *  Sets the power for a channel.
     *
     *  @param  chan   The channel number
     *  @param  value  The power
     *  @throws  VacuumException
     */
    void setPower(int chan, double value) throws VacuumException
    {
        try {
            ipc.setPower(chan, value);
        }
        catch (DriverException e) {
            throw new VacuumException(e);
        }
    }


    /**
     *  Gets the switch device ID.
     * 
     *  @return  The device ID
     */
    @Override
    public int getSwitchDevice()
    {
        return Devices.DEVC_ION_PUMP;
    }


    /**
     *  Turns a switch on or off.
     * 
     *  @param  sw  The switch ID
     *  @param  on  Whether to turn on or off
     */
    @Override
    public void setSwitch(int sw, boolean on)
    {
        setChannelOn(sw, on);
    }


    /**
     *  Gets whether a switch is on.
     * 
     *  @param  sw  The switch ID
     *  @return  Whether on, or null if device is offline
     */
    @Override
    public Boolean isSwitchOn(int sw)
    {
        return isChannelOn(sw);
    }


    /**
     *  Powers on a channel.
     *
     *  @param  chan  The channel number
     *  @throws  VacuumException
     */
    void powerOn(int chan) throws VacuumException
    {
        try {
            ipc.powerOn(chan);
        }
        catch (DriverException e) {
            throw new VacuumException(e);
        }
    }


    /**
     *  Powers off a channel.
     *
     *  @param  chan  The channel number
     *  @throws  VacuumException
     */
    void powerOff(int chan) throws VacuumException
    {
        try {
            ipc.powerOff(chan);
        }
        catch (DriverException e) {
            throw new VacuumException(e);
        }
    }


    /**
     *  Gets ion pump usage.
     *
     *  @return  The array of usage values, one per channel
     */
    double[] getUsage()
    {
        double[] usage = new double[IonPump.NUM_CHANNELS];
        for (int j = 0; j < IonPump.NUM_CHANNELS; j++) {
            usage[j] = savedUsage[j] + accumUsage[j];
        }
        return usage;
    }


    /**
     *  Gets named controller.
     *
     *  @param  name  The channel name
     *  @throws  VacuumException
     */
    private IonPumpControl getControl(String name) throws VacuumException
    {
        IonPumpControl ctrl = ctrlChans.get(name);
        if (ctrl == null) {
            throw new VacuumException("Unrecognized ion pump name: " + name);
        }
        return ctrl;
    }

}
