package org.lsst.ccs.subsystem.vacuum;

import java.util.ArrayList;
import java.time.Duration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.alert.AlertEvent;
import org.lsst.ccs.services.alert.AlertListener;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.common.MonitorTaskControl;
import org.lsst.ccs.subsystem.vacuum.constants.VacuumAlert;
import org.lsst.ccs.subsystem.vacuum.constants.ConditionState;
import org.lsst.ccs.subsystem.vacuum.constants.Conditions;
import org.lsst.ccs.subsystem.vacuum.constants.DeviceState;
import org.lsst.ccs.subsystem.vacuum.constants.Devices;
import org.lsst.ccs.subsystem.vacuum.constants.LatchState;
import org.lsst.ccs.subsystem.vacuum.constants.Latches;
import org.lsst.ccs.subsystem.vacuum.constants.MonitorControl;
import org.lsst.ccs.subsystem.vacuum.constants.PLCState;
import org.lsst.ccs.subsystem.vacuum.constants.SwitchEnable;
import org.lsst.ccs.subsystem.vacuum.constants.Switches;
import org.lsst.ccs.subsystem.vacuum.constants.SwitchState;
import org.lsst.ccs.subsystem.vacuum.constants.VacuumAgentProperties;
import org.lsst.ccs.subsystem.vacuum.constants.VacuumState;
import org.lsst.ccs.subsystem.vacuum.data.VacuumException;
import org.lsst.ccs.subsystem.vacuum.data.VacSysState;

/**
 * The camera vacuum system
 *
 * @author The LSST CCS Team
 */
public class VacuumMain extends Subsystem implements HasLifecycle, ClearAlertHandler, AlertListener {

    private static final double
        PRESS_ATMOS = 759.0,
        PRESS_TURBO_LOW = 5.0,
        PRESS_FORELINE_LOW = 5.0,
        PRESS_DIFF_LOW = 0.09,
        PRESS_DIFF_HIGH = 20.0,
        PRESS_ION_OFF = 1.1e-5,
        PRESS_ION_ENABLE = 1.0e-6,
        PRESS_VACUUM = 1.0e-7,
        TURBO_MAX = 60000,
        TURBO_LOW = 0.1 * TURBO_MAX;
    private static final int
        MAX_PRESS_ERRORS = 3,
        MAX_SPEED_ERRORS = 0,
        DELAY_ION_OFF = 10000;
    private static final Map<Integer, VacuumAlert> alertMap = new HashMap<>();
    static {
        alertMap.put(Latches.LATCH_CR_VACUUM, VacuumAlert.CRYO_VACUUM_BAD);
        alertMap.put(Latches.LATCH_CR_GATE_NFC, VacuumAlert.CRYO_GATE_FORCED_SHUT);
        alertMap.put(Latches.LATCH_CR_GATE_AO, VacuumAlert.CRYO_GATE_CANNOT_OPEN);
        alertMap.put(Latches.LATCH_CR_PUMP, VacuumAlert.CRYO_TURBO_PUMP_BAD);
        alertMap.put(Latches.LATCH_HX_VACUUM, VacuumAlert.HX_VACUUM_BAD);
        alertMap.put(Latches.LATCH_HX_GATE_NFC, VacuumAlert.HX_GATE_FORCED_SHUT);
        alertMap.put(Latches.LATCH_HX_GATE_AO, VacuumAlert.HX_GATE_CANNOT_OPEN);
        alertMap.put(Latches.LATCH_HX_PUMP, VacuumAlert.HX_TURBO_PUMP_BAD);
    }
    private static final Map<String, Integer> revAlertMap = new HashMap<>();
    static {
        for (int cond : alertMap.keySet()) {
            revAlertMap.put(alertMap.get(cond).getId(), cond);
        }
    }

    @LookupName
    private String name;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;
    //@LookupField(strategy = LookupField.Strategy.TREE)
    //private AgentStateService stateService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService propertiesService;

    @LookupField(strategy=LookupField.Strategy.DESCENDANTS)
    private final List<SwitchDevice> switches = new ArrayList<>();

    // From Groovy file
    private Channel cryoVacPressure, cryoTurboPressure, cryoFlinePressure, cryoTurboSpeed,
                    hxVacPressure, hxTurboPressure, hxFlinePressure, hxTurboSpeed, orVacPressure;

    // General
    private static final Logger LOG = Logger.getLogger(VacuumMain.class.getName());
    private MonitorTaskControl monitorControl;
    private final SwitchDevice[] switchDevices = new SwitchDevice[Devices.NUM_DEVICES];
    private VacPlutoDevice plutoDevc;
    private IonPumpDevice ionPumpDevc;
    private CryoTurboDevice cryoTurboDevc;
    private HxTurboDevice hxTurboDevc;
    private final VacSysState vacState = new VacSysState();
    private boolean running = false;
    private final Map<String, Boolean> activeAlertMap = new HashMap<>();
    private long ionOverStartTime = 0;
    private double forelinePress, turboPress, cryoPress, turboSpd;
    private int forelinePrErrs, turboPrErrs, cryoPrErrs, turboSpErrs;


    /**
     *  Constructor
     */
    public VacuumMain() {
        super("vacuum", AgentInfo.AgentType.WORKER);
        getAgentInfo().getAgentProperties().setProperty("org.lsst.ccs.use.full.paths", "true");
    }


    /**
     *  Build phase
     */
    @Override
    public void build() {
        // Create the monitor task control object and node
        monitorControl = MonitorTaskControl.createNode(this, MonitorControl.NODE_NAME);

        // Create and schedule an AgentPeriodicTask to update the vacuum state
        AgentPeriodicTask pt;
        pt = new AgentPeriodicTask("vacuum-state", () -> updateVacuumState()).withPeriod(Duration.ofMillis(1000));
        periodicTaskService.scheduleAgentPeriodicTask(pt);

        // Register vacuum state
        stateService.registerState(VacuumState.class, "Vacuum state", this);
        stateService.updateAgentState(VacuumState.UNKNOWN);

        // Initialize active alert map
        for (VacuumAlert alert : VacuumAlert.values()) {
            activeAlertMap.put(alert.getId(), false);
        }
    }


    /**
     *  Post initialization phase
     */
    @Override
    public void postInit() {

        //Set a property to define that this Agent is a vacuum subsystem.
        propertiesService.setAgentProperty(VacuumAgentProperties.VACUUM_TYPE, VacuumMain.class.getCanonicalName());

        // Add alert listener
        alertService.addListener(this);

        for (SwitchDevice device : switches) {
            switchDevices[device.getSwitchDevice()] = device;
        }
        for (int id = 0; id < Devices.NUM_DEVICES; id++) {
            if (switchDevices[id] == null) {
                ErrorUtils.reportConfigError(LOG, name, Devices.getDescription(id) + " device", "not defined");
            }
        }
        plutoDevc = (VacPlutoDevice)switchDevices[Devices.DEVC_PLUTO];
        ionPumpDevc = (IonPumpDevice)switchDevices[Devices.DEVC_ION_PUMP];
        cryoTurboDevc = (CryoTurboDevice)switchDevices[Devices.DEVC_CRYO_TURBO_PUMP];
        hxTurboDevc = (HxTurboDevice)switchDevices[Devices.DEVC_HX_TURBO_PUMP];

        if (cryoFlinePressure == null) {
            ErrorUtils.reportConfigError(LOG, name, "Cryo foreline pressure channel", "not specified");
        }
        if (cryoVacPressure == null) {
            ErrorUtils.reportConfigError(LOG, name, "Cryo vacuum pressure channel", "not specified");
        }
        if (cryoTurboPressure == null) {
            ErrorUtils.reportConfigError(LOG, name, "Cryo turbo pump pressure channel", "not specified");
        }
        if (cryoTurboSpeed == null) {
            ErrorUtils.reportConfigError(LOG, name, "Cryo turbo pump speed channel", "not specified");
        }
        if (hxFlinePressure == null) {
            ErrorUtils.reportConfigError(LOG, name, "HX foreline pressure channel", "not specified");
        }
        if (hxVacPressure == null) {
            ErrorUtils.reportConfigError(LOG, name, "HX vacuum pressure channel", "not specified");
        }
        if (hxTurboPressure == null) {
            ErrorUtils.reportConfigError(LOG, name, "HX turbo pump pressure channel", "not specified");
        }
        if (hxTurboSpeed == null) {
            ErrorUtils.reportConfigError(LOG, name, "HX turbo pump speed channel", "not specified");
        }
        if (orVacPressure == null) {
            ErrorUtils.reportConfigError(LOG, name, "O-ring vacuum pressure channel", "not specified");
        }

        ionPumpDevc.setCryoPressureChannel(cryoVacPressure);
        ionPumpDevc.setHxPressureChannel(hxVacPressure);
    }


    /**
     *  Post start
     */
    @Override
    public void postStart() {
        LOG.info("Vacuum subsystem started");
        running = true;
    }


    /**
     *  Gets the state of the Vacuum system.
     *
     *  @return  The vacuum state
     */
    @Command(type=Command.CommandType.QUERY, description="Get the vacuum system state", level=0)
    public VacSysState getVacuumState() {
        vacState.setTickMillis(monitorControl.getPublishPeriod());
        return vacState;
    }    


    /**
     *  Gets the list of switch names.
     *
     *  @return  The switch names.
     *  @throws  VacuumException
     */
    @Command(type=Command.CommandType.QUERY, description="Get switch names", level=0)
    public List<String> getSwitchNames() throws VacuumException
    {
        return Switches.getNames();
    }


    /**
     *  Turns a switch on or off.
     *
     *  @param  name  The switch name.
     *  @param  on    Whether to turn on or off
     *  @throws  VacuumException
     */
    @Command(type=Command.CommandType.ACTION, description="Turn on/off a named switch")
    public void setSwitchOn(@Argument(description="The switch name") String name,
                            @Argument(description="Whether to turn on") boolean on) throws VacuumException
    {
        try {
            setSwitch(Switches.getId(name), on);
        }
        finally {
            publishState();
        }
    }


    /**
     *  Turns a switch on or off.
     *
     *  @param  sw  The switch number.
     *  @param  on  Whether to turn on or off
     *  @throws  VacuumException
     */
    private void setSwitch(int sw, boolean on) throws VacuumException
    {
        SwitchState state = vacState.getSwitchState(sw);
        if (state == SwitchState.OFFLINE) return;
        SwitchEnable enable = vacState.getSwitchEnable(sw);
        if (on && enable != SwitchEnable.ON) return;
        try {
            switchDevices[Switches.getDevice(sw)].setSwitch(SwitchInfo.getSwitch(sw), on);
        }
        catch (DriverException e) {
            throw new VacuumException("Error setting switch for " + Devices.getDescription(Switches.getDevice(sw)) + " device: " + e);
        }
    }


    /**
     *  Gets whether a switch is on.
     *
     *  @param  sw  The switch number.
     *  @return  Whether the switch is on
     *  @throws  VacuumException
     */
    private Boolean isSwitchOn(int sw)
    {
        return switchDevices[Switches.getDevice(sw)].isSwitchOn(SwitchInfo.getSwitch(sw));
    }


    /**
     *  Clears a latched condition.
     *
     *  @param  cond  The condition number.
     *  @throws  VacuumException
     */
    @Command(type=Command.CommandType.ACTION, description="Clear a condition")
    public void clearLatch(@Argument(description="The condition number") int cond) throws VacuumException
    {
        try {
            plutoDevc.clearLatch(cond);
        }
        finally {
            publishState();
        }
    }


    /**
     *  Updates the vacuum system state periodically.
     *
     *  The vacuum state consists mainly of the state of the switches (lines) being
     *  controlled, along with whether they can be turned on.
     *
     *  Pressures and pump speeds are read to determine which switches are enabled
     *  and/or need to be turned off.  If the state changes it is published.
     */
    private void updateVacuumState()
    {
        if (!running) return;

        Boolean plcActive = plutoDevc.isPLCActive();
        if (plcActive == null) {
            if (vacState.getPlcState() != PLCState.OFFLINE) {
                raiseAlert(VacuumAlert.VACUUM_PLC_NOT_ALIVE, "Vacuum PLC is offline");
                vacState.setPlcState(PLCState.OFFLINE);
            }
        }
        else if (!plcActive) {
            if (vacState.getPlcState() != PLCState.DEAD) {
                raiseAlert(VacuumAlert.VACUUM_PLC_NOT_ALIVE, "Vacuum PLC has died");
                vacState.setPlcState(PLCState.DEAD);
            }
        }
        else {
            if (vacState.getPlcState() != PLCState.ALIVE) {
                lowerAlert(VacuumAlert.VACUUM_PLC_NOT_ALIVE, "Vacuum PLC is alive");
                vacState.setPlcState(PLCState.ALIVE);
            }
        }

        double forelinePr = cryoFlinePressure.getValue();
        if (!Double.isNaN(forelinePr)) {
            forelinePress = forelinePr;
            forelinePrErrs = 0;
        }
        else if (forelinePrErrs++ < MAX_PRESS_ERRORS) {
            forelinePr = forelinePress;
        }
        double cryoPr = cryoVacPressure.getValue();
        if (!Double.isNaN(cryoPr)) {
            cryoPress = cryoPr;
            cryoPrErrs = 0;
        }
        else if (cryoPrErrs++ < MAX_PRESS_ERRORS) {
            cryoPr = cryoPress;
        }
        double turboPr = cryoTurboPressure.getValue();
        if (!Double.isNaN(turboPr)) {
            turboPress = turboPr;
            turboPrErrs = 0;
        }
        else if (turboPrErrs++ < MAX_PRESS_ERRORS) {
            turboPr = turboPress;
        }
        double turboSp = cryoTurboSpeed.getValue();
        if (!Double.isNaN(turboSp)) {
            turboSpd = turboSp;
            turboSpErrs = 0;
        }
        else if (turboSpErrs++ < MAX_SPEED_ERRORS) {
            turboSp = turboSpd;
        }
        boolean changed = monitorControl.hasPeriodChanged();

        for (int sw = 0; sw < Switches.NUM_SWITCHES; sw++) {
            Boolean enable = false;
            boolean turnOff = false;
            DeviceState devState = null;
            VacuumAlert alert = null;
            switch (sw) {
 
            case Switches.SW_CRYO_GATE_VALVE:
                alert = VacuumAlert.CRYO_GATE_CLOSED;
                boolean cryoTurbAvail = !Double.isNaN(cryoPr) && !Double.isNaN(turboPr) && !Double.isNaN(turboSp);
                if (cryoTurbAvail) {
                    double prDiff = Math.abs(turboPr - cryoPr);
                    enable = (prDiff <= PRESS_DIFF_HIGH && turboSp <= TURBO_LOW)
                                || (prDiff <= PRESS_DIFF_LOW && turboSp > TURBO_LOW);
                }
                boolean enable2 = cryoTurbAvail && !Double.isNaN(forelinePr)
                                    && (forelinePr < PRESS_FORELINE_LOW || turboSp <= TURBO_LOW);
                enable &= enable2;
                turnOff = !enable2;
                break;

            case Switches.SW_CRYO_TURBO_PUMP:
                alert = VacuumAlert.CRYO_TURBO_PUMP_STOPPED;
                if (!Double.isNaN(turboPr) && !Double.isNaN(forelinePr) && !Double.isNaN(turboSp)) {
                    enable = turboPr < PRESS_TURBO_LOW && (forelinePr < PRESS_FORELINE_LOW || turboSp < TURBO_LOW);
                }
                turnOff = !enable;
                devState = cryoTurboDevc.getDeviceState();
                break;

            case Switches.SW_CRYO_ION_PUMP1:
            case Switches.SW_CRYO_ION_PUMP2:
            case Switches.SW_CRYO_ION_PUMP3:
            case Switches.SW_CRYO_ION_PUMP4:
            case Switches.SW_CRYO_ION_PUMP5:
            case Switches.SW_CRYO_ION_PUMP6:
                alert = VacuumAlert.CRYO_ION_PUMPS_STOPPED;
                if (!Double.isNaN(cryoPr)) {
                    enable = cryoPr < PRESS_ION_ENABLE;
                    if (cryoPr >= PRESS_ION_OFF) {
                        if (ionOverStartTime == 0) {
                            ionOverStartTime = System.currentTimeMillis();
                        }
                        else {
                            turnOff = System.currentTimeMillis() - ionOverStartTime >= DELAY_ION_OFF;
                        }
                    }
                    else {
                        ionOverStartTime = 0;
                    }
                }
                else {
                    turnOff = true;
                }
                break;

            case Switches.SW_CRYO_SCROLL_PUMP:
            case Switches.SW_HX_SCROLL_PUMP:
            case Switches.SW_INST_SCROLL_PUMP:
                enable = true;
                break;

            case Switches.SW_INST_FTH_VALVE:
            case Switches.SW_INST_FTPP_VALVE:
            case Switches.SW_INST_L3H_VALVE:
            case Switches.SW_INST_L3LF_VALVE:
                enable = true;
                break;
            }

            SwitchState oldState = vacState.getSwitchState(sw);
            if (turnOff && oldState == SwitchState.ON) {
                try {
                    setSwitch(sw, false);
                    raiseAlert(alert, "Switch was forced to open");
                }
                catch (VacuumException e) {
                    LOG.log(Level.SEVERE, "Error setting switch: {0}", e);
                }
            }
            Boolean isOn = isSwitchOn(sw);
            SwitchState state = isOn != null ? isOn ? SwitchState.ON : SwitchState.OFF : SwitchState.OFFLINE;
            SwitchEnable enabled = enable ? SwitchEnable.ON : SwitchEnable.OFF;
            if (state != oldState || enabled != vacState.getSwitchEnable(sw)) {
                vacState.setSwitchState(sw, state);
                vacState.setSwitchEnable(sw, enabled);
                changed = true;
                if (enable && alert != null && isAlertRaised(alert)) {
                    lowerAlert(alert, "Switch has become re-enabled");
                }
            }
            if (devState != vacState.getDeviceState(sw)) {
                vacState.setDeviceState(sw, devState);
                changed = true;
            }
        }

        for (int cond = 0; cond < Latches.NUM_LATCHES; cond++) {
            Boolean active = plutoDevc.isLatchActive(cond);
            Boolean latched = plutoDevc.isLatchLatched(cond);
            LatchState state = active == null || latched == null ? LatchState.OFFLINE :
                               latched ? LatchState.LATCHED :
                               active ? LatchState.ACTIVE : LatchState.CLEAR;
            LatchState oldState = vacState.getLatch(cond); 
            if (state != oldState) {
                vacState.setLatch(cond, state);
                VacuumAlert alert = alertMap.get(cond);
                if (state == LatchState.ACTIVE) {
                    raiseAlert(alert, "Vacuum PLC error condition set");
                }
                else if (state != LatchState.OFFLINE && oldState != LatchState.OFFLINE) {
                    if (state == LatchState.LATCHED && oldState == LatchState.CLEAR) {
                        raiseAlert(alert, "Vacuum PLC error condition set");
                    }
                    if (state == LatchState.LATCHED || oldState == LatchState.ACTIVE) {
                        lowerAlert(alert, "Vacuum PLC error condition cleared");
                    }
                }
                changed = true;
            }
        }

        for (int cond = 0; cond < Conditions.NUM_CONDITIONS; cond++) {
            Boolean active = plutoDevc.isConditionActive(cond);
            ConditionState state = active == null ? ConditionState.OFF :
                                   active ? ConditionState.YES : ConditionState.NO;
            if (state != vacState.getCondition(cond)) {
                vacState.setCondition(cond, state);
                changed = true;
            }
        }

        VacuumState vState;
        if (Double.isNaN(cryoPr)) {
            vState = VacuumState.UNKNOWN;
        }
        else if (cryoPr <= PRESS_VACUUM) {
            vState = VacuumState.VACUUM;
        }
        else if (vacState.getSwitchState(Switches.SW_CRYO_ION_PUMP1) == SwitchState.ON) {
            vState = VacuumState.ION_ON;
        }
        else if (vacState.getSwitchEnable(Switches.SW_CRYO_ION_PUMP1) == SwitchEnable.ON) {
            vState = VacuumState.ION_OFF;
        }
        else if (vacState.getSwitchState(Switches.SW_CRYO_TURBO_PUMP) == SwitchState.ON) {
            vState = VacuumState.TURBO_ON;
        }
        else if (vacState.getSwitchEnable(Switches.SW_CRYO_TURBO_PUMP) == SwitchEnable.ON) {
            vState = VacuumState.TURBO_OFF;
        }
        else if (cryoPr < PRESS_ATMOS) {
            vState = VacuumState.FORELINE;
        }
        else {
            vState = VacuumState.OFF;
        }
        if (vState != vacState.getVacuumState()) {
            vacState.setVacuumState(vState);
            stateService.updateAgentState(vState);
            changed = true;
        }

        if (changed) {
            publishState();
        }
    }


    /**
     *  Raises an alert.
     *
     *  @param  alert  The vacuum alert to raise
     *  @param  cond   The alert condition
     */
    private void raiseAlert(VacuumAlert alert, String cond)
    {
        alertService.raiseAlert(alert.getAlert(), AlertState.ALARM, cond);
        activeAlertMap.put(alert.getId(), true);
    }


     /**
     *  Checks whether an alert has been raised.
     *
     *  @param  alert  The vacuum alert to check
     *  @return  Whether it has been raised
     */
    private boolean isAlertRaised(VacuumAlert alert)
    {
        return activeAlertMap.get(alert.getId()).equals(Boolean.TRUE);
    }


    /**
     *  Lowers an alert.
     *
     *  @param  alert  The vacuum alert to lower
     *  @param  cond   The alert condition
     */
    private void lowerAlert(VacuumAlert alert, String cond)
    {
        alertService.raiseAlert(alert.getAlert(), AlertState.NOMINAL, cond);
        activeAlertMap.put(alert.getId(), false);
    }


    /**
     * Callback to clear an {@code Alert} instance.
     * 
     * @param alert The Alert instance to clear.
     * @param alertState The AlertState for the provided Alert.
     * @return A ClearAlertCode to indicate which action is to be taken
     *         by the framework.
     */
    @Override
    public ClearAlertCode canClearAlert(Alert alert, AlertState alertState)
    {
        Boolean active = activeAlertMap.get(alert.getAlertId());
        return active == null ? ClearAlertCode.UNKNOWN_ALERT
                              : active ? ClearAlertCode.DONT_CLEAR_ALERT : ClearAlertCode.CLEAR_ALERT;
    }


    /**
     *  Alert event handler.
     *
     *  Resets PLC latch when corresponding alert is cleared.
     *
     *  @param  event  The alert event
     */
    @Override
    public void onAlert(AlertEvent event)
    {
        if (event.getType() != AlertEvent.AlertEventType.ALERT_CLEARED) return;
        for (String id : event.getClearedIds()) {
            Integer cond = revAlertMap.get(id);
            if (cond != null) {
                try {
                    clearLatch(cond);
                }
                catch (VacuumException e) {
                    LOG.log(Level.SEVERE, "Error clearing latched PLC condition ({0}): {1}", new Object[]{cond, e});
                }
            }
        }
    }


    /**
     *  Publishes the state of the vacuum system.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    private void publishState()
    {
        publishSubsystemDataOnStatusBus(new KeyValueData(VacSysState.KEY, getVacuumState()));
    }

}
