package org.lsst.ccs.subsystem.vacuum;

import java.util.ArrayList;
import java.time.Duration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.twistorr.TwisTorr84.PumpStatus;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.alert.AlertEvent;
import org.lsst.ccs.services.alert.AlertListener;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.devices.turbopump.TwisTorr84Device;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.vacuum.constants.VacuumAlert;
import org.lsst.ccs.subsystem.vacuum.constants.ConditionState;
import org.lsst.ccs.subsystem.vacuum.constants.DeviceState;
import org.lsst.ccs.subsystem.vacuum.constants.LatchState;
import org.lsst.ccs.subsystem.vacuum.constants.PLCState;
import org.lsst.ccs.subsystem.vacuum.constants.SwitchEnable;
import org.lsst.ccs.subsystem.vacuum.constants.SwitchNames;
import org.lsst.ccs.subsystem.vacuum.constants.SwitchState;
import org.lsst.ccs.subsystem.vacuum.constants.VacuumAgentProperties;
import org.lsst.ccs.subsystem.vacuum.constants.VacuumState;
import org.lsst.ccs.subsystem.vacuum.data.VacuumException;
import org.lsst.ccs.subsystem.vacuum.data.VacSysState;

/**
 * The cryogenics vacuum system
 *
 * @author The LSST CCS Team
 */
public class VacuumMain extends Subsystem implements HasLifecycle, ClearAlertHandler, AlertListener {

    private static final double
        PRESS_ATMOS = 759.0,
        PRESS_TURBO_LOW = 5.0,
        PRESS_FORELINE_LOW = 5.0,
        PRESS_DIFF_LOW = 0.09,
        PRESS_DIFF_HIGH = 20.0,
        PRESS_ION_OFF = 1.1e-5,
        PRESS_ION_ENABLE = 1.0e-6,
        PRESS_VACUUM = 1.0e-7,
        TURBO_MAX = 60000,
        TURBO_LOW = 0.1 * TURBO_MAX;
    private static final int
        MAX_PRESS_ERRORS = 3,
        MAX_SPEED_ERRORS = 0,
        DELAY_ION_OFF = 10000;
    private static final int[] switchChannels = new int[VacSysState.NUM_SWITCHES];
    static {
        switchChannels[VacSysState.SW_CRYO_ION_PUMP1] = IonPumpDevice.CHAN_CIP1;
        switchChannels[VacSysState.SW_CRYO_ION_PUMP2] = IonPumpDevice.CHAN_CIP2;
        switchChannels[VacSysState.SW_CRYO_ION_PUMP3] = IonPumpDevice.CHAN_CIP3;
        switchChannels[VacSysState.SW_CRYO_ION_PUMP4] = IonPumpDevice.CHAN_CIP4;
        switchChannels[VacSysState.SW_CRYO_ION_PUMP5] = IonPumpDevice.CHAN_CIP5;
        switchChannels[VacSysState.SW_CRYO_ION_PUMP6] = IonPumpDevice.CHAN_CIP6;
        switchChannels[VacSysState.SW_HX_ION_PUMP1] = IonPumpDevice.CHAN_HIP1;
        switchChannels[VacSysState.SW_HX_ION_PUMP2] = IonPumpDevice.CHAN_HIP2;
        switchChannels[VacSysState.SW_OR_ION_PUMP] = IonPumpDevice.CHAN_OIP;
        switchChannels[VacSysState.SW_CRYO_GATE_VALVE] = VacPlutoDevice.SW_OPEN_VCR00;
        switchChannels[VacSysState.SW_HX_GATE_VALVE] = VacPlutoDevice.SW_OPEN_VHX00;
        switchChannels[VacSysState.SW_OR_FPP_VALVE] = VacPlutoDevice.SW_OPEN_VCR01;  // Check!
        switchChannels[VacSysState.SW_OR_FH_VALVE] = VacPlutoDevice.SW_OPEN_VCR02;   // Check!
        switchChannels[VacSysState.SW_OR_L3H_VALVE] = VacPlutoDevice.SW_OPEN_VCR03;  // Check!
        switchChannels[VacSysState.SW_OR_L3_VALVE] = VacPlutoDevice.SW_OPEN_VCR04;   // Check!
    }
    private static final Map<PumpStatus, DeviceState> turboStateMap = new HashMap<>();
    static {
        turboStateMap.put(PumpStatus.STOP, DeviceState.STOPPED);
        turboStateMap.put(PumpStatus.WAIT_INTLK, DeviceState.WAITING);
        turboStateMap.put(PumpStatus.STARTING, DeviceState.STARTNG);
        turboStateMap.put(PumpStatus.NORMAL, DeviceState.NORMAL);
        turboStateMap.put(PumpStatus.BRAKING, DeviceState.BRAKING);
        turboStateMap.put(PumpStatus.FAIL, DeviceState.FAILED);
        turboStateMap.put(PumpStatus.AUTO_TUNING, DeviceState.AUTOTUN);
    }
    private static final Map<Integer, VacuumAlert> alertMap = new HashMap<>();
    static {
        alertMap.put(VacSysState.LATCH_CR_VACUUM, VacuumAlert.CRYO_VACUUM_BAD);
        alertMap.put(VacSysState.LATCH_CR_GATE_NFC, VacuumAlert.CRYO_GATE_FORCED_SHUT);
        alertMap.put(VacSysState.LATCH_CR_GATE_AO, VacuumAlert.CRYO_GATE_CANNOT_OPEN);
        alertMap.put(VacSysState.LATCH_CR_PUMP, VacuumAlert.CRYO_TURBO_PUMP_BAD);
        alertMap.put(VacSysState.LATCH_HX_VACUUM, VacuumAlert.HX_VACUUM_BAD);
        alertMap.put(VacSysState.LATCH_HX_GATE_NFC, VacuumAlert.HX_GATE_FORCED_SHUT);
        alertMap.put(VacSysState.LATCH_HX_GATE_AO, VacuumAlert.HX_GATE_CANNOT_OPEN);
        alertMap.put(VacSysState.LATCH_HX_PUMP, VacuumAlert.HX_TURBO_PUMP_BAD);
    }
    private static final Map<String, Integer> revAlertMap = new HashMap<>();
    static {
        for (int cond : alertMap.keySet()) {
            revAlertMap.put(alertMap.get(cond).getId(), cond);
        }
    }

    @LookupName
    private String name;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;
    //@LookupField(strategy = LookupField.Strategy.TREE)
    //private AgentStateService stateService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService propertiesService;

    @LookupField(strategy=LookupField.Strategy.DESCENDANTS)
    private VacPlutoDevice plutoDevc;
    @LookupField(strategy=LookupField.Strategy.DESCENDANTS)
    private CryoTurboDevice cryoTurboDevc;
    @LookupField(strategy=LookupField.Strategy.DESCENDANTS)
    private HxTurboDevice hxTurboDevc;
    @LookupField(strategy=LookupField.Strategy.DESCENDANTS)
    private IonPumpDevice ionPumpDevc;
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private final Map<String, Channel> channelMap = new LinkedHashMap<>();

    // From Groovy file
    private String cryoPressChan, turboPressChan, turboSpeedChan, forelinePressChan;
    private List<Integer> switches;

    // General
    private static final Logger LOG = Logger.getLogger(VacuumMain.class.getName());
    private final VacSysState vacState = new VacSysState();
    private Set<Integer> switchSet;
    private final Device[] switchDevices = new Device[VacSysState.NUM_SWITCHES];
    private Channel cryoPressure, turboPressure, turboSpeed, forelinePressure;
    private boolean running = false;
    private final Map<String, Integer> switchNameMap = new LinkedHashMap<>();
    private final Map<String, Boolean> activeAlertMap = new HashMap<>();
    private long ionOverStartTime = 0;
    private double forelinePress, turboPress, cryoPress, turboSpd;
    private int forelinePrErrs, turboPrErrs, cryoPrErrs, turboSpErrs;


    /**
     *  Constructor
     */
    public VacuumMain() {
        super("vacuum", AgentInfo.AgentType.WORKER);
        getAgentInfo().getAgentProperties().setProperty("org.lsst.ccs.use.full.paths", "true");
    }


    /**
     *  Build phase
     */
    @Override
    public void build() {
        // Create and schedule an AgentPeriodicTask to update the vacuum state
        AgentPeriodicTask pt;
        pt = new AgentPeriodicTask("vacuum-state", () -> updateVacuumState()).withPeriod(Duration.ofMillis(1000));
        periodicTaskService.scheduleAgentPeriodicTask(pt);

        // Register vacuum state
        stateService.registerState(VacuumState.class, "Vacuum state", this);
        stateService.updateAgentState(VacuumState.UNKNOWN);

        // Initialize active alert map
        for (VacuumAlert alert : VacuumAlert.values()) {
            activeAlertMap.put(alert.getId(), false);
        }
    }


    /**
     *  Post initialization phase
     */
    @Override
    public void postInit() {

        //Set a property to define that this Agent is a vacuum subsystem.
        propertiesService.setAgentProperty(VacuumAgentProperties.VACUUM_TYPE, VacuumMain.class.getCanonicalName());

        // Add alert listener
        alertService.addListener(this);

        if (switches != null) {
            switchSet = new HashSet<>(switches);
        }
        else {
            ErrorUtils.reportConfigError(LOG, name, "Switch list", "not specified");
        }
        if (plutoDevc != null) {
            for (int cond : plutoDevc.getLatchIds()) {
                vacState.addLatch(cond);
                vacState.setLatch(cond, LatchState.CLEAR);
            }
            for (int cond : plutoDevc.getConditionIds()) {
                vacState.addCondition(cond);
                vacState.setCondition(cond, ConditionState.NO);
            }
        }
        else {
            ErrorUtils.reportConfigError(LOG, name, "Pluto device", "not specified");
        }
        if (cryoTurboDevc == null) {
            ErrorUtils.reportConfigError(LOG, name, "Cryo turbo pump device", "not specified");
        }
        /*
        if (hxTurboDevc == null) {
            ErrorUtils.reportConfigError(LOG, name, "HX turbo pump device", "not specified");
        }
        */
        if (ionPumpDevc == null) {
            ErrorUtils.reportConfigError(LOG, name, "Ion pump device", "not specified");
        }
        if (forelinePressChan != null) {
            forelinePressure = channelMap.get(forelinePressChan);
        }
        if (forelinePressure == null) {
            ErrorUtils.reportConfigError(LOG, name, "Foreline pressure channel", "not specified or not defined");
        }
        if (cryoPressChan != null) {
            cryoPressure = channelMap.get(cryoPressChan);
        }
        if (cryoPressure == null) {
            ErrorUtils.reportConfigError(LOG, name, "Cryo pressure channel", "not specified or not defined");
        }
        if (turboPressChan != null) {
            turboPressure = channelMap.get(turboPressChan);
        }
        if (turboPressure == null) {
            ErrorUtils.reportConfigError(LOG, name, "Turbo pump pressure channel", "not specified or not defined");
        }
        if (turboSpeedChan != null) {
            turboSpeed = channelMap.get(turboSpeedChan);
        }
        if (turboSpeed == null) {
            ErrorUtils.reportConfigError(LOG, name, "Turbo pump speed channel", "not specified or not defined");
        }

        List<Integer> ipChannels = ionPumpDevc.getChannelNumbers();
        Iterator swIter = switchSet.iterator();
        while (swIter.hasNext()) {
            int sw = (Integer)swIter.next();
            switch (sw) {
            case VacSysState.SW_CRYO_TURBO_PUMP:
                switchDevices[sw] = cryoTurboDevc;
                break;
            case VacSysState.SW_HX_TURBO_PUMP:
                switchDevices[sw] = hxTurboDevc;
                break;
            case VacSysState.SW_CRYO_GATE_VALVE:
            case VacSysState.SW_HX_GATE_VALVE:
            case VacSysState.SW_OR_FPP_VALVE:
            case VacSysState.SW_OR_FH_VALVE:
            case VacSysState.SW_OR_L3H_VALVE:
            case VacSysState.SW_OR_L3_VALVE:
                switchDevices[sw] = plutoDevc;
                break;
            case VacSysState.SW_CRYO_ION_PUMP1:
            case VacSysState.SW_CRYO_ION_PUMP2:
            case VacSysState.SW_CRYO_ION_PUMP3:
            case VacSysState.SW_CRYO_ION_PUMP4:
            case VacSysState.SW_CRYO_ION_PUMP5:
            case VacSysState.SW_CRYO_ION_PUMP6:
            case VacSysState.SW_HX_ION_PUMP1:
            case VacSysState.SW_HX_ION_PUMP2:
            case VacSysState.SW_OR_ION_PUMP:
                switchDevices[sw] = ionPumpDevc;
                if (!ipChannels.contains(switchChannels[sw])) {
                    swIter.remove();
                }
                break;
            }
        }

        for (int sw : switchSet) {
            vacState.addSwitch(sw);
            vacState.setSwitchState(sw, SwitchState.OFFLINE);
            vacState.setSwitchEnable(sw, SwitchEnable.OFF);
        }

        for (int sw : switchSet) {
            switchNameMap.put(SwitchNames.ID_MAP.get(sw), sw);
        }

        ionPumpDevc.setCryoPressureChannel(cryoPressure);
    }


    /**
     *  Post start
     */
    @Override
    public void postStart() {
        LOG.info("Vacuum subsystem started");
        running = true;
    }


    /**
     *  Gets the state of the Vacuum system.
     *
     *  @return  The vacuum state
     */
    @Command(type=Command.CommandType.QUERY, description="Get the vacuum system state", level=0)
    public VacSysState getVacuumState() {
        vacState.setTickMillis(getTickPeriod());
        return vacState;
    }    


    /**
     *  Gets the list of switch names.
     *
     *  @return  The switch names.
     *  @throws  VacuumException
     */
    @Command(type=Command.CommandType.QUERY, description="Get switch names", level=0)
    public List<String> getSwitchNames() throws VacuumException
    {
        return new ArrayList(switchNameMap.keySet());
    }


    /**
     *  Turns a switch on or off.
     *
     *  @param  sw  The switch number.
     *  @param  on  Whether to turn on or off
     *  @throws  VacuumException
     */
    @Command(type=Command.CommandType.ACTION, description="Turn on/off a switch")
    public void setSwitchOn(@Argument(description="The switch number") int sw,
                            @Argument(description="Whether to turn on") boolean on) throws VacuumException
    {
        try {
            if (!switchSet.contains(sw)) {
                throw new VacuumException("Invalid switch number: " + sw);
            }
            setSwitch(sw, on);
        }
        finally {
            publishState();
        }
        
    }


    /**
     *  Turns a named switch on or off.
     *
     *  @param  name  The switch name.
     *  @param  on    Whether to turn on or off
     *  @throws  VacuumException
     */
    @Command(type=Command.CommandType.ACTION, description="Turn on/off a named switch")
    public void setNamedSwitchOn(@Argument(description="The switch name") String name,
                                 @Argument(description="Whether to turn on") boolean on) throws VacuumException
    {
        Integer sw = switchNameMap.get(name);
        try {
            if (sw == null) {
                throw new VacuumException("Invalid switch name: " + name);
            }
            setSwitch(sw, on);
        }
        finally {
            publishState();
        }
    }


    /**
     *  Turns a switch on or off.
     *
     *  @param  sw  The switch number.
     *  @param  on  Whether to turn on or off
     *  @throws  VacuumException
     */
    private void setSwitch(int sw, boolean on) throws VacuumException
    {
        SwitchState state = vacState.getSwitchState(sw);
        if (state == SwitchState.OFFLINE) return;
        SwitchEnable enable = vacState.getSwitchEnable(sw);
        if (on && enable != SwitchEnable.ON) return;
        Device swDevice = switchDevices[sw];
        try {
            if (swDevice instanceof TwisTorr84Device) {
                if (on) {
                    ((TwisTorr84Device)swDevice).startTurboPump();
                }
                else {
                    ((TwisTorr84Device)swDevice).stopTurboPump();
                }
            }
            else if (swDevice == plutoDevc) {
                plutoDevc.setSwitchOn(switchChannels[sw], on);
            }
            else if (swDevice == ionPumpDevc) {
                ionPumpDevc.setChannelOn(switchChannels[sw], on);
            }
        }
        catch (DriverException e) {
            throw new VacuumException(e);
        }
    }


    /**
     *  Gets whether a switch is on.
     *
     *  @param  sw  The switch number.
     *  @return  Whether the switch is on
     *  @throws  VacuumException
     */
    private Boolean isSwitchOn(int sw)
    {
        Boolean value = false;
        Device swDevice = switchDevices[sw];
        if (swDevice instanceof TwisTorr84Device) {
            try {
                DeviceState st = turboStateMap.get(((TwisTorr84Device)swDevice).readTurboStatus());
                value = st != DeviceState.STOPPED && st != DeviceState.BRAKING;
            }
            catch (DriverException e) {
                value = null;
            }
        }
        else if (swDevice == plutoDevc) {
            value = plutoDevc.isSwitchOn(switchChannels[sw]);
        }
        else if (swDevice == ionPumpDevc) {
            value = ionPumpDevc.isChannelOn(switchChannels[sw]);
        }
        return value;
    }


    /**
     *  Clears a latched condition.
     *
     *  @param  cond  The condition number.
     *  @throws  VacuumException
     */
    @Command(type=Command.CommandType.ACTION, description="Clear a condition")
    public void clearLatch(@Argument(description="The condition number") int cond) throws VacuumException
    {
        try {
            plutoDevc.clearLatch(cond);
        }
        finally {
            publishState();
        }
    }


    /**
     *  Sets the update period.
     *
     *  @param  value  The update period (milliseconds) to set.
     */
    @Command(type=Command.CommandType.ACTION, description="Set the update interval")
    public void setUpdatePeriod(@Argument(description="The tick period (ms)") int value)
    {
        setTickPeriod(value);
        vacState.setTickMillis(getTickPeriod());
        publishState();
    }


    /**
     *  Updates the vacuum system state periodically.
     *
     *  The vacuum state consists mainly of the state of the switches (lines) being
     *  controlled, along with whether they can be turned on.
     *
     *  Pressures and pump speeds are read to determine which switches are enabled
     *  and/or need to be turned off.  If the state changes it is published.
     */
    private void updateVacuumState()
    {
        if (!running) return;

        Boolean plcActive = plutoDevc.isPLCActive();
        if (plcActive == null) {
            if (vacState.getPlcState() != PLCState.OFFLINE) {
                raiseAlert(VacuumAlert.VACUUM_PLC_NOT_ALIVE, "Vacuum PLC is offline");
                vacState.setPlcState(PLCState.OFFLINE);
            }
        }
        else if (!plcActive) {
            if (vacState.getPlcState() != PLCState.DEAD) {
                raiseAlert(VacuumAlert.VACUUM_PLC_NOT_ALIVE, "Vacuum PLC has died");
                vacState.setPlcState(PLCState.DEAD);
            }
        }
        else {
            if (vacState.getPlcState() != PLCState.ALIVE) {
                lowerAlert(VacuumAlert.VACUUM_PLC_NOT_ALIVE, "Vacuum PLC is alive");
                vacState.setPlcState(PLCState.ALIVE);
            }
        }

        double forelinePr = forelinePressure.getValue();
        if (!Double.isNaN(forelinePr)) {
            forelinePress = forelinePr;
            forelinePrErrs = 0;
        }
        else if (forelinePrErrs++ < MAX_PRESS_ERRORS) {
            forelinePr = forelinePress;
        }
        double cryoPr = cryoPressure.getValue();
        if (!Double.isNaN(cryoPr)) {
            cryoPress = cryoPr;
            cryoPrErrs = 0;
        }
        else if (cryoPrErrs++ < MAX_PRESS_ERRORS) {
            cryoPr = cryoPress;
        }
        double turboPr = turboPressure.getValue();
        if (!Double.isNaN(turboPr)) {
            turboPress = turboPr;
            turboPrErrs = 0;
        }
        else if (turboPrErrs++ < MAX_PRESS_ERRORS) {
            turboPr = turboPress;
        }
        double turboSp = turboSpeed.getValue();
        if (!Double.isNaN(turboSp)) {
            turboSpd = turboSp;
            turboSpErrs = 0;
        }
        else if (turboSpErrs++ < MAX_SPEED_ERRORS) {
            turboSp = turboSpd;
        }
        boolean changed = false;

        for (int sw = 0; sw < VacSysState.NUM_SWITCHES; sw++) {
            if (!vacState.hasSwitch(sw)) continue;
            Boolean enable = false;
            boolean turnOff = false;
            DeviceState devState = null;
            VacuumAlert alert = null;
            switch (sw) {
 
            case VacSysState.SW_CRYO_GATE_VALVE:
                alert = VacuumAlert.CRYO_GATE_CLOSED;
                boolean cryoTurbAvail = !Double.isNaN(cryoPr) && !Double.isNaN(turboPr) && !Double.isNaN(turboSp);
                if (cryoTurbAvail) {
                    double prDiff = Math.abs(turboPr - cryoPr);
                    enable = (prDiff <= PRESS_DIFF_HIGH && turboSp <= TURBO_LOW)
                                || (prDiff <= PRESS_DIFF_LOW && turboSp > TURBO_LOW);
                }
                boolean enable2 = cryoTurbAvail && !Double.isNaN(forelinePr)
                                    && (forelinePr < PRESS_FORELINE_LOW || turboSp <= TURBO_LOW);
                enable &= enable2;
                turnOff = !enable2;
                break;

            case VacSysState.SW_CRYO_TURBO_PUMP:
                alert = VacuumAlert.CRYO_TURBO_PUMP_STOPPED;
                if (!Double.isNaN(turboPr) && !Double.isNaN(forelinePr) && !Double.isNaN(turboSp)) {
                    enable = turboPr < PRESS_TURBO_LOW && (forelinePr < PRESS_FORELINE_LOW || turboSp < TURBO_LOW);
                }
                turnOff = !enable;
                try {
                    devState = turboStateMap.get(((CryoTurboDevice)switchDevices[sw]).readTurboStatus());
                }
                catch (DriverException e) {}
                break;

            case VacSysState.SW_CRYO_ION_PUMP1:
            case VacSysState.SW_CRYO_ION_PUMP2:
            case VacSysState.SW_CRYO_ION_PUMP3:
            case VacSysState.SW_CRYO_ION_PUMP4:
            case VacSysState.SW_CRYO_ION_PUMP5:
            case VacSysState.SW_CRYO_ION_PUMP6:
                alert = VacuumAlert.CRYO_ION_PUMPS_STOPPED;
                if (!Double.isNaN(cryoPr)) {
                    enable = cryoPr < PRESS_ION_ENABLE;
                    if (cryoPr >= PRESS_ION_OFF) {
                        if (ionOverStartTime == 0) {
                            ionOverStartTime = System.currentTimeMillis();
                        }
                        else {
                            turnOff = System.currentTimeMillis() - ionOverStartTime >= DELAY_ION_OFF;
                        }
                    }
                    else {
                        ionOverStartTime = 0;
                    }
                }
                else {
                    turnOff = true;
                }
                break;
            }

            SwitchState oldState = vacState.getSwitchState(sw);
            if (turnOff && oldState == SwitchState.ON) {
                try {
                    setSwitch(sw, false);
                    raiseAlert(alert, "Switch was forced to open");
                }
                catch (VacuumException e) {
                    LOG.log(Level.SEVERE, "Error setting switch: {0}", e);
                }
            }
            Boolean isOn = isSwitchOn(sw);
            SwitchState state = isOn != null ? isOn ? SwitchState.ON : SwitchState.OFF : SwitchState.OFFLINE;
            SwitchEnable enabled = enable ? SwitchEnable.ON : SwitchEnable.OFF;
            if (state != oldState || enabled != vacState.getSwitchEnable(sw)) {
                vacState.setSwitchState(sw, state);
                vacState.setSwitchEnable(sw, enabled);
                changed = true;
                if (enable && isAlertRaised(alert)) {
                    lowerAlert(alert, "Switch has become re-enabled");
                }
            }
            if (devState != vacState.getDeviceState(sw)) {
                vacState.setDeviceState(sw, devState);
                changed = true;
            }
        }

        for (int cond = 0; cond < VacSysState.NUM_LATCHES; cond++) {
            if (!vacState.hasLatch(cond)) continue;
            Boolean active = plutoDevc.isLatchActive(cond);
            Boolean latched = plutoDevc.isLatchLatched(cond);
            LatchState state = active == null || latched == null ? LatchState.OFFLINE :
                               latched ? LatchState.LATCHED :
                               active ? LatchState.ACTIVE : LatchState.CLEAR;
            LatchState oldState = vacState.getLatch(cond); 
            if (state != oldState) {
                vacState.setLatch(cond, state);
                VacuumAlert alert = alertMap.get(cond);
                if (state == LatchState.ACTIVE) {
                    raiseAlert(alert, "Vacuum PLC error condition set");
                }
                else if (state != LatchState.OFFLINE && oldState != LatchState.OFFLINE) {
                    if (state == LatchState.LATCHED && oldState == LatchState.CLEAR) {
                        raiseAlert(alert, "Vacuum PLC error condition set");
                    }
                    if (state == LatchState.LATCHED || oldState == LatchState.ACTIVE) {
                        lowerAlert(alert, "Vacuum PLC error condition cleared");
                    }
                }
                changed = true;
            }
        }

        for (int cond = 0; cond < VacSysState.NUM_CONDITIONS; cond++) {
            if (!vacState.hasCondition(cond)) continue;
            Boolean active = plutoDevc.isConditionActive(cond);
            ConditionState state = active == null ? ConditionState.OFF :
                                   active ? ConditionState.YES : ConditionState.NO;
            if (state != vacState.getCondition(cond)) {
                vacState.setCondition(cond, state);
                changed = true;
            }
        }

        VacuumState vState;
        if (Double.isNaN(cryoPr)) {
            vState = VacuumState.UNKNOWN;
        }
        else if (cryoPr <= PRESS_VACUUM) {
            vState = VacuumState.VACUUM;
        }
        else if (vacState.getSwitchState(VacSysState.SW_CRYO_ION_PUMP1) == SwitchState.ON) {
            vState = VacuumState.ION_ON;
        }
        else if (vacState.getSwitchEnable(VacSysState.SW_CRYO_ION_PUMP1) == SwitchEnable.ON) {
            vState = VacuumState.ION_OFF;
        }
        else if (vacState.getSwitchState(VacSysState.SW_CRYO_TURBO_PUMP) == SwitchState.ON) {
            vState = VacuumState.TURBO_ON;
        }
        else if (vacState.getSwitchEnable(VacSysState.SW_CRYO_TURBO_PUMP) == SwitchEnable.ON) {
            vState = VacuumState.TURBO_OFF;
        }
        else if (cryoPr < PRESS_ATMOS) {
            vState = VacuumState.FORELINE;
        }
        else {
            vState = VacuumState.OFF;
        }
        if (vState != vacState.getVacuumState()) {
            vacState.setVacuumState(vState);
            stateService.updateAgentState(vState);
            changed = true;
        }

        if (changed) {
            publishState();
        }
    }


    /**
     *  Raises an alert.
     *
     *  @param  alert  The vacuum alert to raise
     *  @param  cond   The alert condition
     */
    private void raiseAlert(VacuumAlert alert, String cond)
    {
        alertService.raiseAlert(alert.getAlert(), AlertState.ALARM, cond);
        activeAlertMap.put(alert.getId(), true);
    }


     /**
     *  Checks whether an alert has been raised.
     *
     *  @param  alert  The vacuum alert to check
     *  @return  Whether it has been raised
     */
    private boolean isAlertRaised(VacuumAlert alert)
    {
        return activeAlertMap.get(alert.getId()).equals(Boolean.TRUE);
    }


    /**
     *  Lowers an alert.
     *
     *  @param  alert  The vacuum alert to lower
     *  @param  cond   The alert condition
     */
    private void lowerAlert(VacuumAlert alert, String cond)
    {
        alertService.raiseAlert(alert.getAlert(), AlertState.NOMINAL, cond);
        activeAlertMap.put(alert.getId(), false);
    }


    /**
     * Callback to clear an {@code Alert} instance.
     * 
     * @param alert The Alert instance to clear.
     * @param alertState The AlertState for the provided Alert.
     * @return A ClearAlertCode to indicate which action is to be taken
     *         by the framework.
     */
    @Override
    public ClearAlertCode canClearAlert(Alert alert, AlertState alertState)
    {
        Boolean active = activeAlertMap.get(alert.getAlertId());
        return active == null ? ClearAlertCode.UNKNOWN_ALERT
                              : active ? ClearAlertCode.DONT_CLEAR_ALERT : ClearAlertCode.CLEAR_ALERT;
    }


    /**
     *  Alert event handler.
     *
     *  Resets PLC latch when corresponding alert is cleared.
     *
     *  @param  event  The alert event
     */
    @Override
    public void onAlert(AlertEvent event)
    {
        if (event.getType() != AlertEvent.AlertEventType.ALERT_CLEARED) return;
        for (String id : event.getClearedIds()) {
            Integer cond = revAlertMap.get(id);
            if (cond != null) {
                try {
                    clearLatch(cond);
                }
                catch (VacuumException e) {
                    LOG.log(Level.SEVERE, "Error clearing latched PLC condition ({0}): {1}", new Object[]{cond, e});
                }
            }
        }
    }


    /**
     *  Publishes the state of the vacuum system.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    private void publishState()
    {
        publishSubsystemDataOnStatusBus(new KeyValueData(VacSysState.KEY, getVacuumState()));
    }


    /**
     *  Sets the monitoring publishing period.
     */
    private void setTickPeriod(long period)
    {
        periodicTaskService.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }
    

    /**
     *  Gets the monitoring publishing period.
     */
    private int getTickPeriod()
    {
        return (int)periodicTaskService.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }

}
