package org.lsst.ccs.subsystem.vacuum;

import java.util.ArrayList;
import java.util.List;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.subsystem.common.devices.PlutoDevice;
import org.lsst.ccs.subsystem.vacuum.data.VacSysState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Handles the vacuum system Pluto PLC.
 *
 *  @author Owen Saxton
 */
public class VacPlutoDevice extends PlutoDevice {

    /**
     *  Constants.
     */
    public static final int
        SW_BLOCK_HVSTAT = 0,
        SW_BLOCK_CVSTAT = 1,
        SW_OPEN_VHX00   = 2,
        SW_OPEN_VCR00   = 3,
        SW_OPEN_VCR01   = 4,
        SW_OPEN_VCR02   = 5,
        SW_OPEN_VCR03   = 6,
        SW_OPEN_VCR04   = 7,
        SW_BLOCK_HVPUMP = 8,
        SW_BLOCK_CVPUMP = 9,
        NUM_SWITCHES    = 10;

    private static final int
        NUM_AREAS    = 19,
        SWDI_ON_BIT = 0,
        SWDI_OFF_BIT = 1,
        SWDI_READ_AREA = 2,
        SWDI_READ_BIT = 3,
        LTDI_RESET_BIT = 0,
        LTDI_READ_AREA = 1,
        LTDI_READ_BIT = 2,
        LTDI_PEND_AREA = 3,
        LTDI_PEND_BIT = 4,
        CNDI_READ_AREA = 0,
        CNDI_READ_BIT = 1;

    /**
     *  Private lookup maps, etc.
     */
    private static final int[][] switches = new int[NUM_SWITCHES][];
    static {
        switches[SW_OPEN_VCR00] = new int[]{7, 6, 15, 0};   // ON bit, OFF bit, read area, read bit 
        switches[SW_OPEN_VCR01] = new int[]{12, 13, 12, 0};
        switches[SW_OPEN_VCR02] = new int[]{14, 15, 12, 3};
        switches[SW_OPEN_VCR03] = new int[]{16, 17, 13, 0};
        switches[SW_OPEN_VCR04] = new int[]{18, 19, 13, 3};
        switches[SW_OPEN_VHX00] = new int[]{3, 2, 15, 2};
        switches[SW_BLOCK_HVSTAT] = new int[]{0, 20, 14, 0};
        switches[SW_BLOCK_CVSTAT] = new int[]{1, 21, 14, 3};
        switches[SW_BLOCK_HVPUMP] = new int[]{10, 22, 11, 0};
        switches[SW_BLOCK_CVPUMP] = new int[]{11, 23, 11, 3};
    }
    private static final int[][] latches = new int[VacSysState.NUM_LATCHES][];
    static {
        latches[VacSysState.LATCH_CR_VACUUM] = new int[]{1, 14, 3, 18, 0};   // Reset bit, on area, on bit,
        latches[VacSysState.LATCH_CR_GATE_AO] = new int[]{9, 15, 4, 18, 2};  //    ind area, ind bit
        latches[VacSysState.LATCH_CR_GATE_NFC] = new int[]{8, 10, 3, 18, 1};
        latches[VacSysState.LATCH_CR_PUMP] = new int[]{11, 11, 3, 18, 3};
        latches[VacSysState.LATCH_HX_VACUUM] = new int[]{0, 14, 0, 18, 4};
        latches[VacSysState.LATCH_HX_GATE_AO] = new int[]{5, 15, 5, 18, 6};
        latches[VacSysState.LATCH_HX_GATE_NFC] = new int[]{4, 9, 3, 18, 5};
        latches[VacSysState.LATCH_HX_PUMP] = new int[]{10, 11, 0, 18, 7};
    }
    private static final int[][] conditions = new int[VacSysState.NUM_CONDITIONS][];
    static {
        conditions[VacSysState.COND_CR_FORELINE_VAC] = new int[]{11, 6};  // Area, bit
        conditions[VacSysState.COND_CR_TRB_PRESS_10] = new int[]{16, 5};
        conditions[VacSysState.COND_CR_TRB_PUMP_OFF] = new int[]{16, 6};
        conditions[VacSysState.COND_CR_VACUUM_001] = new int[]{17, 5};
        conditions[VacSysState.COND_CR_VACUUM_01] = new int[]{17, 4};
        conditions[VacSysState.COND_HX_FORELINE_VAC] = new int[]{11, 7};
        conditions[VacSysState.COND_HX_TRB_PRESS_10] = new int[]{16, 3};
        conditions[VacSysState.COND_HX_TURBO_OFF] = new int[]{16, 4};
        conditions[VacSysState.COND_HX_VACUUM_001] = new int[]{17, 7};
        conditions[VacSysState.COND_HX_VACUUM_01] = new int[]{17, 6};
        conditions[VacSysState.COND_OR_FH_VALVE] = new int[]{12, 0};
        conditions[VacSysState.COND_OR_FPP_VALVE] = new int[]{12, 3};
        conditions[VacSysState.COND_OR_L3H_VALVE] = new int[]{13, 0};
        conditions[VacSysState.COND_OR_L3_VALVE] = new int[]{13, 3};
    }

    /**
     *  Data fields.
     */
    private List<Integer> latchesUsed;  // Latched conditions to be used
    private List<Integer> condsUsed;    // Conditions to be used

    private static final Logger LOG = Logger.getLogger(VacPlutoDevice.class.getName());


    /**
     *   Constructor.
     */
    public VacPlutoDevice()
    {
        super(NUM_AREAS);
    }


    /**
     *  Performs basic initialization.
     */
    @Override
    protected void initDevice()
    {
        super.initDevice();
        if (latchesUsed == null) {
            latchesUsed = new ArrayList<>();
        }
        for (int latch : latchesUsed) {
            if (latch < 0 || latch >= VacSysState.NUM_LATCHES) {
                MonitorLogUtils.reportConfigError(LOG, name, "latchesUsed item (" + latch + ")", "is invalid");
            }
        }
        if (condsUsed == null) {
            condsUsed = new ArrayList<>();
        }
        for (int cond : condsUsed) {
            if (cond < 0 || cond >= VacSysState.NUM_CONDITIONS) {
                MonitorLogUtils.reportConfigError(LOG, name, "condsUsed item (" + cond + ")", "is invalid");
            }
        }
    }


    /**
     *  Sets a switch on or off.
     *
     *  For the vacuum Pluto, this is implemented as a pair of push buttons,
     *  one for on, one for off.
     *
     *  @param  sw  The switch number.
     *  @param  on  The on state to set: true or false
     */
    public void setSwitchOn(int sw, boolean on)
    {
        int bitNum = switches[sw][on ? SWDI_ON_BIT : SWDI_OFF_BIT];
        toggleBit(bitNum / 16, bitNum & 0x0f);
    }


    /**
     *  Gets the on state of a switch.
     *
     *  The state is not the state of the bit that was toggled, but is read back
     *  either directly from the controlled hardware, or from the PLC output line.
     *
     *  @param  sw  The switch number.
     *  @return  Whether the switch is on
     */
    public Boolean isSwitchOn(int sw)
    {
        int[] swData = switches[sw];
        Integer value = readAddBit(swData[SWDI_READ_AREA], swData[SWDI_READ_BIT] + 16);
        return value != null ? value != 0 : null;
    }


    /**
     *  Gets the list of latched conditions in use.
     * 
     *  @return  The condition identifiers
     */
    public List<Integer> getLatchIds()
    {
        return latchesUsed;
    }


    /**
     *  Gets whether a latched condition is active.
     * 
     *  @param  cond  The condition number
     *  @return  Whether active - indicated by the bit being 0
     */
    public Boolean isLatchActive(int cond)
    {
        int[] condData = latches[cond];
        Integer value = readAddBit(condData[LTDI_READ_AREA], condData[LTDI_READ_BIT] + 16);
        return value != null ? value == 0 : null;
    }


    /**
     *  Gets whether a latched condition is latched.
     * 
     *  @param  cond  The condition number
     *  @return  Whether latched - indicated by the bit being 1
     */
    public Boolean isLatchLatched(int cond)
    {
        int[] condData = latches[cond];
        Integer value = readAddBit(condData[LTDI_PEND_AREA], condData[LTDI_PEND_BIT] + 16);
        return value != null ? value != 0 : null;
    }


    /**
     *  Clears a latched condition.
     * 
     *  @param  cond  The condition number
     */
    public void clearLatch(int cond)
    {
        int bitNum = latches[cond][LTDI_RESET_BIT];
        toggleBit(bitNum / 16, bitNum & 0x0f);
    }


    /**
     *  Gets the list of conditions in use.
     * 
     *  @return  The condition identifiers
     */
    public List<Integer> getConditionIds()
    {
        return condsUsed;
    }


    /**
     *  Gets whether a condition is active.
     * 
     *  @param  cond  The condition number
     *  @return  Whether active - indicated by the bit being 1
     */
    public Boolean isConditionActive(int cond)
    {
        int[] condData = conditions[cond];
        Integer value = readAddBit(condData[CNDI_READ_AREA], condData[CNDI_READ_BIT] + 16);
        return value != null ? value != 0 : null;
    }

}
