package org.lsst.ccs.subsystem.vacuum;

import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.auxelex.IonPump;
import org.lsst.ccs.monitor.Control;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.subsystem.vacuum.config.VacuumConfig;
import org.lsst.ccs.subsystem.vacuum.data.VacuumException;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Implements ion pump control functions.
 *
 *  @author Owen Saxton
 */
public class IonPumpControl extends Control {

    /**
     *  Constants.
     */
    private static final String
        VOLTAGE   = "voltage",
        CURRENT   = "current",
        POWER     = "power";

    /**
     *  Data fields.
     */
    // Supplied configurable fields
    @ConfigurationParameter(name=VOLTAGE, category=VacuumConfig.CRYO)
    private double      voltage;       // Voltage
    @ConfigurationParameter(name=CURRENT, category=VacuumConfig.CRYO)
    private double      current;       // Current (limit)
    @ConfigurationParameter(name=POWER, category=VacuumConfig.CRYO)
    private double      power;         // Power (limit)

    @LookupField (strategy = LookupField.Strategy.TREE)
    private ConfigurationService sce;

    // Derived fields
    private static final Logger LOG = Logger.getLogger(IonPumpControl.class.getName());
    private String        shortName;     // Last part of dot-separated name
    private IonPumpDevice pDevc;         // Parent (ion pump controller) device


    /**
     *  Configures channel description.
     *
     *  @param  mon   The associated monitor
     *  @param  devc  The parent Device
     */
    @Override
    public void configure(Monitor mon, Device devc)
    {
        super.configure(mon, devc);
        pDevc = (IonPumpDevice)devc;
        try {
            if (hwChan < 0 || hwChan >= IonPump.NUM_CHANNELS) {
                MonitorLogUtils.reportError(LOG, getName(), "hw channel number", hwChan);
            }
        }
        catch (Exception e) {
            // Side effect of reportError call
        }
        String[] parts = getName().split("\\.");
        shortName = parts[parts.length - 1];
    }


    /**
     *  Sets/gets the voltage.
     *
     *  @param  value  The value to set
     *  @throws VacuumException
     */
    @ConfigurationParameterChanger
    public void setVoltage(double value) throws VacuumException
    {
        voltage = value;
        if (pDevc != null && pDevc.isOnline()) {
            writeVoltage();
        }
    }

    @Command(type=Command.CommandType.QUERY, description="Gets the set voltage")
    public double getVoltage()
    {
        return voltage;
    }


    /**
     *  Sets/gets the current.
     *
     *  @param  value  The value to set
     *  @throws VacuumException
     */
    @ConfigurationParameterChanger
    public void setCurrent(double value) throws VacuumException
    {
        current = value;
        if (pDevc != null && pDevc.isOnline()) {
            writeCurrent();
        }
    }

    @Command(type=Command.CommandType.QUERY, description="Gets the set current")
    public double getCurrent()
    {
        return current;
    }


    /**
     *  Sets/gets the power.
     *
     *  @param  value  The value to set
     *  @throws VacuumException
     */
    @ConfigurationParameterChanger
    public void setPower(double value) throws VacuumException
    {
        power = value;
        if (pDevc != null && pDevc.isOnline()) {
            writePower();
        }
    }

    @Command(type=Command.CommandType.QUERY, description="Gets the power")
    public double getPower()
    {
        return power;
    }


    /**
     *  Changes the voltage.
     *
     *  @param  value  The value to set
     *  @throws  VacuumException
     */
    @Command(type=Command.CommandType.ACTION, description="Changes the voltage")
    public void changeVoltage(double value) throws VacuumException
    {
        pDevc.setVoltage(hwChan, value);
    }


    /**
     *  Sets all the configuration data.
     */
    void setConfig(VacuumConfig.Channel chan)
    {
        String name = getName();
        sce.change(name, VOLTAGE, chan.getVoltage());
        sce.change(name, CURRENT, chan.getCurrent());
        sce.change(name, POWER, chan.getPower());
    }


    /**
     *  Gets all the configuration data.
     */
    VacuumConfig.Channel getConfig()
    {
        VacuumConfig.Channel chan = new VacuumConfig.Channel();
        chan.setName(shortName);
        chan.setVoltage(voltage);
        chan.setCurrent(current);
        chan.setPower(power);
        return chan;
    }


    /**
     *  Writes all configured quantities to the hardware.
     */
    void writeAll() throws Exception
    {
        writeVoltage();
        writeCurrent();
        writePower();
    }


    /**
     *  Writes the voltage to the hardware.
     */
    void writeVoltage() throws VacuumException
    {
        pDevc.setVoltage(hwChan, voltage);
    }


    /**
     *  Writes the current to the hardware.
     */
    void writeCurrent() throws VacuumException
    {
        pDevc.setCurrent(hwChan, current);
    }


    /**
     *  Writes the power to the hardware.
     */
    void writePower() throws VacuumException
    {
        pDevc.setPower(hwChan, power);
    }


    /**
     *  Turns on the power.
     */
    void powerOn() throws VacuumException
    {
        pDevc.powerOn(hwChan);
    }


    /**
     *  Turns off the power.
     */
    void powerOff() throws VacuumException
    {
        pDevc.powerOff(hwChan);
    }

}
