package org.lsst.ccs.subsystem.vacuum;

import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.mks.Model9XX;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.subsystem.vacuum.config.VacuumConfig;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Interface to an MKS 972/974/925 vacuum gauge.
 *
 *  @author Owen Saxton
 */
public class Mks9xxDevice extends Device {

    /**
     *  Constants.
     */
    private static final String
        DEVC_ID   = "devcId",
        BUS_ADDR  = "busAddr";

    /**
     *  Data fields.
     */
    @ConfigurationParameter(name=DEVC_ID, category=VacuumConfig.CRYO, isFinal=true)
    private String devcId;
    @ConfigurationParameter(name=BUS_ADDR, category=VacuumConfig.CRYO, isFinal=true)
    private int busAddr = Model9XX.DEFAULT_ADDRESS;

    private Model9XX.ConnType connType = Model9XX.ConnType.SERIAL;
    private int baudRate = Model9XX.DEFAULT_BAUDRATE;

    private static final Logger LOG = Logger.getLogger(Mks9xxDevice.class.getName());
    private final Model9XX mks = new Model9XX();
    private int errorCount = 0;


    /**
     *  Performs configuration.
     */
    @Override
    public void initDevice() {
        if (devcId == null) {
            MonitorLogUtils.reportConfigError(LOG, name, DEVC_ID, "is missing");
        }
        fullName = "MKS model 9XX (" + devcId + ":" + busAddr + ")";
    }


    /**
     *  Performs full initialization.
     */
    @Override
    public void initialize()
    {
        try {
            mks.open(connType, devcId, baudRate, busAddr);
            mks.setLock(false);
            mks.setPressureUnit(Model9XX.Unit.TORR);
            mks.setLock(true);
            errorCount = 0;
            setOnline(true);
            initSensors();
            LOG.info("Connected to " + fullName + ": model = " + mks.getModel() + ", serial no. = " + mks.getSerialNumber());
        }
        catch (DriverException e) {
            if (!inited) {
                LOG.error("Error connecting to " + fullName + ": " + e);
            }
            close();
        }
        inited = true;
    }


    /**
     *  Closes the connection.
     */
    @Override
    public void close()
    {
        try {
            mks.close();
        }
        catch (DriverException e) {
        }
    }


    /**
     *  Reads a monitoring channel.
     *
     *  @param  hwChan  The hardware channel number
     *  @param  type    The encoded channel type
     *  @return  The read value
     */
    @Override
    public double readChannel(int hwChan, int type)
    {
        double value = Double.NaN;
        if (online) {
            try {
                value = mks.readPressure(Model9XX.Sensor.COMB4);
                errorCount = 0;
            }
            catch (DriverException e) {
                LOG.error("Error reading pressure from " + fullName + ": " + e);
                if (++errorCount >= 5) {
                    setOnline(false);
                }
            }
        }
        return value;
    }

}
