package org.lsst.ccs.subsystem.vacuum;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.pluto.Pluto;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.subsystem.vacuum.data.VacuumState;

/**
 *  Handles the vacuum system Pluto PLC.
 *
 *  @author Owen Saxton
 */
public class VacPlutoDevice extends Device {

    /**
     *  Constants.
     */
    public static final int
        SW_BLOCK_HVSTAT = 0,
        SW_BLOCK_CVSTAT = 1,
        SW_OPEN_VHX00   = 2,
        SW_OPEN_VCR00   = 3,
        SW_OPEN_VCR01   = 4,
        SW_OPEN_VCR02   = 5,
        SW_OPEN_VCR03   = 6,
        SW_OPEN_VCR04   = 7,
        SW_BLOCK_HVPUMP = 8,
        SW_BLOCK_CVPUMP = 9,
        NUM_SWITCHES    = 10;

    private static final int
        TYPE_GLOBAL  = 0,
        TYPE_ADD_BIT = 1,
        TYPE_ADD_REG = 2,
        NUM_AREAS    = 19,
        SWDI_ON_BIT = 0,
        SWDI_OFF_BIT = 1,
        SWDI_READ_AREA = 2,
        SWDI_READ_BIT = 3,
        CNDI_RESET_BIT = 0,
        CNDI_READ_AREA = 1,
        CNDI_READ_BIT = 2,
        CNDI_PEND_AREA = 3,
        CNDI_PEND_BIT = 4;

    /**
     *  Private lookup maps, etc.
     */
    private static final Map<String, Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("GLOBAL", TYPE_GLOBAL);
        typeMap.put("ADDBIT", TYPE_ADD_BIT);
        typeMap.put("ADDREG", TYPE_ADD_REG);
    }
    private static final int[][] switches = new int[NUM_SWITCHES][];
    static {
        switches[SW_OPEN_VCR00] = new int[]{7, 6, 15, 0};   // ON bit, OFF bit, read area, read bit 
        switches[SW_OPEN_VCR01] = new int[]{12, 13, 12, 0};
        switches[SW_OPEN_VCR02] = new int[]{14, 15, 12, 3};
        switches[SW_OPEN_VCR03] = new int[]{16, 17, 13, 0};
        switches[SW_OPEN_VCR04] = new int[]{18, 19, 13, 3};
        switches[SW_OPEN_VHX00] = new int[]{3, 2, 15, 2};
        switches[SW_BLOCK_HVSTAT] = new int[]{0, 20, 14, 0};
        switches[SW_BLOCK_CVSTAT] = new int[]{1, 21, 14, 3};
        switches[SW_BLOCK_HVPUMP] = new int[]{10, 22, 11, 0};
        switches[SW_BLOCK_CVPUMP] = new int[]{11, 23, 11, 3};
    }
    private static final int[][] conditions = new int[VacuumState.NUM_CONDITIONS][];
    static {
        conditions[VacuumState.COND_CR_VACUUM] = new int[]{1, 14, 3, 18, 0};   // Reset bit, on area, on bit,
        conditions[VacuumState.COND_CR_GATE_AO] = new int[]{9, 15, 4, 18, 2};  //    ind area, ind bit
        conditions[VacuumState.COND_CR_GATE_NFC] = new int[]{8, 10, 3, 18, 1};
        conditions[VacuumState.COND_CR_PUMP] = new int[]{11, 11, 3, 18, 3};
        conditions[VacuumState.COND_HX_VACUUM] = new int[]{0, 14, 0, 18, 4};
        conditions[VacuumState.COND_HX_GATE_AO] = new int[]{5, 15, 5, 18, 6};
        conditions[VacuumState.COND_HX_GATE_NFC] = new int[]{4, 9, 3, 18, 5};
        conditions[VacuumState.COND_HX_PUMP] = new int[]{10, 11, 0, 18, 7};
    }

    /**
     *  Data fields.
     */
    private String node;          // Network node name
    private List<Integer> condsUsed;  // Conditions to be used
    //private List<Integer> areas;  // Area configuration data

    //private int nArea;                      // Area count
    private final Pluto plu = new Pluto();  // Associated Pluto object
    private final int[] globData = new int[Pluto.NUM_MODULES];
    private final int[] addData = new int[Pluto.NUM_ADD_AREAS];
    private final Set<Integer> modsUsed = new HashSet<>();
    private final Set<Integer> areasUsed = new HashSet<>();


    /**
     *  Performs basic initialization.
     */
    @Override
    protected void initDevice()
    {
        if (node == null) {
            MonitorLogUtils.reportConfigError(log, name, "node", "is missing");
        }
        if (condsUsed == null) {
            condsUsed = new ArrayList<>();
        }
        for (int cond : condsUsed) {
            if (cond < 0 || cond >= VacuumState.NUM_CONDITIONS) {
                MonitorLogUtils.reportConfigError(log, name, "condsUsed item", "is invalid");
            }
        }
         /*
        if (areas == null) {
            MonitorLogUtils.reportConfigError(log, name, "areas", "is missing");
        }
        if ((areas.size() & 1) != 0) {
            MonitorLogUtils.reportConfigError(log, name, "areas", "has odd number of elements");
        }
        nArea = areas.size() / 2;
         */
        fullName = "Pluto PLC system (" + node + ")";
    }


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        try {
            plu.open(node);
             /*
            plu.configStart(-1, 15, 0, 100);
            for (int j = 0; j < areas.size(); j += 2) {
                plu.configDataArea(j / 2, areas.get(j), areas.get(j + 1));
            }
            plu.configWrite();
             */
            setOnline(true);
            initSensors();
            setOutputLines();
            log.info("Connected to " + fullName);
        }
        catch (DriverException e) {
            if (!inited) {
                log.error("Error connecting to " + fullName + ": " + e);
            }
            close();
        }
        inited = true;
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        try {
            plu.close();
        }
        catch (DriverException e) {
        }
    }


    /**
     *  Checks a channel's parameters for validity.
     *
     *  The type argument specifies what kind of quantity is being read, and is
     *  of the form type:id
     * 
     *  where type is ADDBIT, ADDREG or GLOBAL
     *  and id is an integer identifying the instance of the type
     *
     *  The subType argument is not used.
     *  The allowed values of hwChan depend on the type:
     *    ADDBIT: 0-15, ADDREG: 0-1, GLOBAL: 0-31
     * 
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel number
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string
     *  @return  A two-element array containing the encoded type [0] and
     *           subtype [1] values.
     *  @throws  Exception if any errors found in the parameters.
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype)
        throws Exception
    {
        Integer iType = null;
        int id = -1;
        String[] typeFields = type.split(":", -1);
        if (typeFields.length == 2) {
            iType = typeMap.get(typeFields[0].toUpperCase());
            try {
                id = Integer.decode(typeFields[1]);
            }
            catch (NumberFormatException e) {
            }
        }
        int maxId = 0;
        if (iType != null) {
            maxId = iType == TYPE_GLOBAL ? Pluto.NUM_MODULES : NUM_AREAS;
            int maxChan = iType == TYPE_GLOBAL ? 32 : iType == TYPE_ADD_BIT ? 16 : 2;
            if (hwChan < 0 || hwChan >= maxChan) {
                MonitorLogUtils.reportError(log, name, "hwchan", hwChan);
            }
        }
        if (iType == null || id < 0 || id >= maxId) {
            MonitorLogUtils.reportError(log, name, "type", type);
        }
        Set used = iType == TYPE_GLOBAL ? modsUsed : areasUsed;
        used.add(id);

        return new int[]{(iType << 8) | id, 0};
    }


    /**
     *  Reads all referenced channels.
     */
    @Override
    protected void readChannelGroup()
    {
        if (!online) return;
        try {
            for (int id : modsUsed) {
                globData[id] = plu.readGlobalData(id);
            }
            for (int id : areasUsed) {
                addData[id] = plu.readAdditionalData(id);
            }
        }
        catch (DriverException e) {
            log.error("Error reading " + fullName + ": " + e);
            setOnline(false);
        }
    }


    /**
     *  Reads a channel.
     *
     *  @param  hwChan   The hardware channel number.
     *  @param  type     The encoded channel type returned by checkChannel.
     *  @return  The read value
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = super.readChannel(hwChan, type);

        if (online) {
            int index = type & 0xff;
            switch (type >> 8) {

            case TYPE_GLOBAL:
                value = (globData[index] >> hwChan) & 1;
                break;

            case TYPE_ADD_BIT:
                value = (addData[index] >> hwChan) & 1;
                break;

            case TYPE_ADD_REG:
                value = (addData[index] << (16 * (1 - hwChan))) >> 16;  // Sign extend
                break;
            }
        }

        return value;
    }


    /**
     *  Sets a switch on or off.
     *
     *  For the vacuum Pluto, this is implemented as a pair of push buttons,
     *  one for on, one for off.
     *
     *  @param  sw  The switch number.
     *  @param  on  The on state to set: true or false
     */
    public void setSwitchOn(int sw, boolean on)
    {
        int bitNum = switches[sw][on ? SWDI_ON_BIT : SWDI_OFF_BIT];
        writeBit(bitNum / 16, bitNum & 0x0f, 1);
        sleep(200);
        writeBit(bitNum / 16, bitNum & 0x0f, 0);
    }


    /**
     *  Gets the on state of a switch.
     *
     *  The state is not the state of the bit that was toggled, but is read back
     *  either directly from the controlled hardware, or from the PLC output line.
     *
     *  @param  sw  The switch number.
     *  @return  Whether the switch is on
     */
    public Boolean isSwitchOn(int sw)
    {
        int[] swData = switches[sw];
        Integer value = readAddBit(swData[SWDI_READ_AREA], swData[SWDI_READ_BIT] + 16);
        return value != null ? value != 0 : null;
    }


    /**
     *  Gets the list of conditions in use.
     * 
     *  @return  The condition identifiers
     */
    public List<Integer> getConditionIds()
    {
        return condsUsed;
    }


    /**
     *  Gets whether a condition is active.
     * 
     *  @param  cond  The condition number
     *  @return  Whether active - indicated by the bit being 0
     */
    public Boolean isConditionActive(int cond)
    {
        int[] condData = conditions[cond];
        Integer value = readAddBit(condData[CNDI_READ_AREA], condData[CNDI_READ_BIT] + 16);
        return value != null ? value == 0 : null;
    }


    /**
     *  Gets whether a condition is latched.
     * 
     *  @param  cond  The condition number
     *  @return  Whether latched - indicated by the bit being 1
     */
    public Boolean isConditionLatched(int cond)
    {
        int[] condData = conditions[cond];
        Integer value = readAddBit(condData[CNDI_PEND_AREA], condData[CNDI_PEND_BIT] + 16);
        return value != null ? value != 0 : null;
    }


    /**
     *  Clears a condition.
     * 
     *  @param  cond  The condition number
     */
    public void clearCondition(int cond)
    {
        int bitNum = conditions[cond][CNDI_RESET_BIT];
        writeBit(bitNum / 16, bitNum & 0x0f, 1);
        sleep(200);
        writeBit(bitNum / 16, bitNum & 0x0f, 0);
    }


    /**
     *  Writes a bit value.
     *
     *  @param  area   The area number
     *  @param  bit    The bit number
     *  @param  value  The bit value
     */
   void writeBit(int area, int bit, int value)
    {
        try {
            plu.writeAreaBit(area, bit, value);
        }
        catch (DriverException e) {
            log.error("Error writing to " + fullName + ": " + e);
            setOnline(false);
        }
    }


    /**
     *  Reads a bit value.
     *
     *  @param  area   The area number
     *  @param  bit    The bit number
     *  @return  The bit value
     */
    int readBit(int area, int bit)
    {
        try {
            return plu.readAreaBit(area, bit);
        }
        catch (DriverException e) {
            log.error("Error reading from " + fullName + ": " + e);
            setOnline(false);
            return 0;
        }
    }


    /**
     *  Reads a bit value from an additional area.
     *
     *  @param  area   The area number
     *  @param  bit    The bit number
     *  @return  The bit value, or null if device offline
     */
    Integer readAddBit(int area, int bit)
    {
        if (!online) return null;
        try {
            return (plu.readAdditionalData(area) >> bit) & 1;
        }
        catch (DriverException e) {
            log.error("Error reading from " + fullName + ": " + e);
            setOnline(false);
            return null;
        }
    }


    /**
     *  Writes a register value.
     *
     *  @param  area   The area number
     *  @param  reg    The register number
     *  @param  value  The register value
     */
    void writeRegister(int area, int reg, int value)
    {
        try {
            plu.writeAreaRegister(area, reg, value);
        }
        catch (DriverException e) {
            log.error("Error writing to " + fullName + ": " + e);
            setOnline(false);
        }
    }


    /**
     *  Sleeps for a while.
     *
     *  @param  time  The time to sleep (ms)
     */
    private void sleep(int time)
    {
        try {
            Thread.sleep(time);
        }
        catch (InterruptedException e) {}
    }

}
