package org.lsst.ccs.subsystem.vacuum;

import java.util.ArrayList;
import java.time.Duration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.lsst.ccs.Subsystem;
//import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.twistorr.TwisTorr84.PumpStatus;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.subsystem.vacuum.constants.ConditionState;
import org.lsst.ccs.subsystem.vacuum.constants.DeviceState;
//import org.lsst.ccs.subsystem.vacuum.constants.ImageChannel;
import org.lsst.ccs.subsystem.vacuum.constants.SwitchEnable;
import org.lsst.ccs.subsystem.vacuum.constants.SwitchState;
import org.lsst.ccs.subsystem.vacuum.constants.VacuumAgentProperties;
import org.lsst.ccs.subsystem.vacuum.data.VacuumException;
import org.lsst.ccs.subsystem.vacuum.data.VacuumState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * The cryogenics pump plate test system
 *
 * @author The LSST CCS Team
 */
public class PumpPlate /*extends Subsystem*/ implements HasLifecycle {

    private static final double
        PRESS_20 = 20.0,
        PRESS_10 = 10.0,
        PRESS_5 = 5.0,
        PRESS_MID = 0.09,
        PRESS_ION_OFF = 1.0e-5,
        PRESS_ION_ENABLE = 1.0e-6,
        TURBO_MAX = 60000,
        TURBO_LOW = 0.1 * TURBO_MAX,
        TURBO_HIGH = 0.5 * TURBO_MAX;
    private static final int[] switchChannels = new int[VacuumState.NUM_SWITCHES];
    static {
        switchChannels[VacuumState.SW_CRYO_ION_PUMP1] = 0;
        switchChannels[VacuumState.SW_CRYO_ION_PUMP2] = 1;
        switchChannels[VacuumState.SW_CRYO_ION_PUMP3] = 2;
        switchChannels[VacuumState.SW_CRYO_ION_PUMP4] = 3;
        switchChannels[VacuumState.SW_CRYO_ION_PUMP5] = 4;
        switchChannels[VacuumState.SW_CRYO_ION_PUMP6] = 5;
        switchChannels[VacuumState.SW_HX_ION_PUMP1] = 6;
        switchChannels[VacuumState.SW_HX_ION_PUMP2] = 7;
        switchChannels[VacuumState.SW_OR_ION_PUMP] = 8;
        switchChannels[VacuumState.SW_CRYO_VALVE] = VacPlutoDevice.SW_OPEN_VCR00;
        switchChannels[VacuumState.SW_HX_VALVE] = VacPlutoDevice.SW_OPEN_VHX00;
        switchChannels[VacuumState.SW_OR_FPP_VALVE] = VacPlutoDevice.SW_OPEN_VCR01;  // Check!
        switchChannels[VacuumState.SW_OR_FH_VALVE] = VacPlutoDevice.SW_OPEN_VCR02;   // Check!
        switchChannels[VacuumState.SW_OR_L3H_VALVE] = VacPlutoDevice.SW_OPEN_VCR03;  // Check!
        switchChannels[VacuumState.SW_OR_L3_VALVE] = VacPlutoDevice.SW_OPEN_VCR04;   // Check!
    }
    private static final Map<String, DeviceState> turboStateMap = new HashMap<>();
    static {
        turboStateMap.put(PumpStatus.STOP.name(), DeviceState.STOPPED);
        turboStateMap.put(PumpStatus.WAIT_INTLK.name(), DeviceState.WAITING);
        turboStateMap.put(PumpStatus.STARTING.name(), DeviceState.STARTNG);
        turboStateMap.put(PumpStatus.NORMAL.name(), DeviceState.NORMAL);
        turboStateMap.put(PumpStatus.BRAKING.name(), DeviceState.BRAKING);
        turboStateMap.put(PumpStatus.FAIL.name(), DeviceState.FAILED);
        turboStateMap.put(PumpStatus.AUTO_TUNING.name(), DeviceState.AUTOTUN);
    }

    @LookupName
    private String name;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService pts;

    @LookupField( strategy=LookupField.Strategy.TOP)
    private Subsystem subsys;

    @LookupField(strategy=LookupField.Strategy.DESCENDANTS)
    private VacPlutoDevice plutoDevc;
    @LookupField(strategy=LookupField.Strategy.DESCENDANTS)
    private CryoTurboDevice cryoTurboDevc;
    @LookupField(strategy=LookupField.Strategy.DESCENDANTS)
    private HxTurboDevice hxTurboDevc;
    @LookupField(strategy=LookupField.Strategy.DESCENDANTS)
    private IonPumpDevice ionPumpDevc;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private final Map<String, Channel> channelMap = new LinkedHashMap<>();

    // From Groovy file
    private String platePressChan, turboPressChan, turboSpeedChan, forelinePressChan;
    private List<Integer> switches;

    // General
    private final Logger log = Logger.getLogger(getClass().getPackage().getName());
    private final VacuumState vacState = new VacuumState();
    private Set<Integer> switchSet;
    private final Device[] switchDevices = new Device[VacuumState.NUM_SWITCHES];
    private Channel platePressure, turboPressure, turboSpeed, forelinePressure;
    private boolean running = false;
    private final Map<String, Integer> switchNameMap = new LinkedHashMap<>();

    /*
    public PumpPlate() {
        super("pump-plate",AgentInfo.AgentType.WORKER);
    }
    */

    /**
     *  Build phase
     */
    @Override
    public void build() {
        //setAgentProperty("org.lsst.ccs.use.full.paths", "true");
        //Create and schedule an AgentPeriodicTask to update the vacuum state
        AgentPeriodicTask pt;
        pt = new AgentPeriodicTask("vacuum-state",
                                   () -> updateVacuumState()).withPeriod(Duration.ofMillis(1000));
        pts.scheduleAgentPeriodicTask(pt);
        
    }


    /**
     *  Post initialization
     */
    @Override
    public void postInit() {
        //Set a property to define that this Agent is a vacuum subsystem.
        subsys.setAgentProperty(VacuumAgentProperties.VACUUM_TYPE, PumpPlate.class.getCanonicalName());

        if (switches != null) {
            switchSet = new HashSet<>(switches);
        }
        else {
            MonitorLogUtils.reportConfigError(log, name, "Switch list", "not specified");
        }
        if (plutoDevc != null) {
            for (int cond : plutoDevc.getConditionIds()) {
                vacState.addCondition(cond);
                vacState.setCondition(cond, ConditionState.CLEAR);
            }
        }
        else {
            MonitorLogUtils.reportConfigError(log, name, "Pluto device", "not specified");
        }
        if (cryoTurboDevc == null) {
            MonitorLogUtils.reportConfigError(log, name, "Cryo turbo pump device", "not specified");
        }
        /*
        if (hxTurboDevc == null) {
            MonitorLogUtils.reportConfigError(log, name, "HX turbo pump device", "not specified");
        }
        */
        if (ionPumpDevc == null) {
            MonitorLogUtils.reportConfigError(log, name, "Ion pump device", "not specified");
        }
        if (forelinePressChan != null) {
            forelinePressure = channelMap.get(forelinePressChan);
        }
        if (forelinePressure == null) {
            MonitorLogUtils.reportConfigError(log, name, "Foreline pressure channel", "not specified or not defined");
        }
        if (platePressChan != null) {
            platePressure = channelMap.get(platePressChan);
        }
        if (platePressure == null) {
            MonitorLogUtils.reportConfigError(log, name, "Plate pressure channel", "not specified or not defined");
        }
        if (turboPressChan != null) {
            turboPressure = channelMap.get(turboPressChan);
        }
        if (turboPressure == null) {
            MonitorLogUtils.reportConfigError(log, name, "Turbo pump pressure channel", "not specified or not defined");
        }
        if (turboSpeedChan != null) {
            turboSpeed = channelMap.get(turboSpeedChan);
        }
        if (turboSpeed == null) {
            MonitorLogUtils.reportConfigError(log, name, "Turbo pump speed channel", "not specified or not defined");
        }

        List<Integer> ipChannels = ionPumpDevc.getChannelNumbers();
        for (int sw : switchSet) {
            switch (sw) {
            case VacuumState.SW_CRYO_TURBO:
                switchDevices[sw] = cryoTurboDevc;
                break;
            case VacuumState.SW_HX_TURBO:
                switchDevices[sw] = hxTurboDevc;
                break;
            case VacuumState.SW_CRYO_VALVE:
            case VacuumState.SW_HX_VALVE:
            case VacuumState.SW_OR_FPP_VALVE:
            case VacuumState.SW_OR_FH_VALVE:
            case VacuumState.SW_OR_L3H_VALVE:
            case VacuumState.SW_OR_L3_VALVE:
                switchDevices[sw] = plutoDevc;
                break;
            case VacuumState.SW_CRYO_ION_PUMP1:
            case VacuumState.SW_CRYO_ION_PUMP2:
            case VacuumState.SW_CRYO_ION_PUMP3:
            case VacuumState.SW_CRYO_ION_PUMP4:
            case VacuumState.SW_CRYO_ION_PUMP5:
            case VacuumState.SW_CRYO_ION_PUMP6:
            case VacuumState.SW_HX_ION_PUMP1:
            case VacuumState.SW_HX_ION_PUMP2:
            case VacuumState.SW_OR_ION_PUMP:
                switchDevices[sw] = ionPumpDevc;
                if (!ipChannels.contains(switchChannels[sw])) {
                    switchSet.remove(sw);
                }
                break;
            }
        }

        for (int sw : switchSet) {
            vacState.addSwitch(sw);
            vacState.setSwitchState(sw, SwitchState.OFFLINE);
            vacState.setSwitchEnable(sw, SwitchEnable.OFF);
        }

        for (String swName : SwitchNames.NAME_MAP.keySet()) {
            int sw = SwitchNames.NAME_MAP.get(swName);
            if (switchSet.contains(sw)) {
                switchNameMap.put(swName, sw);
            }
        }
    }


    /**
     *  Post start
     */
    @Override
    public void postStart() {
        log.info("Pump plate subsystem started");
        running = true;
    }


    /**
     *  Gets the state of the PumpPlate module.
     *
     *  @return  The vacuum state
     */
    @Command(type=Command.CommandType.QUERY, description="Get the pump plate state")
    public VacuumState getVacuumState() {
        vacState.setTickMillis(getTickPeriod());
        return vacState;
    }    


    /**
     *  Gets the list of switch names.
     *
     *  @return  The switch names.
     *  @throws  VacuumException
     */
    @Command(type=Command.CommandType.QUERY, description="Get switch names")
    public List<String> getSwitchNames() throws VacuumException
    {
        return new ArrayList(switchNameMap.keySet());
    }


    /**
     *  Turns a switch on or off.
     *
     *  @param  sw  The switch number.
     *  @param  on  Whether to turn on or off
     *  @throws  VacuumException
     */
    @Command(type=Command.CommandType.ACTION, description="Turn on/off a switch")
    public void setSwitchOn(@Argument(description="The switch number") int sw,
                            @Argument(description="Whether to turn on") boolean on) throws VacuumException
    {
        try {
            if (!switchSet.contains(sw)) {
                throw new VacuumException("Invalid switch number: " + sw);
            }
            setSwitch(sw, on);
        }
        finally {
            publishState();
        }
        
    }


    /**
     *  Turns a named switch on or off.
     *
     *  @param  name  The switch name.
     *  @param  on    Whether to turn on or off
     *  @throws  VacuumException
     */
    @Command(type=Command.CommandType.ACTION, description="Turn on/off a named switch")
    public void setNamedSwitchOn(@Argument(description="The switch name") String name,
                                 @Argument(description="Whether to turn on") boolean on) throws VacuumException
    {
        Integer sw = switchNameMap.get(name);
        try {
            if (sw == null) {
                throw new VacuumException("Invalid switch name: " + name);
            }
            setSwitch(sw, on);
        }
        finally {
            publishState();
        }
    }


    /**
     *  Turns a switch on or off.
     *
     *  @param  sw  The switch number.
     *  @param  on  Whether to turn on or off
     *  @throws  VacuumException
     */
    private void setSwitch(int sw, boolean on) throws VacuumException
    {
        SwitchState state = vacState.getSwitchState(sw);
        if (state == SwitchState.OFFLINE) return;
        SwitchEnable enable = vacState.getSwitchEnable(sw);
        if (on && enable != SwitchEnable.ON && enable != SwitchEnable.WAS_ON) return;
        Device swDevice = switchDevices[sw];
        try {
            if (swDevice instanceof TwisTorr84Device) {
                if (on) {
                    ((TwisTorr84Device)swDevice).startTurboPump();
                }
                else {
                    ((TwisTorr84Device)swDevice).stopTurboPump();
                }
            }
            else if (swDevice == plutoDevc) {
                plutoDevc.setSwitchOn(switchChannels[sw], on);
            }
            else if (swDevice == ionPumpDevc) {
                ionPumpDevc.setChannelOn(switchChannels[sw], on);
            }
        }
        catch (DriverException e) {
            throw new VacuumException(e);
        }
    }


    /**
     *  Gets whether a switch is on.
     *
     *  @param  sw  The switch number.
     *  @return  Whether the switch is on
     *  @throws  VacuumException
     */
    private Boolean isSwitchOn(int sw)
    {
        Boolean value = false;
        Device swDevice = switchDevices[sw];
        if (swDevice instanceof TwisTorr84Device) {
            try {
                DeviceState st = turboStateMap.get(((TwisTorr84Device)swDevice).readTurboStatus());
                value = st != DeviceState.STOPPED && st != DeviceState.BRAKING;
            }
            catch (DriverException e) {
                value = null;
            }
        }
        else if (swDevice == plutoDevc) {
            value = plutoDevc.isSwitchOn(switchChannels[sw]);
        }
        else if (swDevice == ionPumpDevc) {
            value = ionPumpDevc.isChannelOn(switchChannels[sw]);
        }
        return value;
    }


    /**
     *  Clears a condition.
     *
     *  @param  cond  The condition number.
     *  @throws  VacuumException
     */
    @Command(type=Command.CommandType.ACTION, description="Clear a condition")
    public void clearCondition(@Argument(description="The condition number") int cond) throws VacuumException
    {
        try {
            plutoDevc.clearCondition(cond);
        }
        finally {
            publishState();
        }
    }


    /**
     *  Sets the update period.
     *
     *  @param  value  The update period (milliseconds) to set.
     */
    @Command(type=Command.CommandType.ACTION, description="Set the update interval")
    public void setUpdatePeriod(@Argument(description="The tick period (ms)") int value)
    {
        setTickPeriod(value);
        vacState.setTickMillis(getTickPeriod());
        publishState();
    }


    /**
     *  Updates the vacuum state periodically.
     *
     *  The vacuum state consists mainly of the state of the switches (lines) being
     *  controlled, along with whether they can be turned on.
     *
     *  Pressures and pump speeds are read to determine which switches are enabled
     *  and/or need to be turned off.  If the state changes it is published.
     */
    private void updateVacuumState()
    {
        if (!running) return;
        double forelinePr = forelinePressure.getValue();
        double platePr = platePressure.getValue();
        double turboPr = turboPressure.getValue();
        double turboSp = turboSpeed.getValue();
        boolean changed = false;
        for (int sw = 0; sw < VacuumState.NUM_SWITCHES; sw++) {
            if (!vacState.hasSwitch(sw)) continue;
            Boolean enable = false;
            boolean turnOff = false;
            DeviceState devState = null;
            switch (sw) {
 
            case VacuumState.SW_CRYO_VALVE:
                if (!Double.isNaN(platePr) && !Double.isNaN(turboPr) && !Double.isNaN(turboSp)) {
                    double prDiff = Math.abs(turboPr - platePr);
                    enable = (prDiff <= PRESS_20 && turboSp < TURBO_LOW)
                                || (prDiff <= PRESS_MID && turboSp > TURBO_HIGH);
                }
                if (enable && !Double.isNaN(forelinePr) && !Double.isNaN(turboSp)) {
                    enable = forelinePr < PRESS_5 || turboSp < TURBO_HIGH;
                    turnOff = !enable;
                }
                break;

            case VacuumState.SW_CRYO_TURBO:
                if (!Double.isNaN(turboPr)) {
                    enable = turboPr < PRESS_10;
                    turnOff = !enable;
                }
                if (enable && !Double.isNaN(forelinePr) && !Double.isNaN(turboSp)) {
                    enable = forelinePr < PRESS_5 || turboSp < TURBO_HIGH;
                    turnOff = !enable;
                }
                try {
                    devState = turboStateMap.get(((CryoTurboDevice)switchDevices[sw]).readTurboStatus());
                }
                catch (DriverException e) {}
                break;

            case VacuumState.SW_CRYO_ION_PUMP1:
            case VacuumState.SW_CRYO_ION_PUMP2:
            case VacuumState.SW_CRYO_ION_PUMP3:
            case VacuumState.SW_CRYO_ION_PUMP4:
            case VacuumState.SW_CRYO_ION_PUMP5:
            case VacuumState.SW_CRYO_ION_PUMP6:
                if (!Double.isNaN(platePr)) {
                    enable = platePr < PRESS_ION_ENABLE;
                    turnOff = platePr >= PRESS_ION_OFF;
                }
                break;
            }

            SwitchState oldState = vacState.getSwitchState(sw);
            if (turnOff && oldState == SwitchState.ON) {
                try {
                    setSwitch(sw, false);
                }
                catch (VacuumException e) {
                    log.error("Error setting switch: " + e);
                }
            }
            Boolean isOn = isSwitchOn(sw);
            SwitchState state = isOn != null ? isOn ? SwitchState.ON : SwitchState.OFF : SwitchState.OFFLINE;
            SwitchEnable oldEnabled = vacState.getSwitchEnable(sw);
            SwitchEnable enabled;
            if (enable == null) {
                enabled = oldEnabled == SwitchEnable.OFF ? SwitchEnable.WAS_OFF
                            : oldEnabled == SwitchEnable.ON ? SwitchEnable.WAS_ON : oldEnabled;
            }
            else {
                enabled = enable ? SwitchEnable.ON : SwitchEnable.OFF;
            }
            if (state != oldState || enabled != oldEnabled) {
                vacState.setSwitchState(sw, state);
                vacState.setSwitchEnable(sw, enabled);
                changed = true;
            }
            DeviceState oldDevState = vacState.getDeviceState(sw);
            vacState.setDeviceState(sw, devState);
            if (devState != oldDevState) {
                changed = true;
            }
        }
        for (int cond = 0; cond < VacuumState.NUM_CONDITIONS; cond++) {
            if (!vacState.hasCondition(cond)) continue;
            Boolean active = plutoDevc.isConditionActive(cond);
            Boolean latched = plutoDevc.isConditionLatched(cond);
            ConditionState state = active == null || latched == null ? ConditionState.OFFLINE :
                                   latched ? ConditionState.LATCHED :
                                   active ? ConditionState.ACTIVE : ConditionState.CLEAR;
            if (state != vacState.getCondition(cond)) {
                vacState.setCondition(cond, state);
                changed = true;
            }
        }
        if (changed) {
            publishState();
        }
    }


    /**
     *  Publishes the state of the PumpPlate module.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    private void publishState()
    {
        subsys.publishSubsystemDataOnStatusBus(new KeyValueData(VacuumState.KEY, getVacuumState()));
    }


    /**
     *  Sets the monitoring publishing period.
     */
    private void setTickPeriod(long period)
    {
        pts.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }
    

    /**
     *  Gets the monitoring publishing period.
     */
    private int getTickPeriod()
    {
        return (int)pts.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }

}
