package org.lsst.ccs.subsystem.vacuum.ui;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Dimension;
import java.awt.Insets;
import java.awt.Font;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;

import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;

import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import org.lsst.ccs.subsystem.common.ui.TextFieldX;

import java.util.logging.Level;
import java.util.logging.Logger;

import org.lsst.ccs.subsystem.common.ui.SystemStatusPanel;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.common.ui.UiConstants;
import org.lsst.ccs.subsystem.common.ui.UiUtilities;
import org.lsst.ccs.subsystem.vacuum.constants.MonitorControl;
import org.lsst.ccs.subsystem.vacuum.constants.SwitchEnable;
import org.lsst.ccs.subsystem.vacuum.constants.Switches;
import org.lsst.ccs.subsystem.vacuum.constants.SwitchState;
import org.lsst.ccs.subsystem.vacuum.constants.DeviceState;

import org.lsst.ccs.subsystem.vacuum.data.VacSysState;

/**
 *  Implements the utility monitoring panel.
 *
 *  @author Owen Saxton
 */
public class PumpCartControlPanel extends JPanel implements UiUtilities.ActionHandler, CommandSender.ReplyHandler {

    static final Font FONT = new java.awt.Font("Tahoma", 1, 12);

    private static final String CMND_GET_STATE = "getVacuumState";
    private static final int SWTP_PUMP = 0, SWTP_VALVE = 1;
    private static final String DSAB_TEXT = "DSAB";


    //    private static final Map<Integer, String> switchMap = new LinkedHashMap<>();  // Ordered by panel order
    private static final Map<DeviceState, Color> deviceStateColors = new HashMap<>();
    static {
        deviceStateColors.put(DeviceState.AUTOTUN, Color.BLACK);
        deviceStateColors.put(DeviceState.BRAKING, UiConstants.YELLOW);
        deviceStateColors.put(DeviceState.FAILED, UiConstants.RED);
        deviceStateColors.put(DeviceState.NORMAL, UiConstants.GREEN);
        deviceStateColors.put(DeviceState.STARTNG, UiConstants.YELLOW);
        deviceStateColors.put(DeviceState.STOPPED, Color.BLACK);
        deviceStateColors.put(DeviceState.WAITING, UiConstants.YELLOW);
        deviceStateColors.put(DeviceState.OPEN, UiConstants.GREEN);
        deviceStateColors.put(DeviceState.SHUT, Color.BLACK);
        deviceStateColors.put(DeviceState.TRANSIT, UiConstants.YELLOW);
        deviceStateColors.put(DeviceState.ERROR, UiConstants.RED);
        deviceStateColors.put(DeviceState.OFFLINE, UiConstants.BLUE);
    }
    //    private static final int
    //        COND_STATUS_WIDTH = UiUtilities.maxEnumLabelWidth(ConditionState.class);

    private final CommandSender sender;
    private final UiUtilities uiUtils;

    private SystemStatusPanel sysStatPanel;

    private final JPanel headPanel = new JPanel();

    private static final Logger LOG = Logger.getLogger(PumpCartControlPanel.class.getName());

    //    private final JRadioButton[] switchOnRB = new JRadioButton[Switches.NUM_SWITCHES];
    private JRadioButton pumpOn = new JRadioButton();
    private JRadioButton pumpOff = new JRadioButton();

    private JRadioButton ventOn = new JRadioButton();
    private JRadioButton ventOff = new JRadioButton();

    private int pulseVentTime = 1; // default is 1 sec of venting
    private JButton pulseVent = new JButton();

    private TextFieldX cryoPrTextField;
    private TextFieldX cryoTurboPrTextField;
    private TextFieldX cryoFlinePrTextField;
    private TextFieldX hxPrTextField;
    private TextFieldX hxTurboPrTextField;
    private TextFieldX hxFlinePrTextField;
    private TextFieldX ventingTextField;
    private TextFieldX cyclingTextField;

    private TextFieldX pulseVentPeriodTextField;
    private final JLabel pulseVentPeriodLabel = new JLabel("Pulse Period: ");
    private final JLabel pulseVentPeriodUnitsLabel = new JLabel("        sec");

    private int pulsePumpTime = 1; // default is 1 sec of venting
    private JButton pulsePump = new JButton();
    private TextFieldX pulsePumpPeriodTextField;
    private final JLabel pulsePumpPeriodLabel = new JLabel("Pulse Period: ");
    private final JLabel pulsePumpPeriodUnitsLabel = new JLabel("       sec");

    private JPanel pumpCartPanel;


    private volatile VacSysState vacState;
    private final Object vacStateLock = new Object();

    public PumpCartControlPanel(CommandSender cons) {
        uiUtils = new UiUtilities(this);
	sender = cons;
        initComponents();
        (new DisablePanel()).run();
    }

    public void initPanel() {
        sender.sendCommand(true, null, CMND_GET_STATE);
    }

    @Override
    public void onCommandReply(Object reply, String path, String command, Object[] args) {
	//	System.out.println("onCommandReply = " + reply);
	updatePanel((VacSysState)reply);
    }

    @Override
    public void onCommandReject(String path, String command, Object[] args) {
	//	System.out.println("onCommandReject of command " + command);
        if (!command.equals(CMND_GET_STATE)) {
	    updatePanel(vacState);
        }
    }
    
    public void updatePanel(VacSysState vss) {
        synchronized(vacStateLock) {
            vacState = vss;
            SwingUtilities.invokeLater(new UpdateVacState(vss));
        }
    }
    
    public void disablePanel() {
        SwingUtilities.invokeLater(new DisablePanel());
    }

    private void initComponents() {

        // System status line
        sysStatPanel = new SystemStatusPanel(sender, MonitorControl.NODE_NAME, true);
	/* example
            switchOffRB[sw] = uiUtils.newRadioButton(swType == SWTP_PUMP ? "Off" : "Shut", "S" + sw);
            switchOnRB[sw] = uiUtils.newRadioButton(swType == SWTP_PUMP ? "On" : "Open", "O" + sw, confirmDialog);
            ButtonGroup bg = switchBG[sw] = new ButtonGroup();
            bg.add(switchOffRB[sw]);
            bg.add(switchOnRB[sw]);
	*/


	pumpOn = uiUtils.newRadioButton("On","PON",true);
	pumpOff = uiUtils.newRadioButton("Off","POFF");
	ButtonGroup pgroup = new ButtonGroup();
      
	pgroup.add(pumpOn);
	pgroup.add(pumpOff);

	ventOn = uiUtils.newRadioButton("On","VON",true);
	ventOff = uiUtils.newRadioButton("Off","VOFF");
	ButtonGroup vgroup = new ButtonGroup();
      
	vgroup.add(ventOn);
	vgroup.add(ventOff);

        pulseVent = uiUtils.newButton("Pulse Venting", "pulseVent", true);

        pulseVentPeriodLabel.setFont(FONT);

	//	String cryoFlineVac = 
	//	sender.sendCommand(null, "getChannelValue","Cryo/ForelineVac");
        cryoPrTextField = uiUtils.newTextFieldX("---------", "CPTF", TextFieldX.TYPE_TEXT,true);
        cryoTurboPrTextField = uiUtils.newTextFieldX("---------", "CTPTF", TextFieldX.TYPE_TEXT,true);
        cryoFlinePrTextField = uiUtils.newTextFieldX("---------", "CFPF", TextFieldX.TYPE_TEXT,true);
        hxPrTextField = uiUtils.newTextFieldX("---------", "CPTF", TextFieldX.TYPE_TEXT,true);
        hxTurboPrTextField = uiUtils.newTextFieldX("---------", "CTPTF", TextFieldX.TYPE_TEXT,true);
        hxFlinePrTextField = uiUtils.newTextFieldX("---------", "CFPF", TextFieldX.TYPE_TEXT,true);
        ventingTextField = uiUtils.newTextFieldX("-------", "VTF", TextFieldX.TYPE_TEXT,true);
        cyclingTextField = uiUtils.newTextFieldX("-------", "CTF", TextFieldX.TYPE_TEXT,true);


        pulseVentPeriodTextField = uiUtils.newTextFieldX(String.format("%3d",pulseVentTime), "V", TextFieldX.TYPE_TEXT,true);

        pulsePump = uiUtils.newButton("Pulse Pumping", "pulsePump", true);

        pulsePumpPeriodLabel.setFont(FONT);

        pulsePumpPeriodTextField = uiUtils.newTextFieldX(String.format("%3d",pulsePumpTime), "P", TextFieldX.TYPE_TEXT,true);

	//	Dimension d = pulseVentPeriodTextField.getPreferredSize();
	//        d.width = 45;
	//        pulseVentPeriodTextField.setMinimumSize(d);
	//        pulseVentPeriodTextField.setPreferredSize(d);
	//        pulseVentPeriodTextField.setHorizontalAlignment(SwingConstants.CENTER);
	//        pulseVentPeriodTextField.setText(String.valueOf(pulseVentTime));
	/*
        pulseVentPeriodTextField.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setPulsePeriod();
            }
	    });
	*/
        pulsePumpPeriodUnitsLabel.setFont(FONT);

        // Vacuum status line (panel)
        headPanel.setLayout(new GridBagLayout());

	/* ---------------------------------
vacuum-sim ccs>vacuum-sim/P
vacuum-sim/PumpCartTemperature    vacuum-sim/PumpCartCycling        vacuum-sim/PDU                    
vacuum-sim/PumpCart               vacuum-sim/PumpCartPressure       
vacuum-sim ccs>vacuum-sim/PumpCart 
change                                   submitChange                             submitChanges                            
cancelSubmittedChanges                   getSubmittedChanges                      printComponentConfigurationParameters    
applySubmittedChangesForComponent        clearWarnings                            showVersion                              
showStatus                               showDecodedStatus                        readInletPressure                        
readTemperature                          showPrimaryHours                         showHighVacHours                         
showWarningCodes                         showFaultCodes                           clearFaults                              
pumpOn                                   getPumpStatus                            openVentValve                            
getVentValveStatus                       getVentValveParms                        setVentValveParms                        
setVentValveNominal                      writeCommand                             
            -------------------------------- */




        // Pump Cart panel
        pumpCartPanel = UiUtilities.newBorderedPanel("Pump Cart");
        GridBagConstraints gbpc = new GridBagConstraints();
        gbpc.anchor = GridBagConstraints.WEST;
        gbpc.insets = new Insets(0, 0, 4, 4);
        gbpc.gridx = 0;
        gbpc.gridy = 0;
        gbpc.anchor = GridBagConstraints.WEST;
        gbpc.insets.left = 5;

	//	pumpCartPanel.add(sysStatPanel);
	//	gbpc.gridy++;
        pumpCartPanel.add(UiUtilities.newLabel("Cryo P: ", 0), gbpc);
        gbpc.gridx++;
        pumpCartPanel.add(cryoPrTextField, gbpc);
        gbpc.gridx++;
        pumpCartPanel.add(UiUtilities.newLabel("Cryo Turbo P: ", 0), gbpc);
        gbpc.gridx++;
        pumpCartPanel.add(cryoTurboPrTextField, gbpc);
        gbpc.gridx++;
        pumpCartPanel.add(UiUtilities.newLabel("Cryo FLine P: ", 0), gbpc);
        gbpc.gridx++;
        pumpCartPanel.add(cryoFlinePrTextField, gbpc);

        gbpc.gridy++;
        gbpc.gridx=0;
        pumpCartPanel.add(UiUtilities.newLabel("Hx P: ", 0), gbpc);
        gbpc.gridx++;
        pumpCartPanel.add(hxPrTextField, gbpc);
        gbpc.gridx++;
        pumpCartPanel.add(UiUtilities.newLabel("Hx Turbo P: ", 0), gbpc);
        gbpc.gridx++;
        pumpCartPanel.add(hxTurboPrTextField, gbpc);
        gbpc.gridx++;
        pumpCartPanel.add(UiUtilities.newLabel("Hx FLine P: ", 0), gbpc);
        gbpc.gridx++;
        pumpCartPanel.add(hxFlinePrTextField, gbpc);

        gbpc.gridy++;
        gbpc.gridx=0;
        pumpCartPanel.add(UiUtilities.newLabel("Venting State: ", 0), gbpc);
        gbpc.gridx++;
        pumpCartPanel.add(ventingTextField, gbpc);
        gbpc.gridx++;
        pumpCartPanel.add(UiUtilities.newLabel("Cycling State: ", 0), gbpc);
        gbpc.gridx++;
        pumpCartPanel.add(cyclingTextField, gbpc);
        gbpc.gridx = 0;
        gbpc.gridy+=2;


        pumpCartPanel.add(UiUtilities.newLabel("Pump Cycling: ", 0), gbpc);
        gbpc.gridx++;
        pumpCartPanel.add(pumpOn, gbpc);
        gbpc.gridx++;
        pumpCartPanel.add(pumpOff,gbpc);

        gbpc.gridx -= 2;
        gbpc.gridy++;

        pumpCartPanel.add(UiUtilities.newLabel("Venting: ", 0), gbpc);
        gbpc.gridx++;
        pumpCartPanel.add(ventOn, gbpc);
        gbpc.gridx++;
        pumpCartPanel.add(ventOff,gbpc);

        gbpc.gridx -= 2;
        gbpc.gridy++;

        pumpCartPanel.add(UiUtilities.newLabel("Pulse vent: ", 0), gbpc);
        gbpc.gridx++;
        pumpCartPanel.add(pulseVentPeriodTextField, gbpc);
	//        gbpc.gridx++;
        pumpCartPanel.add(pulseVentPeriodUnitsLabel, gbpc);
        gbpc.gridx++;
        pumpCartPanel.add(pulseVent,gbpc);

        gbpc.gridx -= 2;
        gbpc.gridy++;

        pumpCartPanel.add(UiUtilities.newLabel("Pulse pump: ", 0), gbpc);
        gbpc.gridx++;
        pumpCartPanel.add(pulsePumpPeriodTextField, gbpc);
	//        gbpc.gridx++;
        pumpCartPanel.add(pulsePumpPeriodUnitsLabel, gbpc);
        gbpc.gridx++;
        pumpCartPanel.add(pulsePump,gbpc);

        // Lay out the complete panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbm = new GridBagConstraints();
        gbm.insets = new Insets(6, 0, 6, 0);
        gbm.anchor = GridBagConstraints.NORTH;
        gbm.gridx = 0;
        gbm.gridy = 0;
        add(pumpCartPanel, gbm);
    }


    @Override
    public void handleRadioButton(String name) {
	LOG.severe("button name = "+name);
	System.out.println("button name = "+name);
	if (name.charAt(0) == 'P') {
	    if (name.contains("OFF")) {
		sender.sendCommand("PumpCart", "pumpOn",false);
	    } else {
		if (pumpOn.isEnabled()) {
		    sender.sendCommand("PumpCart", "pumpOn",true);
		} else {
		    pumpOn.setEnabled(true);
		}
	    }
	} else if (name.charAt(0) == 'V') {
	    if (name.contains("OFF")) {
		sender.sendCommand("PumpCart", "openVentValve",false);
	    } else {
		if (ventOn.isEnabled()) {
		    sender.sendCommand("PumpCart", "openVentValve",true);
		} else {
		    ventOn.setEnabled(true);
		}
	    }
	}
    }


    @Override
    public void handleButton(String name) {
	LOG.severe("button name = "+name);
	System.out.println("button name = "+name);
	if (name.contains("pulseVent")) {
	    sender.sendCommand("PumpCart", "pulseVentValve",pulseVentTime);
	} else if (name.contains("pulsePump")) {
	    sender.sendCommand("PumpCart", "pulsePumpValve",pulsePumpTime);
	}

    }
    
    class UpdateVacState implements Runnable {

        private final VacSysState vs;

        UpdateVacState(VacSysState vs) {
            this.vs = vs;
        }

        @Override
        public void run() {
	    sysStatPanel.updatePanel(vs.getTickMillis());
	    cryoPrTextField.update(String.format("%5.1f",vs.getCryoPress()),true); 
	    cryoTurboPrTextField.update(String.format("%5.1f",vs.getCryoTurboPress()),true); 
	    cryoFlinePrTextField.update(String.format("%5.1f",vs.getCryoFlinePress()),true); 
	    hxPrTextField.update(String.format("%5.1f",vs.getHXPress()),true); 
	    hxTurboPrTextField.update(String.format("%5.1f",vs.getHXTurboPress()),true); 
	    hxFlinePrTextField.update(String.format("%5.1f",vs.getHXFlinePress()),true); 

	    ventingTextField.update(vs.getVentingStatus()==0 ? "Off" : "Active" ,true); 
	    cyclingTextField.update(vs.getCyclingStatus()==0 ? "Off" : "Active",true); 

        }

    }
    



   @Override
   public void handleTextFieldX(String name, Object value) {

       if (name.contains("V")) {
	    //	    pulseVentTime = (Integer)value;
	   try {
	       pulseVentTime = Integer.decode((String)value);
	   } catch(NumberFormatException nfe) {
	       pulseVentTime = 0;
	   }

	   pulseVentPeriodTextField.update(String.format("%d",pulseVentTime),true); 
       } else  if (name.contains("P")) {
	    //	    pulsePumpTime = (Integer)value;
	   try {
	       pulsePumpTime = Integer.decode((String)value);
	   } catch(NumberFormatException nfe) {
	       pulsePumpTime = 0;
	   }

	   pulsePumpPeriodTextField.update(String.format("%d",pulsePumpTime),true); 
       }
   }

    class DisablePanel implements Runnable {

        @Override
        public void run() {
	    //            blah.setEnabled(false);
	}
    }
    private static final long serialVersionUID = 1L;
}
