package org.lsst.ccs.subsystem.vacuum.ui;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.util.LinkedHashMap;
import java.util.Map;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.SwingUtilities;
import org.lsst.ccs.subsystem.common.ui.SystemStatusPanel;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.common.ui.UiConstants;
import org.lsst.ccs.subsystem.common.ui.UiUtilities;
import org.lsst.ccs.subsystem.vacuum.constants.ConditionState;
import org.lsst.ccs.subsystem.vacuum.constants.Conditions;
import org.lsst.ccs.subsystem.vacuum.constants.CryoVacuumState;
import org.lsst.ccs.subsystem.vacuum.constants.DeviceState;
import org.lsst.ccs.subsystem.vacuum.constants.HxVacuumState;
import org.lsst.ccs.subsystem.vacuum.constants.LatchState;
import org.lsst.ccs.subsystem.vacuum.constants.Latches;
import org.lsst.ccs.subsystem.vacuum.constants.MonitorControl;
import org.lsst.ccs.subsystem.vacuum.constants.PLCState;
import org.lsst.ccs.subsystem.vacuum.constants.SwitchEnable;
import org.lsst.ccs.subsystem.vacuum.constants.Switches;
import org.lsst.ccs.subsystem.vacuum.constants.SwitchState;
import org.lsst.ccs.subsystem.vacuum.data.VacSysState;

/**
 *  Implements the utility monitoring panel.
 *
 *  @author Owen Saxton
 */
public class VacControlPanel extends JPanel implements UiUtilities.ActionHandler, CommandSender.ReplyHandler {

    private static final int SWTP_PUMP = 0, SWTP_VALVE = 1;
    private static final Map<Integer, String> switchMap = new LinkedHashMap<>();  // Ordered by panel order
    static {
        switchMap.put(Switches.SW_CRYO_TURBO_PUMP,  "Cryo Turbo Pump");
        switchMap.put(Switches.SW_CRYO_GATE_VALVE,  "Cryo Gate Valve");
        switchMap.put(Switches.SW_CRYO_SCROLL_PUMP, "Cryo Scroll Pump");
        switchMap.put(Switches.SW_CRYO_ION_PUMP1,   "Cryo Ion Pump 1");
        switchMap.put(Switches.SW_CRYO_ION_PUMP2,   "Cryo Ion Pump 2");
        switchMap.put(Switches.SW_CRYO_ION_PUMP3,   "Cryo Ion Pump 3");
        switchMap.put(Switches.SW_CRYO_ION_PUMP4,   "Cryo Ion Pump 4");
        switchMap.put(Switches.SW_CRYO_ION_PUMP5,   "Cryo Ion Pump 5");
        switchMap.put(Switches.SW_CRYO_ION_PUMP6,   "Cryo Ion Pump 6");
        switchMap.put(Switches.SW_INST_SCROLL_PUMP, "Inst Scroll Pump");
        switchMap.put(Switches.SW_HX_TURBO_PUMP,    "HX Turbo Pump");
        switchMap.put(Switches.SW_HX_GATE_VALVE,    "HX Gate Valve");
        switchMap.put(Switches.SW_HX_SCROLL_PUMP,   "HX Scroll Pump");
        switchMap.put(Switches.SW_HX_ION_PUMP1,     "HX Ion Pump 1");
        switchMap.put(Switches.SW_HX_ION_PUMP2,     "HX Ion Pump 2");
        switchMap.put(Switches.SW_INST_FTPP_VALVE,  "FTPP O-ring Valve");
        switchMap.put(Switches.SW_INST_L3H_VALVE,   "L3H O-ring Valve");
        switchMap.put(Switches.SW_INST_FTH_VALVE,   "FTH O-ring Valve");
        switchMap.put(Switches.SW_INST_L3LF_VALVE,  "L3LF O-ring Valve");
    }
    private static final int[] switchTypes = new int[Switches.NUM_SWITCHES];
    static {
        switchTypes[Switches.SW_CRYO_TURBO_PUMP]  = SWTP_PUMP;
        switchTypes[Switches.SW_CRYO_ION_PUMP1]   = SWTP_PUMP;
        switchTypes[Switches.SW_CRYO_ION_PUMP2]   = SWTP_PUMP;
        switchTypes[Switches.SW_CRYO_ION_PUMP3]   = SWTP_PUMP;
        switchTypes[Switches.SW_CRYO_ION_PUMP4]   = SWTP_PUMP;
        switchTypes[Switches.SW_CRYO_ION_PUMP5]   = SWTP_PUMP;
        switchTypes[Switches.SW_CRYO_ION_PUMP6]   = SWTP_PUMP;
        switchTypes[Switches.SW_CRYO_SCROLL_PUMP] = SWTP_PUMP;
        switchTypes[Switches.SW_CRYO_GATE_VALVE]  = SWTP_VALVE;
        switchTypes[Switches.SW_HX_TURBO_PUMP]    = SWTP_PUMP;
        switchTypes[Switches.SW_HX_ION_PUMP1]     = SWTP_PUMP;
        switchTypes[Switches.SW_HX_ION_PUMP2]     = SWTP_PUMP;
        switchTypes[Switches.SW_HX_GATE_VALVE]    = SWTP_VALVE;
        switchTypes[Switches.SW_HX_SCROLL_PUMP]   = SWTP_PUMP;
        switchTypes[Switches.SW_INST_FTPP_VALVE]  = SWTP_VALVE;
        switchTypes[Switches.SW_INST_FTH_VALVE]   = SWTP_VALVE;
        switchTypes[Switches.SW_INST_L3H_VALVE]   = SWTP_VALVE;
        switchTypes[Switches.SW_INST_L3LF_VALVE]  = SWTP_VALVE;
        switchTypes[Switches.SW_INST_SCROLL_PUMP] = SWTP_PUMP;
    }
    private static final Map<Integer, String> latchMap = new LinkedHashMap<>();  // Ordered by panel order
    static {
        latchMap.put(Latches.LATCH_CR_VACUUM,   "Cryo Vacuum Bad");
        latchMap.put(Latches.LATCH_CR_GATE_AO,  "Cryo Gate Can't Open");
        latchMap.put(Latches.LATCH_CR_GATE_NFC, "Cryo Gate Forced Shut");
        latchMap.put(Latches.LATCH_CR_PUMP,     "Cryo Turbo Pump Bad");
        latchMap.put(Latches.LATCH_HX_VACUUM,   "HX Vacuum Bad");
        latchMap.put(Latches.LATCH_HX_GATE_AO,  "HX Gate Can't Open");
        latchMap.put(Latches.LATCH_HX_GATE_NFC, "HX Gate Forced Shut");
        latchMap.put(Latches.LATCH_HX_PUMP,     "HX Turbo Pump Bad");
    }
    private static final Map<Integer, String> conditionMap = new LinkedHashMap<>();  // Ordered by panel order
    static {
        conditionMap.put(Conditions.COND_CR_FORELINE_VAC, "Cryo Foreline OK");
        conditionMap.put(Conditions.COND_CR_TRB_PRESS_10, "Cryo Turbo Vac < 10");
        conditionMap.put(Conditions.COND_CR_VACUUM_01,    "Cryo Gate Delta OK");
        conditionMap.put(Conditions.COND_CR_TRB_PUMP_OFF, "Cryo Turbo Stopped");
        conditionMap.put(Conditions.COND_CR_VACUUM_001,   "Cryo Vacuum < 0.01");
        conditionMap.put(Conditions.COND_HX_FORELINE_VAC, "HX Foreline OK");
        conditionMap.put(Conditions.COND_HX_TRB_PRESS_10, "HX Turbo Vac < 10");
        conditionMap.put(Conditions.COND_HX_VACUUM_01,    "HX Gate Delta OK");
        conditionMap.put(Conditions.COND_HX_TURBO_OFF,    "HX Turbo Stopped");
        conditionMap.put(Conditions.COND_HX_VACUUM_001,   "HX Vacuum < 0.01");
    }
    private static final int
        PLC_STATE_WIDTH = UiUtilities.maxEnumLabelWidth(PLCState.class),
        CRYO_STATE_WIDTH = UiUtilities.maxEnumLabelWidth(CryoVacuumState.class),
        HX_STATE_WIDTH = UiUtilities.maxEnumLabelWidth(HxVacuumState.class),
        SWITCH_STATE_WIDTH = Math.max(UiUtilities.maxEnumLabelWidth(SwitchState.class), UiUtilities.maxEnumLabelWidth(DeviceState.class)),
        LATCH_STATE_WIDTH = UiUtilities.maxEnumLabelWidth(LatchState.class),
        COND_STATUS_WIDTH = UiUtilities.maxEnumLabelWidth(ConditionState.class);

    private final CommandSender sender;
    private final UiUtilities uiUtils;

    private SystemStatusPanel sysStatPanel;

    private final JPanel headPanel = new JPanel();
    private JLabel plcStateValue;
    private JLabel cryoStateValue;
    private JLabel hxStateValue;

    private JPanel switchPanel;
    private final JLabel[] switchLabel = new JLabel[Switches.NUM_SWITCHES];
    private final JLabel[] switchStatus = new JLabel[Switches.NUM_SWITCHES];
    private final ButtonGroup[] switchBG = new ButtonGroup[Switches.NUM_SWITCHES];
    private final JRadioButton[] switchOffRB = new JRadioButton[Switches.NUM_SWITCHES];
    private final JRadioButton[] switchOnRB = new JRadioButton[Switches.NUM_SWITCHES];

    private JPanel latchPanel;
    private final JLabel[] latchLabel = new JLabel[Latches.NUM_LATCHES];
    private final JLabel[] latchStatus = new JLabel[Latches.NUM_LATCHES];
    private final JButton[] latchResetBtn = new JButton[Latches.NUM_LATCHES];

    private JPanel conditionPanel;
    private final JLabel[] conditionLabel = new JLabel[Conditions.NUM_CONDITIONS];
    private final JLabel[] conditionStatus = new JLabel[Conditions.NUM_CONDITIONS];

    public VacControlPanel(String subsys) {
        uiUtils = new UiUtilities(this);
        sender = new CommandSender(subsys, this);
        initComponents();
        (new DisablePanel()).run();
    }

    public void initPanel() {
        sender.sendCommand(true, null, "getVacuumState");
    }

    @Override
    public void onCommandReply(Object reply, String path, String command, Object[] args) {
        updatePanel((VacSysState)reply);
    }

    public void updatePanel(VacSysState vss) {
        SwingUtilities.invokeLater(new UpdateVacState(vss));
    }

    public void disablePanel() {
        SwingUtilities.invokeLater(new DisablePanel());
    }

    private void initComponents() {

        // System status line
        sysStatPanel = new SystemStatusPanel(sender, MonitorControl.NODE_NAME);

        // Vacuum state line
        plcStateValue = UiUtilities.newLabel("X", PLC_STATE_WIDTH);
        cryoStateValue = UiUtilities.newLabel("X", CRYO_STATE_WIDTH);
        hxStateValue = UiUtilities.newLabel("X", HX_STATE_WIDTH);

        // Switch states
        for (int sw : switchMap.keySet()) {
            switchLabel[sw] = UiUtilities.newLabel(switchMap.get(sw) + ":", 0);
            switchStatus[sw] = UiUtilities.newLabel("X", SWITCH_STATE_WIDTH);
            int swType = switchTypes[sw];
            switchOffRB[sw] = uiUtils.newRadioButton(swType == SWTP_PUMP ? "Off" : "Shut", "S" + sw);
            switchOnRB[sw] = uiUtils.newRadioButton(swType == SWTP_PUMP ? "On" : "Open", "O" + sw);
            ButtonGroup bg = switchBG[sw] = new ButtonGroup();
            bg.add(switchOffRB[sw]);
            bg.add(switchOnRB[sw]);
        }

        // Latched condition states
        for (int cond : latchMap.keySet()) {
            latchLabel[cond] = UiUtilities.newLabel(latchMap.get(cond) + ":", 0);
            latchStatus[cond] = UiUtilities.newLabel("X", LATCH_STATE_WIDTH);
            latchResetBtn[cond] = uiUtils.newButton("Reset", "R" + cond, true);
        }

        // Condition states
        for (int cond : conditionMap.keySet()) {
            conditionLabel[cond] = UiUtilities.newLabel(conditionMap.get(cond) + ":", 0);
            conditionStatus[cond] = UiUtilities.newLabel("X", COND_STATUS_WIDTH);
        }

        // Vacuum status line (panel)
        headPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbh = new GridBagConstraints();
        gbh.insets = new Insets(5, 0, 5, 0);
        gbh.gridx = 0;
        gbh.gridy = 0;
        gbh.anchor = GridBagConstraints.WEST;
        gbh.insets.left = 5;
        headPanel.add(UiUtilities.newLabel("PLC State: ", 0), gbh);
        gbh.gridx++;
        headPanel.add(plcStateValue, gbh);
        gbh.gridx++;
        gbh.insets.left = 25;
        headPanel.add(UiUtilities.newLabel("Cryo Vac State: ", 0), gbh);
        gbh.gridx++;
        gbh.insets.left = 5;
        headPanel.add(cryoStateValue, gbh);
        gbh.gridx++;
        gbh.insets.left = 25;
        headPanel.add(UiUtilities.newLabel("HX Vac State: ", 0), gbh);
        gbh.gridx++;
        gbh.insets.left = 5;
        headPanel.add(hxStateValue, gbh);

        // Switch panel
        switchPanel = UiUtilities.newBorderedPanel("Switches");
        GridBagConstraints gbs = new GridBagConstraints();
        gbs.anchor = GridBagConstraints.WEST;
        gbs.insets = new Insets(0, 0, 4, 4);
        gbs.gridy = 0;
        int colm = 0;
        int nRows = (switchMap.size() + 1) / 2;
        for (int sw : switchMap.keySet()) {
            gbs.insets.top = gbs.gridy == 0 ? 4 : 0;
            gbs.insets.left = colm == 0 ? 4 : 40;
            gbs.gridx = 4 * colm;
            switchPanel.add(switchLabel[sw], gbs);
            gbs.insets.left = 4;
            gbs.gridx++;
            switchPanel.add(switchStatus[sw], gbs);
            gbs.gridx++;
            switchPanel.add(switchOffRB[sw], gbs);
            gbs.gridx++;
            switchPanel.add(switchOnRB[sw], gbs);
            if (++gbs.gridy == nRows) {
                gbs.gridy = 0;
                colm = 1;
            }
        }

        // Latched conditions panel
        latchPanel = UiUtilities.newBorderedPanel("PLC Error Conditions");
        GridBagConstraints gbl = new GridBagConstraints();
        gbl.anchor = GridBagConstraints.WEST;
        gbl.insets = new Insets(0, 0, 4, 4);
        gbl.gridy = 0;
        colm = 0;
        nRows = (latchMap.size() + 1) / 2;
        for (int cond : latchMap.keySet()) {
            gbl.insets.top = gbl.gridy == 0 ? 4 : 0;
            gbl.insets.left = colm == 0 ? 4 : 40;
            gbl.gridx = 3 * colm;
            latchPanel.add(latchLabel[cond], gbl);
            gbl.insets.left = 4;
            gbl.gridx++;
            latchPanel.add(latchStatus[cond], gbl);
            gbl.gridx++;
            latchPanel.add(latchResetBtn[cond], gbl);
            if (++gbl.gridy == nRows) {
                gbl.gridy = 0;
                colm = 1;
            }
        }

        // Conditions panel
        conditionPanel = UiUtilities.newBorderedPanel("PLC Running Conditions");
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = new Insets(0, 0, 4, 4);
        gbc.gridy = 0;
        colm = 0;
        nRows = (conditionMap.size() + 1) / 2;
        for (int cond : conditionMap.keySet()) {
            gbc.insets.top = gbc.gridy == 0 ? 4 : 0;
            gbc.insets.left = colm == 0 ? 4 : 40;
            gbc.gridx = 2 * colm;
            conditionPanel.add(conditionLabel[cond], gbc);
            gbc.gridx++;
            gbc.insets.left = 4;
            conditionPanel.add(conditionStatus[cond], gbc);
            if (++gbc.gridy == nRows) {
                gbc.gridy = 0;
                colm = 1;
            }
        }

        // Lay out the complete panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbm = new GridBagConstraints();
        gbm.insets = new Insets(6, 0, 6, 0);
        gbm.anchor = GridBagConstraints.NORTH;
        gbm.gridx = 0;
        gbm.gridy = 0;
        add(sysStatPanel, gbm);
        gbm.gridy++;
        add(headPanel, gbm);
        gbm.gridy++;
        add(switchPanel, gbm);
        gbm.gridy++;
        add(latchPanel, gbm);
        gbm.gridy++;
        add(conditionPanel, gbm);
    }

    @Override
    public void handleRadioButton(String name) {
        char type = name.charAt(0);
        int sw = Integer.valueOf(name.substring(1));
        switchOffRB[sw].setEnabled(false);
        switchOnRB[sw].setEnabled(false);
        sender.sendCommand(null, "setSwitchOn", Switches.getName(sw), type == 'O');
    }

    @Override
    public void handleButton(String name) {
        int cond = Integer.valueOf(name.substring(1));
        latchResetBtn[cond].setEnabled(false);
        sender.sendCommand(null, "clearLatch", cond);
    }

    class UpdateVacState implements Runnable {

        private final VacSysState vs;

        UpdateVacState(VacSysState vs) {
            this.vs = vs;
        }

        @Override
        public void run() {
            sysStatPanel.updatePanel(vs.getTickMillis());
            PLCState plcState = vs.getPlcState();
            plcStateValue.setText(plcState.name());
            plcStateValue.setForeground(plcState == PLCState.ALIVE ? UiConstants.GREEN :
                                        plcState == PLCState.OFFLINE ? UiConstants.BLUE : UiConstants.RED);
            plcStateValue.setEnabled(true);
            CryoVacuumState cvState = vs.getCryoVacuumState();
            cryoStateValue.setText(cvState.name());
            cryoStateValue.setForeground(cvState == CryoVacuumState.VACUUM ? UiConstants.GREEN :
                                         cvState == CryoVacuumState.UNKNOWN ? UiConstants.BLUE :
                                         cvState == CryoVacuumState.OFF ? UiConstants.RED : Color.BLACK);
            cryoStateValue.setEnabled(true);
            HxVacuumState hvState = vs.getHxVacuumState();
            hxStateValue.setText(hvState.name());
            hxStateValue.setForeground(hvState == HxVacuumState.VACUUM ? UiConstants.GREEN :
                                       hvState == HxVacuumState.UNKNOWN ? UiConstants.BLUE :
                                       hvState == HxVacuumState.OFF ? UiConstants.RED : Color.BLACK);
            hxStateValue.setEnabled(true);
            for (int sw : switchMap.keySet()) {
                SwitchState state = vs.getSwitchState(sw);
                String text = state.name();
                Color color = state == SwitchState.OFF ? Color.black : state == SwitchState.ON ? UiConstants.GREEN : UiConstants.BLUE;
                if (state != SwitchState.OFFLINE) {
                    if (vs.getSwitchEnable(sw) == SwitchEnable.OFF) {
                        text = "DISABLD";
                        color = UiConstants.RED;
                    }
                    else {
                        if (switchTypes[sw] == SWTP_VALVE) {
                            text = state == SwitchState.OFF ? "SHUT" : state == SwitchState.ON ? "OPEN" : text;
                        }
                        DeviceState devState = vs.getDeviceState(sw);
                        if (devState != null) {
                            text = devState.name();
                            if (devState == DeviceState.FAILED) {
                                color = UiConstants.RED;
                            }
                        }
                    }
                }
                switchStatus[sw].setText(text);
                switchStatus[sw].setForeground(color);

                JRadioButton selButton = state == SwitchState.ON ? switchOnRB[sw] : switchOffRB[sw];
                selButton.setSelected(true);
                switchStatus[sw].setEnabled(true);
                switchOffRB[sw].setEnabled(true);
                switchOnRB[sw].setEnabled(true);
            }
            for (int cond : latchMap.keySet()) {
                LatchState state = vs.getLatch(cond);
                latchStatus[cond].setText(state.name());
                latchStatus[cond].setForeground(state == LatchState.OFFLINE ? UiConstants.BLUE :
                                                state == LatchState.CLEAR ? UiConstants.GREEN :
                                                state == LatchState.ACTIVE ? UiConstants.RED : UiConstants.PURPLE);
                latchStatus[cond].setEnabled(true);
                latchResetBtn[cond].setEnabled(true);
            }
            for (int cond : conditionMap.keySet()) {
                ConditionState state = vs.getCondition(cond);
                conditionStatus[cond].setText(state.name());
                conditionStatus[cond].setForeground(state == ConditionState.OFF ? UiConstants.BLUE :
                                                    state == ConditionState.NO ? Color.BLACK : UiConstants.GREEN);
                conditionStatus[cond].setEnabled(true);
            }
        }

    }

    class DisablePanel implements Runnable {

        @Override
        public void run() {
            sysStatPanel.disablePanel();
            plcStateValue.setEnabled(false);
            cryoStateValue.setEnabled(false);
            hxStateValue.setEnabled(false);
            for (int sw : switchMap.keySet()) {
                switchStatus[sw].setEnabled(false);
                switchOffRB[sw].setEnabled(false);
                switchOnRB[sw].setEnabled(false);
            }
            for (int cond : latchMap.keySet()) {
                latchStatus[cond].setEnabled(false);
                latchResetBtn[cond].setEnabled(false);
            }
            for (int cond : conditionMap.keySet()) {
                conditionStatus[cond].setEnabled(false);
            }
        } 
    }

    private static final long serialVersionUID = 1L;
}
