package org.lsst.ccs.subsystem.vacuum.ui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.border.TitledBorder;
import javax.swing.BorderFactory;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JTextField;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.border.LineBorder;
import org.lsst.ccs.subsystem.vacuum.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.vacuum.constants.ConditionState;
import org.lsst.ccs.subsystem.vacuum.constants.DeviceState;
import org.lsst.ccs.subsystem.vacuum.constants.LatchState;
import org.lsst.ccs.subsystem.vacuum.constants.PLCState;
import org.lsst.ccs.subsystem.vacuum.constants.SwitchEnable;
import org.lsst.ccs.subsystem.vacuum.constants.SwitchState;
import org.lsst.ccs.subsystem.vacuum.data.VacSysState;

/**
 *  Implements the utility monitoring panel.
 *
 *  @author Owen Saxton
 */
public class VacControlPanel extends javax.swing.JPanel {

    private static final int SWTP_PUMP = 0, SWTP_VALVE = 1;
    private static final String[] switchNames = new String[VacSysState.NUM_SWITCHES];
    static {
        switchNames[VacSysState.SW_CRYO_TURBO]     = "Cryo Turbo Pump";
        switchNames[VacSysState.SW_CRYO_ION_PUMP1] = "Cryo Ion Pump 1";
        switchNames[VacSysState.SW_CRYO_ION_PUMP2] = "Cryo Ion Pump 2";
        switchNames[VacSysState.SW_CRYO_ION_PUMP3] = "Cryo Ion Pump 3";
        switchNames[VacSysState.SW_CRYO_ION_PUMP4] = "Cryo Ion Pump 4";
        switchNames[VacSysState.SW_CRYO_ION_PUMP5] = "Cryo Ion Pump 5";
        switchNames[VacSysState.SW_CRYO_ION_PUMP6] = "Cryo Ion Pump 6";
        switchNames[VacSysState.SW_CRYO_VALVE]     = "Cryo Gate Valve";
        switchNames[VacSysState.SW_HX_TURBO]       = "Hx Turbo Pump";
        switchNames[VacSysState.SW_HX_ION_PUMP1]   = "Hx Ion Pump 1";
        switchNames[VacSysState.SW_HX_ION_PUMP2]   = "Hx Ion Pump 2";
        switchNames[VacSysState.SW_HX_VALVE]       = "Hx Gate Valve";
        switchNames[VacSysState.SW_OR_ION_PUMP]    = "O-ring Ion Pump";
        switchNames[VacSysState.SW_OR_FPP_VALVE]   = "O-ring Fpp Valve";
        switchNames[VacSysState.SW_OR_FH_VALVE]    = "O-ring Fh Valve";
        switchNames[VacSysState.SW_OR_L3H_VALVE]   = "O-ring L3h Valve";
        switchNames[VacSysState.SW_OR_L3_VALVE]    = "O-ring L3 Valve";
    }
    private static final int[] switchTypes = new int[VacSysState.NUM_SWITCHES];
    static {
        switchTypes[VacSysState.SW_CRYO_TURBO] = SWTP_PUMP;
        switchTypes[VacSysState.SW_CRYO_ION_PUMP1] = SWTP_PUMP;
        switchTypes[VacSysState.SW_CRYO_ION_PUMP2] = SWTP_PUMP;
        switchTypes[VacSysState.SW_CRYO_ION_PUMP3] = SWTP_PUMP;
        switchTypes[VacSysState.SW_CRYO_ION_PUMP4] = SWTP_PUMP;
        switchTypes[VacSysState.SW_CRYO_ION_PUMP5] = SWTP_PUMP;
        switchTypes[VacSysState.SW_CRYO_ION_PUMP6] = SWTP_PUMP;
        switchTypes[VacSysState.SW_CRYO_VALVE] = SWTP_VALVE;
        switchTypes[VacSysState.SW_HX_TURBO] = SWTP_PUMP;
        switchTypes[VacSysState.SW_HX_ION_PUMP1] = SWTP_PUMP;
        switchTypes[VacSysState.SW_HX_ION_PUMP2] = SWTP_PUMP;
        switchTypes[VacSysState.SW_HX_VALVE] = SWTP_VALVE;
        switchTypes[VacSysState.SW_OR_ION_PUMP] = SWTP_PUMP;
        switchTypes[VacSysState.SW_OR_FPP_VALVE] = SWTP_VALVE;
        switchTypes[VacSysState.SW_OR_FH_VALVE] = SWTP_VALVE;
        switchTypes[VacSysState.SW_OR_L3H_VALVE] = SWTP_VALVE;
        switchTypes[VacSysState.SW_OR_L3_VALVE] = SWTP_VALVE;
    }
    private static final String[] latchNames = new String[VacSysState.NUM_LATCHES];
    static {
        latchNames[VacSysState.LATCH_CR_GATE_AO]  = "Cryo Gate Can't Open";
        latchNames[VacSysState.LATCH_CR_GATE_NFC] = "Cryo Gate Forced Shut";
        latchNames[VacSysState.LATCH_CR_VACUUM]   = "Cryo Vacuum Bad";
        latchNames[VacSysState.LATCH_CR_PUMP]     = "Cryo Turbo Pump Bad";
        latchNames[VacSysState.LATCH_HX_GATE_AO]  = "Hx Gate Can't Open";
        latchNames[VacSysState.LATCH_HX_GATE_NFC] = "Hx Gate Forced Shut";
        latchNames[VacSysState.LATCH_HX_VACUUM]   = "Hx Vacuum Bad";
        latchNames[VacSysState.LATCH_HX_PUMP]     = "Hx Turbo Pump Bad";
    }
    private static final String[] conditionNames = new String[VacSysState.NUM_CONDITIONS];
    static {
        conditionNames[VacSysState.COND_CR_FORELINE_VAC] = "Cryo Foreline OK";
        conditionNames[VacSysState.COND_CR_TRB_PRESS_10] = "Cryo Turbo Pr < 10";
        conditionNames[VacSysState.COND_CR_TRB_PUMP_OFF] = "Cryo Turbo Pump Off";
        conditionNames[VacSysState.COND_CR_VACUUM_001]   = "Cryo Vacuum < 0.001";
        conditionNames[VacSysState.COND_CR_VACUUM_01]    = "Cryo Vacuum < 0.1";
        conditionNames[VacSysState.COND_HX_FORELINE_VAC] = "HX Foreline OK";
        conditionNames[VacSysState.COND_HX_TRB_PRESS_10] = "HX Turbo Pr < 10";
        conditionNames[VacSysState.COND_HX_TURBO_OFF]    = "HX Turbo Pump Off";
        conditionNames[VacSysState.COND_HX_VACUUM_001]   = "HX Vacuum < 0.001";
        conditionNames[VacSysState.COND_HX_VACUUM_01]    = "HX Vacuum < 0.1";
        conditionNames[VacSysState.COND_OR_FH_VALVE]     = "FH Gate Okay";
        conditionNames[VacSysState.COND_OR_FPP_VALVE]    = "FPP Gate Okay";
        conditionNames[VacSysState.COND_OR_L3H_VALVE]    = "L3H Gate Okay";
        conditionNames[VacSysState.COND_OR_L3_VALVE]     = "L3 Gate Okay";
    }
    private static final int switchStatusWidth, latchStatusWidth, condStatusWidth;
    static {
        JLabel label = new JLabel();
        label.setFont(UiConstants.FONT);
        int width = 0;
        for (SwitchState state : SwitchState.values()) {
            label.setText(state.name());
            width = Math.max(width, label.getPreferredSize().width);
        }
        for (DeviceState state : DeviceState.values()) {
            label.setText(state.name());
            width = Math.max(width, label.getPreferredSize().width);
        }
        switchStatusWidth = width;
        width = 0;
        for (LatchState state : LatchState.values()) {
            label.setText(state.name());
            width = Math.max(width, label.getPreferredSize().width);
        }
        latchStatusWidth = width;
        width = 0;
        for (ConditionState state : ConditionState.values()) {
            label.setText(state.name());
            width = Math.max(width, label.getPreferredSize().width);
        }
        condStatusWidth = width;
    }

    private final CommandSender sender;
    private String subsys;
    private double mainPeriod = 5.0;
    private int switchMask, latchMask, conditionMask;

    private final JPanel mainPanel = new JPanel();

    private final JPanel headPanel = new JPanel();
    private final JLabel sysStateLabel = new JLabel("System State:  ");
    private final JLabel sysStateValue = new JLabel("STOPPED");
    private final JLabel plcStateLabel = new JLabel("PLC State:  ");
    private final JLabel plcStateValue = new JLabel("OFFLINE");

    private final JPanel periodPanel = new JPanel();
    private final JLabel periodLabel = new JLabel("Update Period: ");
    private final JTextField periodTextField = new JTextField();
    private final JLabel periodUnitsLabel = new JLabel("sec");

    //private final JPanel[] switchPanel = new JPanel[VacuumState.NUM_SWITCHES];
    private final JPanel switchPanel = new JPanel();
    private final JLabel[] switchLabel = new JLabel[VacSysState.NUM_SWITCHES];
    private final JLabel[] switchStatus = new JLabel[VacSysState.NUM_SWITCHES];
    private final ButtonGroup[] switchBG = new ButtonGroup[VacSysState.NUM_SWITCHES];
    private final JRadioButton[] switchOffRB = new JRadioButton[VacSysState.NUM_SWITCHES];
    private final JRadioButton[] switchOnRB = new JRadioButton[VacSysState.NUM_SWITCHES];

    private final JPanel latchPanel = new JPanel();
    private final JLabel[] latchLabel = new JLabel[VacSysState.NUM_LATCHES];
    private final JLabel[] latchStatus = new JLabel[VacSysState.NUM_LATCHES];
    private final JButton[] latchResetBtn = new JButton[VacSysState.NUM_LATCHES];

    private final JPanel conditionPanel = new JPanel();
    private final JLabel[] conditionLabel = new JLabel[VacSysState.NUM_CONDITIONS];
    private final JLabel[] conditionStatus = new JLabel[VacSysState.NUM_CONDITIONS];

    public VacControlPanel(CommandSender cons) {
        this.sender = cons;
        initComponents();
        (new UpdateVacStatus(new VacSysState())).run();
        (new DisableSystem()).run();
    }

    public void setSubsystem(String name) {
        this.subsys = name;
    }

    private void initComponents() {

        // System state
        sysStateLabel.setFont(UiConstants.FONT);
        sysStateValue.setFont(UiConstants.FONT);
        Dimension d = sysStateValue.getPreferredSize();
        d.width = 100;
        sysStateValue.setPreferredSize(d);

        // PLC state
        plcStateLabel.setFont(UiConstants.FONT);
        plcStateValue.setFont(UiConstants.FONT);
        d = plcStateValue.getPreferredSize();
        d.width = 100;
        plcStateValue.setPreferredSize(d);

        // Update period
        periodLabel.setFont(UiConstants.FONT);
        d = periodTextField.getPreferredSize();
        d.width = 45;
        periodTextField.setPreferredSize(d);
        periodTextField.setHorizontalAlignment(SwingConstants.CENTER);
        periodTextField.setText("...");
        periodTextField.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setUpdatePeriod();
            }
        });
        periodUnitsLabel.setFont(UiConstants.FONT);

        // Switch states
        for (int sw = 0; sw < VacSysState.NUM_SWITCHES; sw++) {
            JLabel label = switchLabel[sw] = new JLabel(switchNames[sw] + ":");
            label.setFont(UiConstants.FONT);
            //d = label.getPreferredSize();
            //d.width = 132;
            //label.setPreferredSize(d);
            label = switchStatus[sw] = new JLabel("XXX");
            label.setFont(UiConstants.FONT);
            d = label.getPreferredSize();
            d.width = switchStatusWidth;
            label.setPreferredSize(d);
            int swType = switchTypes[sw];
            JRadioButton buttonOff = switchOffRB[sw] = new JRadioButton(swType == SWTP_PUMP ? "Off" : "Shut");
            buttonOff.setFont(UiConstants.FONT);
            //buttonOff.setText("Off");
            buttonOff.setFocusable(false);
            buttonOff.setName(Integer.toString(sw));
            buttonOff.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent evt) {
                    String name = ((JRadioButton)evt.getSource()).getName();
                    setSwitchOn(Integer.valueOf(name), false);
                }
            });
            JRadioButton buttonOn = switchOnRB[sw] = new JRadioButton(swType == SWTP_PUMP ? "On" : "Open");
            buttonOn.setFont(UiConstants.FONT);
            //buttonOn.setText("On");
            buttonOn.setFocusable(false);
            buttonOn.setName(Integer.toString(sw));
            buttonOn.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent evt) {
                    String name = ((JRadioButton)evt.getSource()).getName();
                    setSwitchOn(Integer.valueOf(name), true);
                }
            });
            ButtonGroup bg = switchBG[sw] = new ButtonGroup();
            bg.add(buttonOff);
            bg.add(buttonOn);
        }

        // Latched condition states
        for (int cond = 0; cond < VacSysState.NUM_LATCHES; cond++) {
            JLabel label = latchLabel[cond] = new JLabel(latchNames[cond] + ":");
            label.setFont(UiConstants.FONT);
            //d = label.getPreferredSize();
            //d.width = 171;
            //label.setPreferredSize(d);
            label = latchStatus[cond] = new JLabel("XXX");
            label.setFont(UiConstants.FONT);
            d = label.getPreferredSize();
            d.width = latchStatusWidth;
            label.setPreferredSize(d);
            JButton button = latchResetBtn[cond] = new JButton("Reset");
            Insets i = button.getMargin();
            i.left -= 3;
            i.right -= 3;
            button.setMargin(i);
            d = button.getPreferredSize();
            d.height -= 9;
            button.setPreferredSize(d);
            button.setFont(UiConstants.FONT);
            button.setFocusable(false);
            button.setName(Integer.toString(cond));
            button.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent evt) {
                    String name = ((JButton)evt.getSource()).getName();
                    clearLatch(Integer.valueOf(name));
                }
            });
        }

        // Condition states
        for (int cond = 0; cond < VacSysState.NUM_CONDITIONS; cond++) {
            JLabel label = conditionLabel[cond] = new JLabel(conditionNames[cond] + ":");
            label.setFont(UiConstants.FONT);
            //d = label.getPreferredSize();
            //d.width = 160;
            //label.setPreferredSize(d);
            label = conditionStatus[cond] = new JLabel("XXX");
            label.setFont(UiConstants.FONT);
            d = label.getPreferredSize();
            d.width = condStatusWidth;
            label.setPreferredSize(d);
        }

        // Lay out all the sub-panels
        periodPanel.add(periodTextField);
        periodPanel.add(periodUnitsLabel);

        // Add states & period items to head panel
        headPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbh = new GridBagConstraints();
        gbh.insets = new Insets(5, 0, 0, 0);
        gbh.gridx = 0;
        gbh.gridy = 0;
        gbh.anchor = GridBagConstraints.WEST;
        headPanel.add(sysStateLabel, gbh);
        gbh.gridx++;
        //gbh.anchor = GridBagConstraints.WEST;
        headPanel.add(sysStateValue, gbh);
        gbh.gridx++;
        headPanel.add(plcStateLabel, gbh);
        gbh.gridx++;
        //gbh.anchor = GridBagConstraints.WEST;
        headPanel.add(plcStateValue, gbh);
        gbh.gridx++;
        //gbh.gridwidth = 2;
        //gbh.anchor = GridBagConstraints.EAST;
        headPanel.add(periodLabel, gbh);
        gbh.gridx++;
        //gbh.gridwidth = 1;
        //gbh.anchor = GridBagConstraints.WEST;
        headPanel.add(periodPanel, gbh);
        //gbh.gridx = 0;

        // Prepare the switch panel
        TitledBorder border = BorderFactory.createTitledBorder(new LineBorder(Color.BLACK), "Switches");
        border.setTitleJustification(TitledBorder.CENTER);
        border.setTitleFont(UiConstants.FONT);
        border.setTitleColor(UiConstants.BLUE);
        switchPanel.setBorder(border);
        switchPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbs = new GridBagConstraints();
        gbs.gridx = 0;
        gbs.gridy = 0;
        gbs.insets.bottom = -8;
        switchPanel.add(new JLabel(" "), gbs);

        // Prepare the latched conditions panel
        border = BorderFactory.createTitledBorder(new LineBorder(Color.BLACK), "PLC Error Conditions");
        border.setTitleJustification(TitledBorder.CENTER);
        border.setTitleFont(UiConstants.FONT);
        border.setTitleColor(UiConstants.BLUE);
        latchPanel.setBorder(border);
        latchPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbl = new GridBagConstraints();
        gbl.gridx = 0;
        gbl.gridy = 0;
        gbl.insets.bottom = -8;
        latchPanel.add(new JLabel(" "), gbl);

        // Prepare the conditions panel
        border = BorderFactory.createTitledBorder(new LineBorder(Color.BLACK), "PLC Running Conditions");
        border.setTitleJustification(TitledBorder.CENTER);
        border.setTitleFont(UiConstants.FONT);
        border.setTitleColor(UiConstants.BLUE);
        conditionPanel.setBorder(border);
        conditionPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.insets.bottom = -8;
        conditionPanel.add(new JLabel(" "), gbc);

        // Lay out the main panel
        mainPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbm = new GridBagConstraints();
        gbm.insets = new Insets(0, 0, 0, 0);
        gbm.anchor = GridBagConstraints.NORTH;
        gbm.gridx = 0;
        gbm.gridy = 0;
        mainPanel.add(headPanel, gbm);
        gbm.gridy++;
        mainPanel.add(switchPanel, gbm);
        gbm.gridy++;
        gbm.insets.top = 6;
        mainPanel.add(latchPanel, gbm);
        gbm.gridy++;
        mainPanel.add(conditionPanel, gbm);

        // Lay out the complete panel
        add(mainPanel);

    }

    private void updateSwitchLayout(int switches) {
        for (int sw = 0; sw < VacSysState.NUM_SWITCHES; sw++) {
            if ((switchMask & (1 << sw)) != 0) {
                switchPanel.remove(switchLabel[sw]);
                switchPanel.remove(switchStatus[sw]);
                switchPanel.remove(switchOffRB[sw]);
                switchPanel.remove(switchOnRB[sw]);
            }
        }
        GridBagConstraints gbs = new GridBagConstraints();
        gbs.anchor = GridBagConstraints.NORTHWEST;
        gbs.gridy = 0;
        int colm = 0;
        for (int sw = 0; sw < VacSysState.NUM_SWITCHES; sw++) {
            if ((switches & (1 << sw)) != 0) {
                if (colm == 0) {
                    gbs.gridx = 0;
                    gbs.gridy++;
                }
                gbs.insets.top = 0;
                gbs.insets.left = colm == 0 ? 4 : 40;
                //gbs.anchor = GridBagConstraints.NORTHEAST;
                switchPanel.add(switchLabel[sw], gbs);
                gbs.insets.left = 6;
                gbs.gridx++;
                //gbs.anchor = GridBagConstraints.NORTHWEST;
                switchPanel.add(switchStatus[sw], gbs);
                gbs.gridx++;
                gbs.insets.top = -4;
                switchPanel.add(switchOffRB[sw], gbs);
                gbs.gridx++;
                gbs.insets.right = 3;
                switchPanel.add(switchOnRB[sw], gbs);
                gbs.insets.right = 0;
                gbs.gridx++;
                colm = (colm + 1) % 2;
            }
        }
        switchMask = switches;
    }

    private void updateLatchLayout(int conditions) {
        for (int cond = 0; cond < VacSysState.NUM_LATCHES; cond++) {
            if ((latchMask & (1 << cond)) != 0) {
                latchPanel.remove(latchLabel[cond]);
                latchPanel.remove(latchStatus[cond]);
                latchPanel.remove(latchResetBtn[cond]);
            }
        }
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.gridy = 0;
        int colm = 0;
        for (int cond = 0; cond < VacSysState.NUM_LATCHES; cond++) {
            if ((conditions & (1 << cond)) != 0) {
                if (colm == 0) {
                    gbc.gridx = 0;
                    gbc.gridy++;
                }
                gbc.insets.top = 0;
                gbc.insets.bottom = 4;
                gbc.insets.left = colm == 0 ? 4 : 40;
                latchPanel.add(latchLabel[cond], gbc);
                gbc.insets.left = 6;
                gbc.gridx++;
                latchPanel.add(latchStatus[cond], gbc);
                gbc.gridx++;
                gbc.insets.top = -1;
                gbc.insets.right = 4;
                latchPanel.add(latchResetBtn[cond], gbc);
                gbc.insets.right = 0;
                gbc.gridx++;
                colm = (colm + 1) % 2;
            }
        }
        latchMask = conditions;
    }

    private void updateConditionLayout(int conditions) {
        for (int cond = 0; cond < VacSysState.NUM_LATCHES; cond++) {
            if ((conditionMask & (1 << cond)) != 0) {
                conditionPanel.remove(conditionLabel[cond]);
                conditionPanel.remove(conditionStatus[cond]);
            }
        }
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.gridy = 0;
        int colm = 0;
        for (int cond = 0; cond < VacSysState.NUM_CONDITIONS; cond++) {
            if ((conditions & (1 << cond)) != 0) {
                if (colm == 0) {
                    gbc.gridx = 0;
                    gbc.gridy++;
                }
                gbc.insets.top = 0;
                gbc.insets.bottom = 4;
                gbc.insets.left = colm == 0 ? 4 : 40;
                conditionPanel.add(conditionLabel[cond], gbc);
                gbc.gridx++;
                gbc.insets.left = 6;
                gbc.insets.right = 4;
                conditionPanel.add(conditionStatus[cond], gbc);
                gbc.insets.right = 0;
                gbc.gridx++;
                colm = (colm + 1) % 2;
            }
        }
        conditionMask = conditions;
    }

    private void setUpdatePeriod() {
        try {
            double value = Double.valueOf(periodTextField.getText());
            sender.sendCommand(subsys, "setUpdatePeriod", (int)(1000 * value));
            periodTextField.setEnabled(false);
        }
        catch(NumberFormatException nfe) {
            periodTextField.setText(String.valueOf(mainPeriod));
        }
    }

    private void setSwitchOn(int sw, boolean value) {
        switchOffRB[sw].setEnabled(false);
        switchOnRB[sw].setEnabled(false);
        sender.sendCommand(subsys, "setSwitchOn", sw, value);
    }
    
    private void clearLatch(int cond) {
        latchResetBtn[cond].setEnabled(false);
        sender.sendCommand(subsys, "clearLatch", cond);
    }
    
    public void updateControlPanel(VacSysState rs) {
        SwingUtilities.invokeLater(new UpdateVacStatus(rs));
    }

    public void disableSystem() {
        SwingUtilities.invokeLater(new DisableSystem());
    }

    class UpdateVacStatus implements Runnable {

        private final VacSysState vs;

        UpdateVacStatus(VacSysState vs) {
            this.vs = vs;
        }

        @Override
        public void run() {
            sysStateValue.setText("RUNNING");
            sysStateValue.setForeground(UiConstants.GREEN);
            PLCState plcState = vs.getPlcState();
            plcStateValue.setText(plcState.name());
            plcStateValue.setForeground(plcState == PLCState.ALIVE ? UiConstants.GREEN :
                                        plcState == PLCState.OFFLINE ? UiConstants.BLUE : UiConstants.RED);
            plcStateValue.setEnabled(true);
            periodTextField.setEnabled(true);
            mainPeriod = vs.getTickMillis() / 1000.0;
            periodTextField.setText(String.valueOf(mainPeriod));
            int switches = 0;
            for (int sw = 0; sw < VacSysState.NUM_SWITCHES; sw++) {
                if (!vs.hasSwitch(sw)) continue;
                switches |= (1 << sw);
                SwitchState state = vs.getSwitchState(sw);
                String text;
                Color color;
                SwitchEnable enable = vs.getSwitchEnable(sw);
                if (enable == SwitchEnable.OFF) {
                    text = "DISABLD";
                    color = UiConstants.RED;
                }
                else {
                    text = state.name();
                    if (switchTypes[sw] == SWTP_VALVE) {
                        text = state == SwitchState.OFF ? "SHUT" : state == SwitchState.ON ? "OPEN" : text;
                    }
                    color = state == SwitchState.OFF ? Color.black : state == SwitchState.ON ? UiConstants.GREEN : UiConstants.BLUE;
                    DeviceState devState = vs.getDeviceState(sw);
                    if (devState != null && state != SwitchState.OFFLINE) {
                        text = devState.name();
                        if (devState == DeviceState.FAILED) {
                            color = UiConstants.RED;
                        }
                    }
                }
                switchStatus[sw].setText(text);
                switchStatus[sw].setForeground(color);

                JRadioButton selButton = state == SwitchState.ON ? switchOnRB[sw] : switchOffRB[sw];
                selButton.setSelected(true);
                switchStatus[sw].setEnabled(true);
                switchOffRB[sw].setEnabled(true);
                switchOnRB[sw].setEnabled(true);
            }
            if (switches != switchMask) {
                updateSwitchLayout(switches);
            }
            int conditions = 0;
            for (int cond = 0; cond < VacSysState.NUM_LATCHES; cond++) {
                if (!vs.hasLatch(cond)) continue;
                conditions |= (1 << cond);
                LatchState state = vs.getLatch(cond);
                latchStatus[cond].setText(state.name());
                latchStatus[cond].setForeground(state == LatchState.OFFLINE ? UiConstants.BLUE :
                                                state == LatchState.CLEAR ? UiConstants.GREEN :
                                                state == LatchState.ACTIVE ? UiConstants.RED : UiConstants.PURPLE);
                latchStatus[cond].setEnabled(true);
                latchResetBtn[cond].setEnabled(true);
            }
            if (conditions != latchMask) {
                updateLatchLayout(conditions);
            }
            conditions = 0;
            for (int cond = 0; cond < VacSysState.NUM_CONDITIONS; cond++) {
                if (!vs.hasCondition(cond)) continue;
                conditions |= (1 << cond);
                ConditionState state = vs.getCondition(cond);
                conditionStatus[cond].setText(state.name());
                conditionStatus[cond].setForeground(state == ConditionState.OFF ? UiConstants.BLUE :
                                                    state == ConditionState.NO ? Color.BLACK : UiConstants.GREEN);
                conditionStatus[cond].setEnabled(true);
            }
            if (conditions != conditionMask) {
                updateConditionLayout(conditions);
            }
        }

    }

    class DisableSystem implements Runnable {

        @Override
        public void run() {
            sysStateValue.setText("STOPPED");
            sysStateValue.setForeground(UiConstants.RED);
            plcStateValue.setEnabled(false);
            periodTextField.setEnabled(false);
            for (int sw = 0; sw < VacSysState.NUM_SWITCHES; sw++) {
                switchStatus[sw].setEnabled(false);
                switchOffRB[sw].setEnabled(false);
                switchOnRB[sw].setEnabled(false);
            }
            for (int cond = 0; cond < VacSysState.NUM_LATCHES; cond++) {
                latchStatus[cond].setEnabled(false);
                latchResetBtn[cond].setEnabled(false);
            }
            for (int cond = 0; cond < VacSysState.NUM_CONDITIONS; cond++) {
                conditionStatus[cond].setEnabled(false);
            }
        } 
    }

    private static final long serialVersionUID = 1L;
}
