package org.lsst.ccs.subsystem.vacuum.ui;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.SwingUtilities;
import org.lsst.ccs.subsystem.common.ui.TextFieldX;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.common.ui.UiConstants;
import org.lsst.ccs.subsystem.common.ui.UiUtilities;
import org.lsst.ccs.subsystem.vacuum.constants.ConditionState;
import org.lsst.ccs.subsystem.vacuum.constants.DeviceState;
import org.lsst.ccs.subsystem.vacuum.constants.LatchState;
import org.lsst.ccs.subsystem.vacuum.constants.PLCState;
import org.lsst.ccs.subsystem.vacuum.constants.SwitchEnable;
import org.lsst.ccs.subsystem.vacuum.constants.SwitchNames;
import org.lsst.ccs.subsystem.vacuum.constants.SwitchState;
import org.lsst.ccs.subsystem.vacuum.data.VacSysState;

/**
 *  Implements the utility monitoring panel.
 *
 *  @author Owen Saxton
 */
public class VacControlPanel extends JPanel implements UiUtilities.ActionHandler, CommandSender.ReplyHandler {

    private static final int SWTP_PUMP = 0, SWTP_VALVE = 1;
    private static final String[] switchNames = new String[VacSysState.NUM_SWITCHES];
    static {
        switchNames[VacSysState.SW_CRYO_TURBO_PUMP] = "Cryo Turbo Pump";
        switchNames[VacSysState.SW_CRYO_ION_PUMP1]  = "Cryo Ion Pump 1";
        switchNames[VacSysState.SW_CRYO_ION_PUMP2]  = "Cryo Ion Pump 2";
        switchNames[VacSysState.SW_CRYO_ION_PUMP3]  = "Cryo Ion Pump 3";
        switchNames[VacSysState.SW_CRYO_ION_PUMP4]  = "Cryo Ion Pump 4";
        switchNames[VacSysState.SW_CRYO_ION_PUMP5]  = "Cryo Ion Pump 5";
        switchNames[VacSysState.SW_CRYO_ION_PUMP6]  = "Cryo Ion Pump 6";
        switchNames[VacSysState.SW_CRYO_GATE_VALVE] = "Cryo Gate Valve";
        switchNames[VacSysState.SW_HX_TURBO_PUMP]   = "Hx Turbo Pump";
        switchNames[VacSysState.SW_HX_ION_PUMP1]    = "Hx Ion Pump 1";
        switchNames[VacSysState.SW_HX_ION_PUMP2]    = "Hx Ion Pump 2";
        switchNames[VacSysState.SW_HX_GATE_VALVE]   = "Hx Gate Valve";
        switchNames[VacSysState.SW_OR_ION_PUMP]     = "O-ring Ion Pump";
        switchNames[VacSysState.SW_OR_FPP_VALVE]    = "O-ring Fpp Valve";
        switchNames[VacSysState.SW_OR_FH_VALVE]     = "O-ring Fh Valve";
        switchNames[VacSysState.SW_OR_L3H_VALVE]    = "O-ring L3h Valve";
        switchNames[VacSysState.SW_OR_L3_VALVE]     = "O-ring L3 Valve";
    }
    private static final int[] switchTypes = new int[VacSysState.NUM_SWITCHES];
    static {
        switchTypes[VacSysState.SW_CRYO_TURBO_PUMP] = SWTP_PUMP;
        switchTypes[VacSysState.SW_CRYO_ION_PUMP1] = SWTP_PUMP;
        switchTypes[VacSysState.SW_CRYO_ION_PUMP2] = SWTP_PUMP;
        switchTypes[VacSysState.SW_CRYO_ION_PUMP3] = SWTP_PUMP;
        switchTypes[VacSysState.SW_CRYO_ION_PUMP4] = SWTP_PUMP;
        switchTypes[VacSysState.SW_CRYO_ION_PUMP5] = SWTP_PUMP;
        switchTypes[VacSysState.SW_CRYO_ION_PUMP6] = SWTP_PUMP;
        switchTypes[VacSysState.SW_CRYO_GATE_VALVE] = SWTP_VALVE;
        switchTypes[VacSysState.SW_HX_TURBO_PUMP] = SWTP_PUMP;
        switchTypes[VacSysState.SW_HX_ION_PUMP1] = SWTP_PUMP;
        switchTypes[VacSysState.SW_HX_ION_PUMP2] = SWTP_PUMP;
        switchTypes[VacSysState.SW_HX_GATE_VALVE] = SWTP_VALVE;
        switchTypes[VacSysState.SW_OR_ION_PUMP] = SWTP_PUMP;
        switchTypes[VacSysState.SW_OR_FPP_VALVE] = SWTP_VALVE;
        switchTypes[VacSysState.SW_OR_FH_VALVE] = SWTP_VALVE;
        switchTypes[VacSysState.SW_OR_L3H_VALVE] = SWTP_VALVE;
        switchTypes[VacSysState.SW_OR_L3_VALVE] = SWTP_VALVE;
    }
    private static final String[] latchNames = new String[VacSysState.NUM_LATCHES];
    static {
        latchNames[VacSysState.LATCH_CR_GATE_AO]  = "Cryo Gate Can't Open";
        latchNames[VacSysState.LATCH_CR_GATE_NFC] = "Cryo Gate Forced Shut";
        latchNames[VacSysState.LATCH_CR_VACUUM]   = "Cryo Vacuum Bad";
        latchNames[VacSysState.LATCH_CR_PUMP]     = "Cryo Turbo Pump Bad";
        latchNames[VacSysState.LATCH_HX_GATE_AO]  = "Hx Gate Can't Open";
        latchNames[VacSysState.LATCH_HX_GATE_NFC] = "Hx Gate Forced Shut";
        latchNames[VacSysState.LATCH_HX_VACUUM]   = "Hx Vacuum Bad";
        latchNames[VacSysState.LATCH_HX_PUMP]     = "Hx Turbo Pump Bad";
    }
    private static final String[] conditionNames = new String[VacSysState.NUM_CONDITIONS];
    static {
        conditionNames[VacSysState.COND_CR_FORELINE_VAC] = "Cryo Foreline OK";
        conditionNames[VacSysState.COND_CR_TRB_PRESS_10] = "Cryo Turbo Pr < 10";
        conditionNames[VacSysState.COND_CR_TRB_PUMP_OFF] = "Cryo Turbo Pump Off";
        conditionNames[VacSysState.COND_CR_VACUUM_001]   = "Cryo Vacuum < 0.001";
        conditionNames[VacSysState.COND_CR_VACUUM_01]    = "Cryo Vacuum < 0.1";
        conditionNames[VacSysState.COND_HX_FORELINE_VAC] = "HX Foreline OK";
        conditionNames[VacSysState.COND_HX_TRB_PRESS_10] = "HX Turbo Pr < 10";
        conditionNames[VacSysState.COND_HX_TURBO_OFF]    = "HX Turbo Pump Off";
        conditionNames[VacSysState.COND_HX_VACUUM_001]   = "HX Vacuum < 0.001";
        conditionNames[VacSysState.COND_HX_VACUUM_01]    = "HX Vacuum < 0.1";
        conditionNames[VacSysState.COND_OR_FH_VALVE]     = "FH Gate Okay";
        conditionNames[VacSysState.COND_OR_FPP_VALVE]    = "FPP Gate Okay";
        conditionNames[VacSysState.COND_OR_L3H_VALVE]    = "L3H Gate Okay";
        conditionNames[VacSysState.COND_OR_L3_VALVE]     = "L3 Gate Okay";
    }
    private static final int sysStateWidth = UiUtilities.maxLabelWidth(new String[]{"RUNNING", "STOPPED"}, "");
    private static final int plcStateWidth = UiUtilities.maxEnumLabelWidth(PLCState.class);
    private static final int switchStatusWidth = Math.max(UiUtilities.maxEnumLabelWidth(SwitchState.class),
                                                          UiUtilities.maxEnumLabelWidth(DeviceState.class));
    private static final int latchStatusWidth = UiUtilities.maxEnumLabelWidth(LatchState.class);
    private static final int condStatusWidth = UiUtilities.maxEnumLabelWidth(ConditionState.class);

    private final CommandSender sender;
    private final UiUtilities uiUtils;
    private int switchMask = 0, latchMask = 0, conditionMask = 0;

    private final JPanel headPanel = new JPanel();
    private JLabel sysStateValue;
    private JLabel plcStateValue;
    private TextFieldX periodTextField;

    private JPanel switchPanel;
    private final JLabel[] switchLabel = new JLabel[VacSysState.NUM_SWITCHES];
    private final JLabel[] switchStatus = new JLabel[VacSysState.NUM_SWITCHES];
    private final ButtonGroup[] switchBG = new ButtonGroup[VacSysState.NUM_SWITCHES];
    private final JRadioButton[] switchOffRB = new JRadioButton[VacSysState.NUM_SWITCHES];
    private final JRadioButton[] switchOnRB = new JRadioButton[VacSysState.NUM_SWITCHES];

    private JPanel latchPanel;
    private final JLabel[] latchLabel = new JLabel[VacSysState.NUM_LATCHES];
    private final JLabel[] latchStatus = new JLabel[VacSysState.NUM_LATCHES];
    private final JButton[] latchResetBtn = new JButton[VacSysState.NUM_LATCHES];

    private JPanel conditionPanel;
    private final JLabel[] conditionLabel = new JLabel[VacSysState.NUM_CONDITIONS];
    private final JLabel[] conditionStatus = new JLabel[VacSysState.NUM_CONDITIONS];

    public VacControlPanel(String subsys) {
        uiUtils = new UiUtilities(this);
        sender = new CommandSender(subsys, this);
        initComponents();
        (new DisablePanel()).run();
    }

    public void initPanel() {
        sender.sendCommand(true, null, "getVacuumState");
    }

    @Override
    public void onCommandReply(Object reply, String path, String command, Object[] args) {
        updatePanel((VacSysState)reply);
    }

    public void updatePanel(VacSysState vss) {
        SwingUtilities.invokeLater(new UpdateVacState(vss));
    }

    public void disablePanel() {
        SwingUtilities.invokeLater(new DisablePanel());
    }

    private void initComponents() {

        // System status line
        sysStateValue = UiUtilities.newLabel("X", sysStateWidth);
        plcStateValue = UiUtilities.newLabel("X", plcStateWidth);
        periodTextField = uiUtils.newTextFieldX("00000.0", "U", TextFieldX.TYPE_DOUBLE);

        // Switch states
        for (int sw = 0; sw < VacSysState.NUM_SWITCHES; sw++) {
            switchLabel[sw] = UiUtilities.newLabel(switchNames[sw] + ":", 0);
            switchStatus[sw] = UiUtilities.newLabel("X", switchStatusWidth);
            int swType = switchTypes[sw];
            switchOffRB[sw] = uiUtils.newRadioButton(swType == SWTP_PUMP ? "Off" : "Shut", "S" + sw);
            switchOnRB[sw] = uiUtils.newRadioButton(swType == SWTP_PUMP ? "On" : "Open", "O" + sw);
            ButtonGroup bg = switchBG[sw] = new ButtonGroup();
            bg.add(switchOffRB[sw]);
            bg.add(switchOnRB[sw]);
        }

        // Latched condition states
        for (int cond = 0; cond < VacSysState.NUM_LATCHES; cond++) {
            latchLabel[cond] = UiUtilities.newLabel(latchNames[cond] + ":", 0);
            latchStatus[cond] = UiUtilities.newLabel("X", latchStatusWidth);
            latchResetBtn[cond] = uiUtils.newButton("Reset", "R" + cond, true);
        }

        // Condition states
        for (int cond = 0; cond < VacSysState.NUM_CONDITIONS; cond++) {
            conditionLabel[cond] = UiUtilities.newLabel(conditionNames[cond] + ":", 0);
            conditionStatus[cond] = UiUtilities.newLabel("X", condStatusWidth);
        }

        // System status line (panel)
        headPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbh = new GridBagConstraints();
        gbh.insets = new Insets(5, 0, 5, 0);
        gbh.gridx = 0;
        gbh.gridy = 0;
        gbh.anchor = GridBagConstraints.WEST;
        gbh.insets.left = 5;
        headPanel.add(UiUtilities.newLabel("System State: ", 0), gbh);
        gbh.gridx++;
        headPanel.add(sysStateValue, gbh);
        gbh.gridx++;
        gbh.insets.left = 25;
        headPanel.add(UiUtilities.newLabel("PLC State: ", 0), gbh);
        gbh.gridx++;
        gbh.insets.left = 5;
        headPanel.add(plcStateValue, gbh);
        gbh.gridx++;
        gbh.insets.left = 25;
        headPanel.add(UiUtilities.newLabel("Update Period: ", 0), gbh);
        gbh.gridx++;
        gbh.insets.left = 5;
        headPanel.add(periodTextField, gbh);
        gbh.gridx++;
        gbh.insets.right = 5;
        headPanel.add(UiUtilities.newLabel("sec", 0), gbh);

        // Prepare the switch panel
        switchPanel = UiUtilities.newBorderedPanel("Switches");
        GridBagConstraints gbs = new GridBagConstraints();
        gbs.gridx = 0;
        gbs.gridy = 0;
        gbs.insets.bottom = -8;
        switchPanel.add(new JLabel(" "), gbs);

        // Prepare the latched conditions panel
        latchPanel = UiUtilities.newBorderedPanel("PLC Error Conditions");
        GridBagConstraints gbl = new GridBagConstraints();
        gbl.gridx = 0;
        gbl.gridy = 0;
        gbl.insets.bottom = -8;
        latchPanel.add(new JLabel(" "), gbl);

        // Prepare the conditions panel
        conditionPanel = UiUtilities.newBorderedPanel("PLC Running Conditions");
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.insets.bottom = -8;
        conditionPanel.add(new JLabel(" "), gbc);

        // Lay out the complete panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbm = new GridBagConstraints();
        gbm.insets = new Insets(6, 0, 6, 0);
        gbm.anchor = GridBagConstraints.NORTH;
        gbm.gridx = 0;
        gbm.gridy = 0;
        add(headPanel, gbm);
        gbm.gridy++;
        add(switchPanel, gbm);
        gbm.gridy++;
        add(latchPanel, gbm);
        gbm.gridy++;
        add(conditionPanel, gbm);
    }

    private void updateSwitchLayout(int switches) {
        for (int sw = 0; sw < VacSysState.NUM_SWITCHES; sw++) {
            if ((switchMask & (1 << sw)) != 0) {
                switchPanel.remove(switchLabel[sw]);
                switchPanel.remove(switchStatus[sw]);
                switchPanel.remove(switchOffRB[sw]);
                switchPanel.remove(switchOnRB[sw]);
            }
        }
        GridBagConstraints gbs = new GridBagConstraints();
        gbs.anchor = GridBagConstraints.WEST;
        gbs.insets = new Insets(0, 0, 4, 4);
        gbs.gridy = -1;
        int colm = 0;
        for (int sw = 0; sw < VacSysState.NUM_SWITCHES; sw++) {
            if ((switches & (1 << sw)) != 0) {
                if (colm == 0) {
                    gbs.gridx = 0;
                    gbs.gridy++;
                }
                gbs.insets.top = gbs.gridy == 0 ? 4 : 0;
                gbs.insets.left = colm == 0 ? 4 : 40;
                switchPanel.add(switchLabel[sw], gbs);
                gbs.insets.left = 4;
                gbs.gridx++;
                switchPanel.add(switchStatus[sw], gbs);
                gbs.gridx++;
                switchPanel.add(switchOffRB[sw], gbs);
                gbs.gridx++;
                switchPanel.add(switchOnRB[sw], gbs);
                gbs.gridx++;
                colm = (colm + 1) % 2;
            }
        }
        switchMask = switches;
    }

    private void updateLatchLayout(int conditions) {
        for (int cond = 0; cond < VacSysState.NUM_LATCHES; cond++) {
            if ((latchMask & (1 << cond)) != 0) {
                latchPanel.remove(latchLabel[cond]);
                latchPanel.remove(latchStatus[cond]);
                latchPanel.remove(latchResetBtn[cond]);
            }
        }
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = new Insets(0, 0, 4, 4);
        gbc.gridy = -1;
        int colm = 0;
        for (int cond = 0; cond < VacSysState.NUM_LATCHES; cond++) {
            if ((conditions & (1 << cond)) != 0) {
                if (colm == 0) {
                    gbc.gridx = 0;
                    gbc.gridy++;
                }
                gbc.insets.top = gbc.gridy == 0 ? 4 : 0;
                gbc.insets.left = colm == 0 ? 4 : 40;
                latchPanel.add(latchLabel[cond], gbc);
                gbc.insets.left = 4;
                gbc.gridx++;
                latchPanel.add(latchStatus[cond], gbc);
                gbc.gridx++;
                latchPanel.add(latchResetBtn[cond], gbc);
                gbc.gridx++;
                colm = (colm + 1) % 2;
            }
        }
        latchMask = conditions;
    }

    private void updateConditionLayout(int conditions) {
        for (int cond = 0; cond < VacSysState.NUM_LATCHES; cond++) {
            if ((conditionMask & (1 << cond)) != 0) {
                conditionPanel.remove(conditionLabel[cond]);
                conditionPanel.remove(conditionStatus[cond]);
            }
        }
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = new Insets(0, 0, 4, 4);
        gbc.gridy = -1;
        int colm = 0;
        for (int cond = 0; cond < VacSysState.NUM_CONDITIONS; cond++) {
            if ((conditions & (1 << cond)) != 0) {
                if (colm == 0) {
                    gbc.gridx = 0;
                    gbc.gridy++;
                }
                gbc.insets.top = gbc.gridy == 0 ? 4 : 0;
                gbc.insets.left = colm == 0 ? 4 : 40;
                conditionPanel.add(conditionLabel[cond], gbc);
                gbc.gridx++;
                gbc.insets.left = 4;
                conditionPanel.add(conditionStatus[cond], gbc);
                gbc.gridx++;
                colm = (colm + 1) % 2;
            }
        }
        conditionMask = conditions;
    }

    @Override
    public void handleRadioButton(String name) {
        char type = name.charAt(0);
        int sw = Integer.valueOf(name.substring(1));
        switchOffRB[sw].setEnabled(false);
        switchOnRB[sw].setEnabled(false);
        sender.sendCommand(null, "setNamedSwitchOn", SwitchNames.ID_MAP.get(sw), type == 'O');
    }

    @Override
    public void handleButton(String name) {
        int cond = Integer.valueOf(name.substring(1));
        latchResetBtn[cond].setEnabled(false);
        sender.sendCommand(null, "clearLatch", cond);
    }

    @Override
    public void handleTextFieldX(String name, Object value) {
        periodTextField.setEnabled(false);
        sender.sendCommand(null, "setUpdatePeriod", (int)(1000 * (Double)value));
    }

    class UpdateVacState implements Runnable {

        private final VacSysState vs;

        UpdateVacState(VacSysState vs) {
            this.vs = vs;
        }

        @Override
        public void run() {
            sysStateValue.setText("RUNNING");
            sysStateValue.setForeground(UiConstants.GREEN);
            PLCState plcState = vs.getPlcState();
            plcStateValue.setText(plcState.name());
            plcStateValue.setForeground(plcState == PLCState.ALIVE ? UiConstants.GREEN :
                                        plcState == PLCState.OFFLINE ? UiConstants.BLUE : UiConstants.RED);
            plcStateValue.setEnabled(true);
            periodTextField.update(vs.getTickMillis() / 1000.0, true);
            int switches = 0;
            for (int sw = 0; sw < VacSysState.NUM_SWITCHES; sw++) {
                if (!vs.hasSwitch(sw)) continue;
                switches |= (1 << sw);
                SwitchState state = vs.getSwitchState(sw);
                String text = state.name();
                Color color = state == SwitchState.OFF ? Color.black : state == SwitchState.ON ? UiConstants.GREEN : UiConstants.BLUE;
                if (state != SwitchState.OFFLINE) {
                    if (vs.getSwitchEnable(sw) == SwitchEnable.OFF) {
                        text = "DISABLD";
                        color = UiConstants.RED;
                    }
                    else {
                        if (switchTypes[sw] == SWTP_VALVE) {
                            text = state == SwitchState.OFF ? "SHUT" : state == SwitchState.ON ? "OPEN" : text;
                        }
                        DeviceState devState = vs.getDeviceState(sw);
                        if (devState != null) {
                            text = devState.name();
                            if (devState == DeviceState.FAILED) {
                                color = UiConstants.RED;
                            }
                        }
                    }
                }
                switchStatus[sw].setText(text);
                switchStatus[sw].setForeground(color);

                JRadioButton selButton = state == SwitchState.ON ? switchOnRB[sw] : switchOffRB[sw];
                selButton.setSelected(true);
                switchStatus[sw].setEnabled(true);
                switchOffRB[sw].setEnabled(true);
                switchOnRB[sw].setEnabled(true);
            }
            if (switches != switchMask) {
                updateSwitchLayout(switches);
            }
            int conditions = 0;
            for (int cond = 0; cond < VacSysState.NUM_LATCHES; cond++) {
                if (!vs.hasLatch(cond)) continue;
                conditions |= (1 << cond);
                LatchState state = vs.getLatch(cond);
                latchStatus[cond].setText(state.name());
                latchStatus[cond].setForeground(state == LatchState.OFFLINE ? UiConstants.BLUE :
                                                state == LatchState.CLEAR ? UiConstants.GREEN :
                                                state == LatchState.ACTIVE ? UiConstants.RED : UiConstants.PURPLE);
                latchStatus[cond].setEnabled(true);
                latchResetBtn[cond].setEnabled(true);
            }
            if (conditions != latchMask) {
                updateLatchLayout(conditions);
            }
            conditions = 0;
            for (int cond = 0; cond < VacSysState.NUM_CONDITIONS; cond++) {
                if (!vs.hasCondition(cond)) continue;
                conditions |= (1 << cond);
                ConditionState state = vs.getCondition(cond);
                conditionStatus[cond].setText(state.name());
                conditionStatus[cond].setForeground(state == ConditionState.OFF ? UiConstants.BLUE :
                                                    state == ConditionState.NO ? Color.BLACK : UiConstants.GREEN);
                conditionStatus[cond].setEnabled(true);
            }
            if (conditions != conditionMask) {
                updateConditionLayout(conditions);
            }
        }

    }

    class DisablePanel implements Runnable {

        @Override
        public void run() {
            sysStateValue.setText("STOPPED");
            sysStateValue.setForeground(UiConstants.RED);
            plcStateValue.setEnabled(false);
            periodTextField.setDisabled();
            for (int sw = 0; sw < VacSysState.NUM_SWITCHES; sw++) {
                switchStatus[sw].setEnabled(false);
                switchOffRB[sw].setEnabled(false);
                switchOnRB[sw].setEnabled(false);
            }
            for (int cond = 0; cond < VacSysState.NUM_LATCHES; cond++) {
                latchStatus[cond].setEnabled(false);
                latchResetBtn[cond].setEnabled(false);
            }
            for (int cond = 0; cond < VacSysState.NUM_CONDITIONS; cond++) {
                conditionStatus[cond].setEnabled(false);
            }
        } 
    }

    private static final long serialVersionUID = 1L;
}
