package org.lsst.ccs.subsystem.vacuum.ui;

import java.awt.image.BufferedImage;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import javax.imageio.ImageIO;
import static javax.swing.JComponent.*;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.gconsole.plugins.monitor.CellTableView;
import org.lsst.ccs.gconsole.plugins.monitor.ChannelHandle;
import org.lsst.ccs.gconsole.plugins.monitor.DefaultChannelHandle;
import org.lsst.ccs.gconsole.plugins.monitor.FormattedValue;
import org.lsst.ccs.gconsole.plugins.monitor.ImageView;
import org.lsst.ccs.gconsole.plugins.monitor.MonitorCell;
import org.lsst.ccs.gconsole.plugins.monitor.MonitorField;

/**
 * Displays the vacuum plate schematics.
 *
 * @author onoprien
 */
public class VacuumPage {

// -- Fields : -----------------------------------------------------------------
    
    private final AgentInfo agent;
    
    // Gauges:
    
    private final String VHX_UTT_GCC_00 = "";
    private final String VCR_UTT_GCC_00 = "main/CryoVacGauge/CryoVac";
    private final String VHX_UTT_GCC_01 = "";
    private final String VCR_UTT_GHF_00 = "main/TurboVacGauge/TurboVac";
    private final String VCR_UTT_GCC_02 = "";
    
    // Valves:
    
    private final String VHX_UUT_VGC_01 = "";
    private final String VCR_UTT_VGC_00 = "";
    private final String L3_O_RING = "";
    private final String L3_HOUSING_0_RING = "";
    private final String FEEDTHRU_RING_O_RING = "";
    private final String PUMP_PLATE_0_RING = "";
    
    // pumps:
    
    private final String[] VCR_UTT_PIP_00 = {"main/IonPumps/CIP1_V", "main/IonPumps/CIP1_I"};
    private final String[] VCR_UTT_PIP_01 = {"main/IonPumps/CIP2_V", "main/IonPumps/CIP2_I"};
    private final String[] VCR_UTT_PIP_02 = {"main/IonPumps/CIP3_V", "main/IonPumps/CIP3_I"};
    private final String[] VHX_UTT_PIP_03 = {"", ""};
    private final String[] VHX_UTT_PIP_04 = {"", ""};
    private final String[] VCR_UTT_PIP_05 = {"", ""};
    private final String[] VCR_UTT_PIP_06 = {"main/IonPumps/CIP4_V", "main/IonPumps/CIP4_I"};
    private final String[] VCR_UTT_PIP_07 = {"main/IonPumps/CIP5_V", "main/IonPumps/CIP5_I"};
    private final String[] VCR_UTT_PIP_08 = {"main/IonPumps/CIP6_V", "main/IonPumps/CIP6_I"};
    
    // rpm:
    
    private final String VHX_UTT_PTM_01 = "";
    private final String VCR_UTT_PTM_00 = "main/TurboPump/TurboSpeed";
    
    private ImageView view;
    
    private final MonitorField gaugeField, maField, lsField, rpmField, valveField;

// -- Life cycle : -------------------------------------------------------------
    
    public VacuumPage(AgentInfo agent) {
        
        this.agent = agent;
        
        FormattedValue fv = new FormattedValue();
        fv.horizontalAlignment = SwingUtilities.CENTER;
        fv.format = "<html><center>%1$6.1e<br>Torr";
        gaugeField = new MonitorField(MonitorField.VALUE.getKey(), null, fv, MonitorField.VALUE.getKeys());
        fv.format = "%1$6.2f ma";
        maField = new MonitorField(MonitorField.VALUE.getKey(), null, fv, MonitorField.VALUE.getKeys());
        fv.format = "%1$7.1f V";
        lsField = new MonitorField(MonitorField.VALUE.getKey(), null, fv, MonitorField.VALUE.getKeys());
        fv.format = "%1$6.0f rpm";
        rpmField = new MonitorField(MonitorField.VALUE.getKey(), null, fv, MonitorField.VALUE.getKeys());
        fv.format = "open";  //FIXME
        valveField = new MonitorField(MonitorField.VALUE.getKey(), null, fv, MonitorField.VALUE.getKeys());
        
    }
    
    /**
     * Creates a {@code MonitorView} that displays the vacuum plate schematics.
     * @return {@code MonitorView} that displays the vacuum plate schematics.
     */
    public ImageView getView() {
        
        if (view != null) return view;
        
        try {
            
            BufferedImage back = ImageIO.read(ImageView.class.getResource("/org/lsst/ccs/subsystem/vacuum/ui/vacuum.png"));
            view = new ImageView("Vacuum", back, .5, 2.);
            
            
            placeGauge(0.4231, 0.0820, VHX_UTT_GCC_00);
            placeGauge(0.3739, 0.2730, VCR_UTT_GCC_00);
            placeGauge(0.5572, 0.2570, VHX_UTT_GCC_01);
            placeGauge(0.5203, 0.4696, VCR_UTT_GHF_00);
            placeGauge(0.5510, 0.8048, VCR_UTT_GCC_02);
            
            placeValve(0.5055, 0.3338, VHX_UUT_VGC_01, CENTER_ALIGNMENT, BOTTOM_ALIGNMENT);
            placeValve(0.3469, 0.4936, VCR_UTT_VGC_00, CENTER_ALIGNMENT, BOTTOM_ALIGNMENT);
            placeValve(0.0701, 0.8628, L3_O_RING, RIGHT_ALIGNMENT, CENTER_ALIGNMENT);
            placeValve(0.1562, 0.8586, L3_HOUSING_0_RING, RIGHT_ALIGNMENT, CENTER_ALIGNMENT);
            placeValve(0.2312, 0.8614, FEEDTHRU_RING_O_RING, RIGHT_ALIGNMENT, CENTER_ALIGNMENT);
            placeValve(0.3137, 0.8586, PUMP_PLATE_0_RING, RIGHT_ALIGNMENT, CENTER_ALIGNMENT);
            
            placeMa(0.5351, 0.1655, VHX_UTT_PIP_04, LEFT_ALIGNMENT, CENTER_ALIGNMENT);
            placeMa(0.5166, 0.6393, VHX_UTT_PIP_03, LEFT_ALIGNMENT, CENTER_ALIGNMENT);
            placeMa(0.4096, 0.7624, VCR_UTT_PIP_05, CENTER_ALIGNMENT, TOP_ALIGNMENT);
            
            placeRpm(0.5879, 0.3585, VHX_UTT_PTM_01);
            placeRpm(0.5855, 0.5203, VCR_UTT_PTM_00);
            
            // Ion pumps average
            
            FormattedValue fv = new FormattedValue();
            fv.horizontalAlignment = SwingUtilities.CENTER;
            fv.format = "%2$d x %1$7.1f V";
            MonitorField ipVField = new MonitorField.Average(MonitorField.VALUE.getKey(), null, fv, MonitorField.VALUE.getKeys());
            fv.format = "%1$6.2f mA";
            MonitorField ipIField = new MonitorField.Average(MonitorField.VALUE.getKey(), null, fv, MonitorField.VALUE.getKeys());
            
            String[][] pumps = new String[][] {VCR_UTT_PIP_00, VCR_UTT_PIP_01, VCR_UTT_PIP_02, VCR_UTT_PIP_06, VCR_UTT_PIP_07, VCR_UTT_PIP_08};
            List<ChannelHandle> chV = new ArrayList<>(pumps.length);
            List<ChannelHandle> chI = new ArrayList<>(pumps.length);
            for (String[] ss : pumps) {
                chV.add(new DefaultChannelHandle(agent.getName() + "/" + ss[0], null, null));
                chI.add(new DefaultChannelHandle(agent.getName() + "/" + ss[1], null, null));
            }
            MonitorCell c0 = new MonitorCell(chV, ipVField);
            MonitorCell c1 = new MonitorCell(chI, ipIField);

            CellTableView table = new CellTableView(2, 1, c0, c1);
            ImageView.Item item = new ImageView.Item(table);
            item.setPosition(0.3124, 0.3946);
            item.setAlignment(LEFT_ALIGNMENT, CENTER_ALIGNMENT);
            item.setMaxSize(0.2420, 0.0820);
            view.add(item);
            
        } catch (IOException e) {
        }
        return view;
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private void placeGauge(double x, double y, String ch) throws IOException {

        MonitorCell c = new MonitorCell(new DefaultChannelHandle(agent.getName() +"/"+ ch, null, null), gaugeField);
        
        CellTableView table = new CellTableView(1, 1, c);
        ImageView.Item item = new ImageView.Item(table);
        item.setPosition(x, y);
        item.setAlignment(CENTER_ALIGNMENT, CENTER_ALIGNMENT);
//        item.setSize(0.06, 0.07);
        item.setMaxSize(0.085, 0.055);
        view.add(item);
    }
    
    private void placeValve(double x, double y, String ch, float horAlign, float vertAlign) throws IOException {

        MonitorCell c = new MonitorCell(new DefaultChannelHandle(agent.getName() +"/"+ ch, null, null), valveField);
        
        CellTableView table = new CellTableView(1, 1, c);
        ImageView.Item item = new ImageView.Item(table);
        item.setPosition(x, y);
        item.setAlignment(horAlign, vertAlign);
//        item.setSize(0.06, 0.07);
        item.setMaxSize(0.2420, 0.0820);
        view.add(item);
    }
    
    private void placeMa(double x, double y, String[] ch, float horAlign, float vertAlign) throws IOException {

        MonitorCell c0 = new MonitorCell(new DefaultChannelHandle(agent.getName() +"/"+ ch[0], null, null), maField);
        MonitorCell c1 = new MonitorCell(new DefaultChannelHandle(agent.getName() +"/"+ ch[1], null, null), lsField);
        
        CellTableView table = new CellTableView(2, 1, c0, c1);
        ImageView.Item item = new ImageView.Item(table);
        item.setPosition(x, y);
        item.setAlignment(horAlign, vertAlign);
//        item.setSize(0.06, 0.07);
        item.setMaxSize(0.2420, 0.0820);
        view.add(item);
    }
    
    private void placeRpm(double x, double y, String ch) throws IOException {

        MonitorCell c = new MonitorCell(new DefaultChannelHandle(agent.getName() +"/"+ ch, null, null), rpmField);
        
        CellTableView table = new CellTableView(1, 1, c);
        ImageView.Item item = new ImageView.Item(table);
        item.setPosition(x, y);
        item.setAlignment(LEFT_ALIGNMENT, CENTER_ALIGNMENT);
//        item.setSize(0.06, 0.07);
        item.setMaxSize(0.1132, 0.0382);
        view.add(item);
    }

}
