package org.lsst.ccs.subsystem.vacuum.data;

import org.lsst.ccs.subsystem.vacuum.constants.ConditionState;
import org.lsst.ccs.subsystem.vacuum.constants.DeviceState;
import org.lsst.ccs.subsystem.vacuum.constants.SwitchState;
import org.lsst.ccs.subsystem.vacuum.constants.SwitchEnable;
import java.io.Serializable;

/**
 *  Contains the state of the vacuum subsystem
 *
 *  @author Owen Saxton
 */
public class VacuumState implements Serializable {
    
    public static final String KEY = "VacuumState";
    public static final int
        SW_CRYO_TURBO     = 0,
        SW_CRYO_ION_PUMP1 = 1,
        SW_CRYO_ION_PUMP2 = 2,
        SW_CRYO_ION_PUMP3 = 3,
        SW_CRYO_ION_PUMP4 = 4,
        SW_CRYO_ION_PUMP5 = 5,
        SW_CRYO_ION_PUMP6 = 6,
        SW_CRYO_VALVE     = 7,
        SW_HX_TURBO       = 8,
        SW_HX_ION_PUMP1   = 9,
        SW_HX_ION_PUMP2   = 10,
        SW_HX_VALVE       = 11,
        SW_OR_ION_PUMP    = 12,
        SW_OR_FPP_VALVE   = 13,
        SW_OR_FH_VALVE    = 14,
        SW_OR_L3H_VALVE   = 15,
        SW_OR_L3_VALVE    = 16,
        NUM_SWITCHES      = 17;
    public static final int
        COND_CR_VACUUM   = 0,
        COND_CR_GATE_NFC = 1,
        COND_CR_GATE_AO  = 2,
        COND_CR_PUMP     = 3,
        COND_HX_VACUUM   = 4,
        COND_HX_GATE_NFC = 5,
        COND_HX_GATE_AO  = 6,
        COND_HX_PUMP     = 7,
        NUM_CONDITIONS   = 8;

    private int tickMillis;        // Data status update period (ms)
    private int switchMask;        // Bit mask of defined switches
    private int conditionMask;     // Bit mask of defined conditions
    private final SwitchState[] switchState = new SwitchState[NUM_SWITCHES];  // Switch state array
    private final SwitchEnable[] switchEnable = new SwitchEnable[NUM_SWITCHES];  // Switch enable array
    private final DeviceState[] deviceState = new DeviceState[NUM_SWITCHES];  // Extra device state array
    private final ConditionState[] conditionState = new ConditionState[NUM_CONDITIONS];
    
    public void setTickMillis(int value) {
        tickMillis = value;
    }

    public int getTickMillis() {
        return tickMillis;
    }

    public void addSwitch(int sw) {
        switchMask |= (1 << sw);
    }

    public boolean hasSwitch(int sw) {
        return (switchMask & (1 << sw)) != 0;
    }

    public void setSwitchState(int sw, SwitchState state) {
        switchState[sw] = state;
    }

    public SwitchState getSwitchState(int sw) {
        return switchState[sw];
    }

    public void setSwitchEnable(int sw, SwitchEnable enable) {
        switchEnable[sw] = enable;
    }

    public SwitchEnable getSwitchEnable(int sw) {
        return switchEnable[sw];
    }

    public void setDeviceState(int sw, DeviceState state) {
        deviceState[sw] = state;
    }

    public DeviceState getDeviceState(int sw) {
        return deviceState[sw];
    }

    public void addCondition(int cond) {
        conditionMask |= (1 << cond);
    }

    public boolean hasCondition(int cond) {
        return (conditionMask & (1 << cond)) != 0;
    }

    public void setCondition(int cond, ConditionState state) {
        conditionState[cond] = state;
    }

    public ConditionState getCondition(int cond) {
        return conditionState[cond];
    }

    private static final long serialVersionUID = -838278252519832878L;
}
