package org.lsst.ccs.subsystem.utility;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.localdb.trendserver.TrendingClientService;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.common.MonitorTaskControl;
import org.lsst.ccs.subsystem.common.actions.MpmAction;
import org.lsst.ccs.subsystem.utility.constants.FanState;
import org.lsst.ccs.subsystem.utility.constants.HeaterState;
import org.lsst.ccs.subsystem.utility.constants.MonitorControl;
import org.lsst.ccs.subsystem.utility.constants.UtilTrunkAlert;
import org.lsst.ccs.subsystem.utility.constants.UtilTrunkFans;
import org.lsst.ccs.subsystem.utility.constants.UtilTrunkHeaters;
import org.lsst.ccs.subsystem.utility.constants.UtilTrunkSwitches;
import org.lsst.ccs.subsystem.utility.constants.UtilTrunkValves;
import org.lsst.ccs.subsystem.utility.constants.UtilityAgentProperties;
import org.lsst.ccs.subsystem.utility.constants.ValveState;
import org.lsst.ccs.subsystem.utility.constants.VpcControlState;
import org.lsst.ccs.subsystem.utility.data.UtilTrunkState;
import org.lsst.ccs.subsystem.utility.data.UtilityException;

/**
 *  Implements the utility trunk subsystem.
 *
 *  @author Owen Saxton
 */
public class UtilTrunkMain extends Subsystem implements HasLifecycle {

    private static final int[] devNums = new int[UtilTrunkSwitches.NUM_SWITCHES];
    static {
        devNums[UtilTrunkSwitches.SW_BFR_HEATER_1] = UtilTrunkSwitches.DEVC_BFR;
        devNums[UtilTrunkSwitches.SW_BFR_HEATER_2] = UtilTrunkSwitches.DEVC_BFR;
        devNums[UtilTrunkSwitches.SW_PDU_MPC_FAN] = UtilTrunkSwitches.DEVC_PDU_48V;
    }
    private static final int[] swNums = new int[UtilTrunkSwitches.NUM_SWITCHES];
    static {
        swNums[UtilTrunkSwitches.SW_BFR_HEATER_1] = BfrDeviceUT.CHAN_HEATER_1;
        swNums[UtilTrunkSwitches.SW_BFR_HEATER_2] = BfrDeviceUT.CHAN_HEATER_2;
        swNums[UtilTrunkSwitches.SW_PDU_MPC_FAN] = Pdu48VDeviceUT.CHAN_MPC_FAN;
    }
    private static final Map<Integer, String> typeMap = new HashMap<>();
    static {
        typeMap.put(UtilTrunkSwitches.DEVC_BFR, "BFR");
        typeMap.put(UtilTrunkSwitches.DEVC_PDU_48V, "48V PDU");
    }

    private static final double
        VALVE_POSN_TOL = 0.01,
        RPM_ERROR_WARN = 0.05,
        RPM_ERROR_ALARM = 0.1;
    private static final int
        VALVE_SETTLE_TIME = 12000,  // msec
        FAN_SETTLE_TIME = 2000,     // msec
        ALERT_NOMINAL = 0,
        ALERT_WARNING = 1,
        ALERT_ALARM = 2,
        NUM_UT_TEMPS = 9;

    @LookupName
    String name;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService propertiesService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private ConfigurationService configService;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private final List<SwitchControl> swDeviceList = new ArrayList<>();
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private VpcControlUT vpcControl;
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private Maq20DeviceUT utControl;

    @ConfigurationParameter(category="Util")
    private volatile double maxTemperature;

    private FanPIControl utFan, vpcFan, mpcFan;
    private Channel utValvePosn, vpcValvePosn, mpcValvePosn;
    private Channel utFanSpeed, vpcFanSpeed, mpcFanSpeed;
    private Channel[] utTemps;

    private static final Logger LOG = Logger.getLogger(UtilTrunkMain.class.getName());
    private final SwitchControl[] swDevices = new SwitchControl[UtilTrunkSwitches.NUM_DEVICES];
    private final UtilTrunkState utilTrunkState = new UtilTrunkState();
    private final FanPIControl[] fanControl = new FanPIControl[UtilTrunkFans.NUM_FANS];
    private final Channel[] valvePosnChans = new Channel[UtilTrunkValves.NUM_VALVES];
    private final boolean [] valvePosnBad = new boolean[UtilTrunkValves.NUM_VALVES];
    private final Channel[] fanSpeedChans = new Channel[UtilTrunkFans.NUM_FANS];
    private final int[] fanSpeedAlerts = new int[UtilTrunkFans.NUM_FANS];
    private boolean utOverTemp = false, gotCommand = false;
    private MonitorTaskControl monitorControl;
    private int alarmDelay = 1;


    /**
     *  Constructor.
     */
    public UtilTrunkMain() {
        super("utiltrunk", AgentInfo.AgentType.WORKER);
        System.setProperty("org.lsst.ccs.agent." + TrendingClientService.USES_TRENDING_SERVICE, "true");
    }


    /**
     *  Build phase
     */
    @Override
    public void build() {
        // Create the monitor task control object and node
        monitorControl = MonitorTaskControl.createNode(this, MonitorControl.NODE_NAME);

        // Create and schedule an AgentPeriodicTask to update and publish the utility trunk system state
        AgentPeriodicTask pt = new AgentPeriodicTask("UT-state", () -> updateState()).withPeriod(Duration.ofMillis(1000));
        periodicTaskService.scheduleAgentPeriodicTask(pt);

        // Create and schedule an AgentPeriodicTask to check for alarm conditions
        pt = new AgentPeriodicTask("UT-alarms", () -> checkForAlarms()).withPeriod(Duration.ofMillis(1000));
        periodicTaskService.scheduleAgentPeriodicTask(pt);
    }

    /**
     * Init phase.
     * Register the Alerts raised by this subsystem.
     */
    @Override
    public void init() {
        for (int id = 0; id < UtilTrunkFans.NUM_FANS; id++) {
            alertService.registerAlert(UtilTrunkAlert.FAN_SPEED_INCORRECT.newAlert(UtilTrunkFans.getName(id)));
        }
        for (int id = 0; id < UtilTrunkValves.NUM_VALVES; id++) {
            alertService.registerAlert(UtilTrunkAlert.VALVE_POSN_INCORRECT.newAlert(UtilTrunkValves.getName(id)));
        }
        alertService.registerAlert(UtilTrunkAlert.OVER_TEMPERATURE.newAlert());
    }
    

    /**
     *  Subsystem post-initialization
     */
    @Override
    public void postInit()
    {
        // Set a property to define this Agent's type.
        propertiesService.setAgentProperty(UtilityAgentProperties.UTIL_TRUNK_TYPE, getClass().getCanonicalName());

        // Check power devices and control channels
        for (SwitchControl sd : swDeviceList) {
            swDevices[sd.getSwitchDevice()] = sd;
        }
        for (int j = 0; j < swDevices.length; j++) {
            if (swDevices[j] == null) {
                ErrorUtils.reportConfigError(LOG, name, typeMap.get(j), "has not been defined");
            }
        }

        // Check fan controllers
        if (utFan == null) {
            ErrorUtils.reportConfigError(LOG, name, "utFan", "has not been specified");
        }
        fanControl[UtilTrunkFans.FAN_UT_ID] = utFan;
        if (vpcFan == null) {
            ErrorUtils.reportConfigError(LOG, name, "vpcFan", "has not been specified");
        }
        fanControl[UtilTrunkFans.FAN_VPC_ID] = vpcFan;
        if (mpcFan == null) {
            ErrorUtils.reportConfigError(LOG, name, "mpcFan", "has not been specified");
        }
        fanControl[UtilTrunkFans.FAN_MPC_ID] = mpcFan;

        // Check fan speed readback channels
        if (utFanSpeed == null) {
            ErrorUtils.reportConfigError(LOG, name, "utFanSpeed", "has not been specified");
        }
        fanSpeedChans[UtilTrunkFans.FAN_UT_ID] = utFanSpeed;
        if (vpcFanSpeed == null) {
            ErrorUtils.reportConfigError(LOG, name, "vpcFanSpeed", "has not been specified");
        }
        fanSpeedChans[UtilTrunkFans.FAN_VPC_ID] = vpcFanSpeed;
        if (mpcFanSpeed == null) {
            ErrorUtils.reportConfigError(LOG, name, "mpcFanSpeed", "has not been specified");
        }
        fanSpeedChans[UtilTrunkFans.FAN_MPC_ID] = mpcFanSpeed;

        // Check valve position readback channels
        if (utValvePosn == null) {
            ErrorUtils.reportConfigError(LOG, name, "utValvePosn", "has not been specified");
        }
        valvePosnChans[UtilTrunkValves.VALVE_UT_ID] = utValvePosn;
        if (vpcValvePosn == null) {
            ErrorUtils.reportConfigError(LOG, name, "vpcValvePosn", "has not been specified");
        }
        valvePosnChans[UtilTrunkValves.VALVE_VPC_ID] = vpcValvePosn;
        if (mpcValvePosn == null) {
            ErrorUtils.reportConfigError(LOG, name, "mpcValvePosn", "has not been specified");
        }
        valvePosnChans[UtilTrunkValves.VALVE_MPC_ID] = mpcValvePosn;

        // Check temperature channels
        if (utTemps == null) {
            ErrorUtils.reportConfigError(LOG, name, "utTemps", "has not been specified");
        }
        if (utTemps.length != NUM_UT_TEMPS) {
            ErrorUtils.reportConfigError(LOG, name, "utTemps", "must have exactly " + NUM_UT_TEMPS + " elements");
        }
        for (Channel tempChan : utTemps) {
            Double value = Double.valueOf(configService.getConfigurationParameterValue(tempChan.getPath(), "limitHi"));
            if (value != maxTemperature) {
                LOG.log(Level.WARNING, "Temperature channel {0} high limit ({1}) is not equal to maxTemperature ({2})",
                        new Object[]{tempChan.getPath(), value, maxTemperature});
            }
        }

        // Check other controls
        if (vpcControl == null) {
            ErrorUtils.reportConfigError(LOG, name, "VpcControlUT component", "has not been defined");
        }
        if (utControl == null) {
            ErrorUtils.reportConfigError(LOG, name, "Maq20DeviceUT component", "has not been defined");
        }

        // Initialize VPC heater values (temperatures)
        utilTrunkState.setHeaterValue(UtilTrunkHeaters.HEATER_VPC1_ID, vpcControl.getDeltaTemp(UtilTrunkHeaters.HEATER_VPC1_ID));
        utilTrunkState.setHeaterValue(UtilTrunkHeaters.HEATER_VPC2_ID, vpcControl.getDeltaTemp(UtilTrunkHeaters.HEATER_VPC2_ID));

        Arrays.fill(valvePosnBad, false);
        Arrays.fill(fanSpeedAlerts, ALERT_NOMINAL);
    }


    /**
     *  Starts the subsystem.
     */
    @Override
    public void postStart()
    {
        publishState();        // For any GUIs
        LOG.info("Utility trunk subsystem started");
    }


    /**
     *  Gets the subsystem state.
     *
     *  @return  The full PDU system state
     */
    @Command(type=CommandType.QUERY, description="Get the full state")
    public UtilTrunkState getFullState()
    {
        utilTrunkState.setTickMillis(monitorControl.getFastPeriod());
        return utilTrunkState;
    }


    /**
     *  Gets the list of fan names.
     * 
     *  @return  The list of names
     */
    @Command(type=Command.CommandType.QUERY, description="Get list of fan names")
    public List<String> getFanNames()
    {
        return UtilTrunkFans.getNames();
    }


    /**
     *  Sets a fan state (using its name).
     *
     *  @param  name   The fan name
     *  @param  state  The state to set
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Set a fan control state")
    public void setFanState(@Argument(description="The fan name") String name,
                            @Argument(description="The control state") FanState state) throws UtilityException
    {
        gotCommand = true;
        int fanId = UtilTrunkFans.getId(name);
        if (state == FanState.OFFLINE || (state == FanState.TEMP && !UtilTrunkFans.hasTempState(fanId))) {
            throw new UtilityException("Invalid " + name + " state: " + state);
        }
        synchronized (utilTrunkState) {
            if (state == utilTrunkState.getFanBaseState(fanId)) return;
            utilTrunkState.setFanBaseState(fanId, state);
            if (utilTrunkState.getFanState(fanId) == FanState.OFFLINE) return;
            utilTrunkState.setFanState(fanId, state);
            FanPIControl ctrlr = fanControl[fanId];
            if (state == FanState.TEMP) {
                ctrlr.setTemperature(utilTrunkState.getDeltaTemp(fanId));
                ctrlr.startLoop(utilTrunkState.getFanSpeed(fanId));
            }
            else {
                if (UtilTrunkFans.hasTempState(fanId)) {
                    ctrlr.stopLoop();
                }
                ctrlr.setSpeed(state == FanState.SPEED ? utilTrunkState.getFanSpeed(fanId) : 0.0);
            }
        }
    }


    /**
     *  Sets a fan speed (using its name).
     *
     *  @param  name   The fan name
     *  @param  speed  The fan speed
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Set a fan speed")
    public void setFanSpeed(@Argument(description="The fan name") String name,
                            @Argument(description="The fan speed") double speed) throws UtilityException
    {
        gotCommand = true;
        int fanId = UtilTrunkFans.getId(name);
        synchronized (utilTrunkState) {
            utilTrunkState.setFanSpeed(fanId, speed);
            if (utilTrunkState.getFanState(fanId) == FanState.SPEED) {
                fanControl[fanId].setSpeed(speed);
            }
        }
    }


    /**
     *  Sets the VPC control state.
     *
     *  @param  state  The state to set
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Set the VPC control state")
    public void setVpcState(@Argument(description="The control state") VpcControlState state) throws UtilityException
    {
        gotCommand = true;
        synchronized (utilTrunkState) {
            if (state == utilTrunkState.getVpcState()) return;
            utilTrunkState.setVpcState(state);
            HeaterState h1State = utilTrunkState.getHeaterState(UtilTrunkHeaters.HEATER_VPC1_ID);
            HeaterState h2State = utilTrunkState.getHeaterState(UtilTrunkHeaters.HEATER_VPC2_ID);
            if (state == VpcControlState.TEMP) {
                if (h1State != HeaterState.OFFLINE) {
                    utilTrunkState.setHeaterState(UtilTrunkHeaters.HEATER_VPC1_ID, HeaterState.TEMP);
                }
                if (h2State != HeaterState.OFFLINE) {
                    utilTrunkState.setHeaterState(UtilTrunkHeaters.HEATER_VPC2_ID, HeaterState.TEMP);
                }
                vpcControl.startLoop();
            }
            else {
                vpcControl.stopLoop();
                if (h1State != HeaterState.OFFLINE) {
                    utilTrunkState.setHeaterState(UtilTrunkHeaters.HEATER_VPC1_ID,
                                                  utilTrunkState.getHeaterBaseState(UtilTrunkHeaters.HEATER_VPC1_ID));
                    operateHeater(UtilTrunkHeaters.HEATER_VPC1_ID);
                }
                if (h2State != HeaterState.OFFLINE) {
                    utilTrunkState.setHeaterState(UtilTrunkHeaters.HEATER_VPC2_ID,
                                                  utilTrunkState.getHeaterBaseState(UtilTrunkHeaters.HEATER_VPC2_ID));
                    operateHeater(UtilTrunkHeaters.HEATER_VPC2_ID);
                }
            }
        }
    }


    /**
     *  Gets the list of heater names.
     * 
     *  @return  The list of names
     */
    @Command(type=Command.CommandType.QUERY, description="Get list of heater names")
    public List<String> getHeaterNames()
    {
        return UtilTrunkHeaters.getNames();
    }


    /**
     *  Sets a heater control state (using its name).
     *
     *  @param  name   The heater name
     *  @param  state  The state to set
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Set a heater control state")
    public void setHeaterState(@Argument(description="The heater name") String name,
                               @Argument(description="The control state") HeaterState state) throws UtilityException
    {
        gotCommand = true;
        int htrId = UtilTrunkHeaters.getId(name);
        if (state == HeaterState.OFFLINE || state == HeaterState.TEMP) {
            throw new UtilityException("Invalid " + name + " state: " + state);
        }
        synchronized (utilTrunkState) {
            if (state == utilTrunkState.getHeaterBaseState(htrId)) return;
            utilTrunkState.setHeaterBaseState(htrId, state);
            HeaterState currState = utilTrunkState.getHeaterState(htrId);
            if (currState == HeaterState.OFFLINE || currState == HeaterState.TEMP) return;
            utilTrunkState.setHeaterState(htrId, state);
            operateHeater(htrId);
        }
    }


    /**
     *  Sets a heater value (using its name).
     *
     *  @param  name   The heater name
     *  @param  value  The heater value (power duty cycle or delta temperature)
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Set a heater power")
    public void setHeaterValue(@Argument(description="The heater name") String name,
                               @Argument(description="The value") double value) throws UtilityException
    {
        gotCommand = true;
        int htrId = UtilTrunkHeaters.getId(name);
        synchronized (utilTrunkState) {
            utilTrunkState.setHeaterValue(htrId, value);
            if (utilTrunkState.getHeaterState(htrId) == HeaterState.TEMP) {
                vpcControl.setDeltaTemp(htrId, value);
            }
        }
    }


    /**
     *  Gets the list of valve names.
     * 
     *  @return  The list of names
     */
    @Command(type=Command.CommandType.QUERY, description="Get list of valve names")
    public List<String> getValveNames()
    {
        return UtilTrunkValves.getNames();
    }


    /**
     *  Sets a valve state (using its name).
     *
     *  @param  name   The valve name
     *  @param  state  The state to set
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Set a valve control state")
    public void setValveState(@Argument(description="The valve name") String name,
                              @Argument(description="The control state") ValveState state) throws UtilityException
    {
        gotCommand = true;
        int valveId = UtilTrunkValves.getId(name);
        if (state == ValveState.OFFLINE || state == ValveState.TEMP) {
            throw new UtilityException("Invalid " + name + " valve state: " + state);
        }
        synchronized (utilTrunkState) {
            if (state == utilTrunkState.getValveBaseState(valveId)) return;
            utilTrunkState.setValveBaseState(valveId, state);
            if (utilTrunkState.getValveState(valveId) == ValveState.OFFLINE) return;
            utilTrunkState.setValveState(valveId, state);
            operateValve(valveId);
        }
    }


    /**
     *  Sets a valve position (using its name).
     *
     *  @param  name   The valve name
     *  @param  position  The valve position
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Set a valve position")
    public void setValvePosition(@Argument(description="The valve name") String name,
                                 @Argument(description="The valve position") double position) throws UtilityException
    {
        gotCommand = true;
        int valveId = UtilTrunkValves.getId(name);
        synchronized (utilTrunkState) {
            utilTrunkState.setValvePosition(valveId, position);
            operateValve(valveId);
        }
    }


    /**
     *  Sets a fan delta temperature (using its name).
     *
     *  @param  name  The fan name
     *  @param  temp  The delta temperature to set
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Set a delta temperature")
    public void setFanDeltaTemp(@Argument(description="The fan name") String name,
                                @Argument(description="The delta temperature") double temp) throws UtilityException
    {
        gotCommand = true;
        int fanId = UtilTrunkFans.getId(name);
        synchronized (utilTrunkState) {
            utilTrunkState.setDeltaTemp(fanId, temp);
            if (utilTrunkState.getFanState(fanId) == FanState.TEMP) {
                if (fanId != UtilTrunkFans.FAN_VPC_ID) {
                    fanControl[fanId].setTemperature(temp);
                }
            }
        }
    }


    /**
     *  Turns a switch on or off using its ID.
     *
     *  @param  sw  The switch ID.
     *  @param  on  Whether to turn on or off
     *  @throws  UtilityException
     */
    private void setSwitchOn(int sw, boolean on) throws UtilityException
    {
        if (sw < 0 || sw >= UtilTrunkSwitches.NUM_SWITCHES) {
            throw new UtilityException("Invalid switch number: " + sw);
        }
        if (on) {
            swDevices[devNums[sw]].switchOn(swNums[sw]);
        }
        else {
            swDevices[devNums[sw]].switchOff(swNums[sw]);
        }
    }


    /**
     *  Operates a heater.
     *
     *  @param  htrId  The heater ID
     *  @throws  UtitliyException
     */
    private void operateHeater(int htrId) throws UtilityException
    {
        HeaterState state = utilTrunkState.getHeaterState(htrId);
        if (state == HeaterState.OFFLINE || state == HeaterState.TEMP) return;
        setSwitchOn(UtilTrunkHeaters.getSwitch(htrId), state == HeaterState.ON);
    }


    /**
     *  Operates a coolant valve.
     *
     *  @param  valveId  The valve ID
     *  @throws  UtilityException
     */
    private void operateValve(int valveId) throws UtilityException
    {
        ValveState state = utilTrunkState.getValveState(valveId);
        if (state == ValveState.OFFLINE || state == ValveState.TEMP) return;
        double position = state == ValveState.SHUT ? 0.0 : utilTrunkState.getValvePosition(valveId);
        utControl.setValvePosition(valveId, position);
    }


    /**
     *  Timer task that updates the utility trunk system state when devices go offline or online.
     */
    private void updateState()
    {
        synchronized (utilTrunkState) {
            boolean changed = monitorControl.hasPeriodChanged();
            for (int fan = 0; fan < UtilTrunkFans.NUM_FANS; fan++) {
                FanState oldState = utilTrunkState.getFanState(fan);
                FanState state = oldState;
                if (utControl.isOnline()) {
                    if (oldState == FanState.OFFLINE) {
                        Double speed = fanControl[fan].getSpeed();
                        utilTrunkState.setFanSpeed(fan, speed == null ? 0.0 : Math.rint(1000.0 * speed) / 1000.0);
                        state = speed == null ? FanState.OFFLINE : speed > 0.0 ? FanState.SPEED : FanState.OFF;
                        utilTrunkState.setFanBaseState(fan, state);
                    }
                }
                else {
                    state = FanState.OFFLINE;
                }
                if (state != oldState) {
                    utilTrunkState.setFanState(fan, state);
                    changed = true;
                }
            }
            for (int htr = 0; htr < UtilTrunkHeaters.NUM_HEATERS; htr++) {
                int sw = UtilTrunkHeaters.getSwitch(htr);
                Boolean ss = swDevices[devNums[sw]].isSwitchOn(swNums[sw]);
                HeaterState oldState = utilTrunkState.getHeaterState(htr);
                HeaterState state = ss == null ? HeaterState.OFFLINE :
                                    utilTrunkState.getVpcState() == VpcControlState.TEMP
                                      ? HeaterState.TEMP : ss ? HeaterState.ON : HeaterState.OFF;
                if (state != oldState) {
                    if (state != HeaterState.TEMP && state != HeaterState.OFFLINE) {
                        utilTrunkState.setHeaterBaseState(htr, state);
                    }
                    utilTrunkState.setHeaterState(htr, state);
                }
            }
            for (int valve = 0; valve < UtilTrunkValves.NUM_VALVES; valve++) {
                ValveState oldState = utilTrunkState.getValveState(valve);
                ValveState state = oldState;
                if (utControl.isOnline()) {
                    if (oldState == ValveState.OFFLINE) {
                        try {
                            double posn = utControl.getValvePosition(valve);
                            utilTrunkState.setValvePosition(valve, Math.rint(100.0 * posn) / 100.0);
                            if (posn > 0.0) {
                                utilTrunkState.setValveBaseState(valve, ValveState.POSN);
                            }
                            state = utilTrunkState.getValveBaseState(valve);
                        }
                        catch (UtilityException e) {
                            LOG.log(Level.SEVERE, "Unexpected error", e);
                        }
                    }
                }
                else {
                    state = ValveState.OFFLINE;
                }
                if (state != oldState) {
                    utilTrunkState.setValveState(valve, state);
                    changed = true;
                }
            }
            if (gotCommand) {
                changed = true;
                gotCommand = false;
            }
            if (changed) {
                publishState();
            }
        }
    }


    /**
     *  Publishes the state of the system.
     *
     *  This is intended to be called whenever any element of the state is changed.
     */
    private void publishState()
    {
        utilTrunkState.setTickMillis(monitorControl.getFastPeriod());
        publishSubsystemDataOnStatusBus(new KeyValueData(UtilTrunkState.KEY, utilTrunkState));
    }    


    /**
     *  Timer task that checks for alert conditions.
     */
    private void checkForAlarms()
    {
        if (alarmDelay-- > 0) return;
        boolean overTemp = false;
        for (Channel tempChan : utTemps) {
            if (tempChan.getValue() > maxTemperature) {  // False if NaN
                overTemp = true;
                break;
            }
        }
        if (overTemp != utOverTemp) {
            utOverTemp = overTemp;
            Alert alert = UtilTrunkAlert.OVER_TEMPERATURE.newAlert();
            MpmAction.addData(alert, MpmAction.Action.BLOCK_UT_POWER);
            if (overTemp) {
                alertService.raiseAlert(alert, AlertState.ALARM,
                                        "A UT temperature went above the limit (" + maxTemperature + "). Turning off power.");
            }
            else {
                alertService.raiseAlert(alert, AlertState.NOMINAL,
                                        "All UT temperatures are now at or below the limit (" + maxTemperature + "). Re-enabling power.");
            }
        }
        for (int id = 0; id < UtilTrunkFans.NUM_FANS; id++) {
            double setRpm = fanControl[id].getRpm();
            double readRpm = fanSpeedChans[id].getValue();
            if (Double.isNaN(setRpm + readRpm)) continue;  // I.e. if either is NaN
            double error = Math.abs((readRpm - setRpm) / setRpm);
            int alertType = error > RPM_ERROR_ALARM ? ALERT_ALARM : error > RPM_ERROR_WARN ? ALERT_WARNING : ALERT_NOMINAL;
            if (alertType != fanSpeedAlerts[id] && (System.currentTimeMillis() - fanControl[id].getSpeedSetTime()) >= FAN_SETTLE_TIME) {
                fanSpeedAlerts[id] = alertType;
                Alert alert = UtilTrunkAlert.FAN_SPEED_INCORRECT.newAlert(UtilTrunkFans.getName(id));
                if (alertType == ALERT_ALARM) {
                    alertService.raiseAlert(alert, AlertState.ALARM,
                                            String.format("Read fan speed (%.2f) is not within %s%% of set speed (%.2f)",
                                                          readRpm, 100.0 * RPM_ERROR_ALARM, setRpm));
                }
                else if (alertType == ALERT_WARNING) {
                    alertService.raiseAlert(alert, AlertState.WARNING,
                                            String.format("Read fan speed (%.2f) is not within %s%% of set speed (%.2f)",
                                                          readRpm, 100.0 * RPM_ERROR_WARN, setRpm));
                }
                else {
                    alertService.raiseAlert(alert, AlertState.NOMINAL,
                                            String.format("Read fan speed is within %s%% of set speed", 100.0 * RPM_ERROR_WARN));
                }
            }
        }
        for (int id = 0; id < UtilTrunkValves.NUM_VALVES; id++) {
            double setPosn;
            try {
                setPosn = utControl.getValvePosition(id);
            }
            catch (UtilityException e) {
                continue;  // Don't check
            }
            double readPosn = valvePosnChans[id].getValue() / 100.0;  // Channel value is %
            if (Double.isNaN(readPosn)) continue;
            boolean badPosn = Math.abs(readPosn - setPosn) > VALVE_POSN_TOL;
            if (badPosn != valvePosnBad[id] && (System.currentTimeMillis() - utControl.getValveSetTime(id)) >= VALVE_SETTLE_TIME) {
                valvePosnBad[id] = badPosn;
                Alert alert = UtilTrunkAlert.VALVE_POSN_INCORRECT.newAlert(UtilTrunkValves.getName(id));
                if (badPosn) {
                    alertService.raiseAlert(alert, AlertState.WARNING,
                                            String.format("Read valve position (%.2f%%) is not within %.1f%% of set position (%.2f%%)",
                                                          100.0 * readPosn, 100.0 * VALVE_POSN_TOL, 100.0 * setPosn));
                }
                else {
                    alertService.raiseAlert(alert, AlertState.NOMINAL,
                                            String.format("Read valve position is within %.1f%% of set position", 100.0 * VALVE_POSN_TOL));
                }
            }
        }
    }


}
