package org.lsst.ccs.subsystem.utility;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.subsystem.common.devices.dataforth.Maq20Device;
import org.lsst.ccs.subsystem.utility.data.UtilityException;

/**
 *  Handles the simulated MAQ20 device.
 *
 *  @author Owen Saxton
 */
public class SimMaq20Device extends Maq20Device {

    private static final Logger LOG = Logger.getLogger(SimMaq20Device.class.getName());
    private final Map<String, Integer> channelMap = new LinkedHashMap<>();
    private final List<Double> channelValues = new ArrayList<>();
    private final List<Double> channelScales = new ArrayList<>();
    private final List<Double> channelOffsets = new ArrayList<>();


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        LOG.log(Level.INFO, "Connected to {0} (simulated MAQ20)", path);
        setOnline(true);
    }


    /**
     *  Closes device connection.
     */
    @Override
    protected void close()
    {
    }


    /**
     *  Checks a channel's parameters for validity.
     *
     *  @param  ch  The channel to check
     *  @return  A two-element array containing the encoded type [0] and subtype [1] values.
     *  @throws  Exception if there is a parameter error
     */
    @Override
    protected int[] checkChannel(Channel ch) throws Exception
    {
        int index = channelValues.size();
        channelMap.put(ch.getName(), index);
        channelValues.add((double)index);
        double offset = ch.convertRawValue(0.0);
        double scale = ch.convertRawValue(1.0) - offset;
        channelOffsets.add(offset);
        channelScales.add(scale);
        return new int[]{index, 0};
    }


    /**
     *  Reads a channel.
     *
     *  @param  ch  The channel to read
     *  @return  The read value
     */
    @Override
    protected double readChannel(Channel ch)
    {
        int type = ch.getType();
        return (channelValues.get(type) - channelOffsets.get(type)) / channelScales.get(type);
    }


    @Command(type=Command.CommandType.ACTION, level=0, description="Set a channel value")
    public void setChannelValue(@Argument(description="Channel name") String chan,
                                @Argument(description="Channel value") double value) throws UtilityException
    {
        Integer index = channelMap.get(chan);
        if (index == null) {
            throw new UtilityException("Invalid channel name");
        }
        channelValues.set(index, value);
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="Get the list of channel names")
    public String getChannelNames()
    {
        return channelMap.keySet().toString();
    }

}
