package org.lsst.ccs.subsystem.utility;

import java.time.Duration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.alert.AlertEvent;
import org.lsst.ccs.services.alert.AlertListener;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.utility.constants.MpmAlert;
import org.lsst.ccs.subsystem.utility.constants.ConditionState;
import org.lsst.ccs.subsystem.utility.constants.LatchState;
import org.lsst.ccs.subsystem.utility.constants.MpmConditions;
import org.lsst.ccs.subsystem.utility.constants.MpmLatches;
import org.lsst.ccs.subsystem.utility.constants.MpmLimits;
import org.lsst.ccs.subsystem.utility.constants.ComCamMpmPlcs;
import org.lsst.ccs.subsystem.utility.constants.MpmSwitches;
import org.lsst.ccs.subsystem.utility.constants.PLCState;
import org.lsst.ccs.subsystem.utility.constants.SwitchState;
import org.lsst.ccs.subsystem.utility.constants.UtilityAgentProperties;
import org.lsst.ccs.subsystem.utility.data.MpmSysState;
import org.lsst.ccs.subsystem.utility.data.UtilityException;

/**
 * The master protection system
 *
 * @author The LSST CCS Team
 */
public class ComCamMpmMain extends Subsystem implements HasLifecycle, AlertListener {

    private static final Map<Integer, MpmAlert> alertMap = new HashMap<>();
    static {
        alertMap.put(MpmLatches.LATCH_COLD_TEMP_HIGH, MpmAlert.COLD_TEMP_HIGH);
        alertMap.put(MpmLatches.LATCH_COLD_TEMP_LOW, MpmAlert.COLD_TEMP_LOW);
        alertMap.put(MpmLatches.LATCH_CRYO_TEMP_HIGH, MpmAlert.CRYO_TEMP_HIGH);
        alertMap.put(MpmLatches.LATCH_CRYO_TEMP_LOW, MpmAlert.CRYO_TEMP_LOW);
        alertMap.put(MpmLatches.LATCH_CRYO_VACUUM, MpmAlert.CRYO_VACUUM_BAD);
        alertMap.put(MpmLatches.LATCH_HEX_VACUUM, MpmAlert.HEX_VACUUM_BAD);
        alertMap.put(MpmLatches.LATCH_UT_LEAK, MpmAlert.UT_COOLANT_LEAK);
        alertMap.put(MpmLatches.LATCH_UT_LEAK_FAULT, MpmAlert.UT_LEAK_DETC_FAULT);
        alertMap.put(MpmLatches.LATCH_UT_SMOKE, MpmAlert.UT_SMOKE_DETC);
        alertMap.put(MpmLatches.LATCH_UT_SMOKE_FAULT, MpmAlert.UT_SMOKE_DETC_FAULT);
        alertMap.put(MpmLatches.LATCH_UT_TEMP, MpmAlert.UT_TEMP_HIGH);
    }
    private static final Map<String, String> revAlertMap = new HashMap<>();
    static {
        for (Map.Entry e : alertMap.entrySet()) {
            revAlertMap.put(((MpmAlert)e.getValue()).getId(), MpmLatches.getName((int)e.getKey()));
        }
    }
    private static final MpmAlert[] plcAlerts = new MpmAlert[ComCamMpmPlcs.NUM_PLCS];
    static {
        plcAlerts[0] = MpmAlert.POWER_PLC_NOT_ALIVE;
        plcAlerts[1] = MpmAlert.CRYO_PLC_NOT_ALIVE;
    }
    private static final String[] plcNames = new String[ComCamMpmPlcs.NUM_PLCS];
    private static final int[] plcID = new int[ComCamMpmPlcs.NUM_PLCS];
    static {
        plcID[0] = ComCamMpmPlcs.PLC_TRUNK;
        plcNames[0] = "Trunk";

	//        plcNames[ComCamMpmPlcs.PLC_COLD] = "Cold";
        plcID[1] = ComCamMpmPlcs.PLC_CRYO;
        plcNames[1] = "Cryo";
    }

    @LookupName
    private String name;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService propertiesService;

    @LookupField(strategy=LookupField.Strategy.DESCENDANTS)
    private MpmPlutoDevice plutoDevc;

    // General
    private static final Logger LOG = Logger.getLogger(ComCamMpmMain.class.getName());
    private final MpmSysState mpmState = new MpmSysState();
    private boolean running = false, gotCommand = false;

    
    public ComCamMpmMain() {
        super("comcam-mpm", AgentInfo.AgentType.WORKER);
    }
    

    /**
     *  Build phase
     */
    @Override
    public void build() {
        //Create and schedule an AgentPeriodicTask to update the protection state
        AgentPeriodicTask pt;
        pt = new AgentPeriodicTask("Protection-state",
                                   () -> updateMpmState()).withPeriod(Duration.ofMillis(1000));
        periodicTaskService.scheduleAgentPeriodicTask(pt);
        //ass.registerState(ProtectionState.class, "Protection state", this);
        //ass.updateAgentState(ProtectionState.UNKNOWN);
    }

    /**
     * Init phase.
     * We register alerts raised by this subsystem.
     */
    @Override
    public void init() {
        for (MpmAlert alert : plcAlerts) {
            alertService.registerAlert(alert.getAlert());
        }
        
        for (MpmAlert alert : alertMap.values()) {
            alertService.registerAlert(alert.getAlert());
        }
        
    }

    /**
     *  Post initialization
     */
    @Override
    public void postInit() {
        // Set a property to define that this Agent is a protection subsystem.
        propertiesService.setAgentProperty(UtilityAgentProperties.MPM_TYPE, ComCamMpmMain.class.getCanonicalName());

        // Add alert listener
        alertService.addListener(this);

        if (plutoDevc != null) {
            for (int cond = 0; cond < MpmLatches.NUM_LATCHES; cond++) {
                mpmState.setLatch(cond, LatchState.CLEAR);
            }
            for (int cond = 0; cond < MpmConditions.NUM_CONDITIONS; cond++) {
                mpmState.setCondition(cond, ConditionState.NO);
            }
        }
        else {
            ErrorUtils.reportConfigError(LOG, name, "Pluto device", "not specified");
        }
    }


    /**
     *  Post start
     */
    @Override
    public void postStart() {
        LOG.info("Protection subsystem started");
        running = true;
    }


    /**
     *  Gets the state of the protection system.
     *
     *  @return  The protection system state
     */
    @Command(type=Command.CommandType.QUERY, description="Get the protection system state")
    public MpmSysState getSystemState() {
        mpmState.setTickMillis(getTickPeriod());
        return mpmState;
    }    


    /**
     *  Gets the list of switch names.
     *
     *  @return  The switch names.
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.QUERY, description="Get switch names")
    public List<String> getSwitchNames() throws UtilityException
    {
        return MpmSwitches.getNames();
    }


    /**
     *  Turns a (named) switch on or off.
     *
     *  @param  swch  The switch name.
     *  @param  on    Whether to turn on or off
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Turn on/off a named switch")
    public void setNamedSwitchOn(@Argument(description="The switch name") String swch,
                                 @Argument(description="Whether to turn on") boolean on) throws UtilityException
    {
        gotCommand = true;
        int sw = MpmSwitches.getId(swch);
        SwitchState state = mpmState.getSwitchState(sw);
        if (state == SwitchState.OFFLINE) return;
        plutoDevc.setSwitchOn(sw, on, false);
    }


    /**
     *  Gets the list of latched condition names.
     *
     *  @return  The condition names.
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.QUERY, description="Get latched condition names")
    public List<String> getLatchNames() throws UtilityException
    {
        return MpmLatches.getNames();
    }


    /**
     *  Clears a (named) latched condition.
     *
     *  @param  cond  The condition name.
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Clear a latched condition")
    public void clearLatch(@Argument(description="The condition name") String cond) throws UtilityException
    {
        gotCommand = true;
        plutoDevc.clearLatch(MpmLatches.getId(cond));
    }


    /**
     *  Sets the update period.
     *
     *  @param  value  The update period (milliseconds) to set.
     */
    @Command(type=Command.CommandType.ACTION, description="Set the update interval")
    public void setUpdatePeriod(@Argument(description="The tick period (ms)") int value)
    {
        gotCommand = true;
        setTickPeriod(value);
    }


    /**
     *  Updates the protection system state periodically.
     *
     *  The protection state consists mainly of the state of the switches (lines) being
     *  controlled, along with whether they can be turned on.
     */
    private void updateMpmState()
    {
        if (!running) return;

        boolean changed = false;

        for (int iplc = 0; iplc < ComCamMpmPlcs.NUM_PLCS; iplc++) {
	    int plc  = plcID[iplc];
            Boolean active = plutoDevc.isPlcActive(plc);
            PLCState newState = active == null ? PLCState.OFFLINE : active ? PLCState.ALIVE : PLCState.DEAD;
            if (newState != mpmState.getPlcState(plc)) {
                changed = true;
                mpmState.setPlcState(plc, newState);
                MpmAlert alert = plcAlerts[iplc];
                String plcDesc = plcNames[iplc] + " protection PLC ";
                if (newState == PLCState.ALIVE) {
                    lowerAlert(alert, plcDesc + "is alive");
                }
                else {
                    String errDesc = newState == PLCState.DEAD ? "has died: error = " + plutoDevc.getErrorCode(plc) : "is offline"; 
                    raiseAlert(alert, plcDesc + errDesc);
                }
            }
        }

        for (int sw = 0; sw < MpmSwitches.NUM_SWITCHES; sw++) {
            SwitchState state = plutoDevc.getSwitchState(sw);
            if (state != mpmState.getSwitchState(sw)) {
                mpmState.setSwitchState(sw, state);
                changed = true;
            }
        }

        for (int cond = 0; cond < MpmLatches.NUM_LATCHES; cond++) {
            Boolean active = plutoDevc.isLatchActive(cond);
            Boolean latched = plutoDevc.isLatchLatched(cond);
            LatchState state = active == null || latched == null ? LatchState.OFFLINE :
                               latched ? LatchState.LATCHED :
                               active ? LatchState.ACTIVE : LatchState.CLEAR;
            MpmAlert alert = alertMap.get(cond);
            LatchState oldState = mpmState.getLatch(cond); 
            if (state != oldState) {
                mpmState.setLatch(cond, state);
                if (state == LatchState.ACTIVE) {
                    raiseAlert(alert, "Protection PLC error condition set");
                }
                else if (state != LatchState.OFFLINE) {
                    if (oldState != LatchState.ACTIVE && state == LatchState.LATCHED) {
                        raiseAlert(alert, "Protection PLC error condition set");
                    }
                    if (oldState == LatchState.ACTIVE || state == LatchState.LATCHED) {
                        lowerAlert(alert, "Protection PLC error condition cleared");
                    }
                }
                changed = true;
            }
        }

        for (int cond = 0; cond < MpmConditions.NUM_CONDITIONS; cond++) {
            Boolean active = plutoDevc.isConditionActive(cond);
            ConditionState state = active == null ? ConditionState.OFF :
                                   active ? ConditionState.YES : ConditionState.NO;
            if (state != mpmState.getCondition(cond)) {
                mpmState.setCondition(cond, state);
                changed = true;
            }
        }

        if (mpmState.getLimit(0) == Integer.MAX_VALUE) {
            int[] limits = plutoDevc.getTempLimits();
            for (int j = 0; j < MpmLimits.NUM_LIMITS; j++) {
                mpmState.setLimit(j, limits[j]);
            }
            changed = true;
        }

        if (gotCommand) {
            changed = true;
            gotCommand = false;
        }
        if (changed) {
            publishState();
        }
    }


    /**
     *  Raises an alert.
     *
     *  @param  alert  The protection alert to raise
     *  @param  cond   The alert condition
     */
    private void raiseAlert(MpmAlert alert, String cond)
    {
        alertService.raiseAlert(alert.getAlert(), AlertState.ALARM, cond);
    }


    /**
     *  Lowers an alert.
     *
     *  @param  alert  The protection alert to lower
     *  @param  cond   The alert condition
     */
    private void lowerAlert(MpmAlert alert, String cond)
    {
        alertService.raiseAlert(alert.getAlert(), AlertState.NOMINAL, cond);
    }


    /**
     *  Alert event handler.
     *
     *  Resets PLC latch when corresponding alert is cleared.
     *
     *  @param  event  The alert event
     */
    @Override
    public void onAlert(AlertEvent event)
    {
        if (event.getType() != AlertEvent.AlertEventType.ALERT_CLEARED) return;
        for (String id : event.getClearedIds()) {
            String cond = revAlertMap.get(id);
            if (cond != null) {
                try {
                    clearLatch(cond);
                }
                catch (UtilityException e) {
                    LOG.log(Level.SEVERE, "Error clearing latched PLC condition ({0}): {1}", new Object[]{cond, e});
                }
            }
        }
    }


    /**
     *  Publishes the state of the protection system.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    private void publishState()
    {
        mpmState.setTickMillis(getTickPeriod());
        publishSubsystemDataOnStatusBus(new KeyValueData(MpmSysState.KEY, mpmState));
    }


    /**
     *  Sets the monitoring publishing period.
     */
    private void setTickPeriod(long period)
    {
        periodicTaskService.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }
    

    /**
     *  Gets the monitoring publishing period.
     */
    private int getTickPeriod()
    {
        return (int)periodicTaskService.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }

}
