package org.lsst.ccs.subsystem.utility;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.lambda.Cpfe1000;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.subsystem.common.ErrorUtils;

/**
 *  Handles a Lambda CPFE1000 power supply system.
 *
 *  @author Owen Saxton
 */
public class CpfeDevice extends Device {

   /**
    *  Constants.
    */
    public static final int
        CHAN_STATUS = 0,
        CHAN_TEMPERATURE = 1,
        CHAN_VOLTAGE = 2,
        CHAN_CURRENT = 3,
        NUM_CHANS = 4;

   /**
    *  Private lookup maps.
    */
    private static final Map<String, Cpfe1000.ConnType> cTypeMap = new HashMap<>();
    static {
        cTypeMap.put("SERIAL", Cpfe1000.ConnType.SERIALUSB);
    }

   /**
    *  Data fields.
    */
    private String connType;     // Connection type string
    private String devcId;       // Device identification
    private int address = 0xa0;  // I2C bus address

    private static final Logger LOG = Logger.getLogger(CpfeDevice.class.getName());
    private final Cpfe1000 cpfe = new Cpfe1000();  // Associated CPFE object
    private Cpfe1000.ConnType cType;
    private boolean initError = false;


   /**
    *  Performs basic initialization.
    */
    @Override
    protected void initDevice()
    {
        if (connType == null) {
            ErrorUtils.reportConfigError(LOG, name, "connType", "is missing");
        }
        cType = cTypeMap.get(connType.toUpperCase());
        if (cType == null) {
            ErrorUtils.reportConfigError(LOG, name, "connType", "is invalid");
        }
        if (devcId == null) {
            ErrorUtils.reportConfigError(LOG, name, "devcId", "is missing");
        }
        fullName = "CPFE1000 (" + devcId + ":" + address + ")";
    }


   /**
    *  Performs full initialization.
    */
    @Override
    protected void initialize()
    {
        try {
            cpfe.open(cType, devcId, address);
            initSensors();
            LOG.log(Level.INFO, "Connected to {0}", fullName);
            initError = false;
            setOnline(true);
        }
        catch (DriverException e) {
            if (!initError) {
                LOG.log(Level.SEVERE, "Error connecting to {0}: {1}", new Object[]{fullName, e});
                initError = true;
            }
        }
    }


   /**
    *  Closes the connection.
    */
    @Override
    protected void close()
    {
        try {
            cpfe.close();
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error disconnecting from {0}: {1}", new Object[]{fullName, e});
        }
    }


   /**
    *  Checks a channel's parameters for validity.
    *
    *  @param  ch  The channel to check
    *  @return  A two-element array containing the encoded type [0] and
    *           subtype [1] values.
    *  @throws  Exception if any errors found in the parameters.
    */
    @Override
    protected int[] checkChannel(Channel ch) throws Exception
    {
        int hwChan = ch.getHwChan();
        if (hwChan < 0 || hwChan >= NUM_CHANS) {
            ErrorUtils.reportChannelError(LOG, ch.getPath(), "hwchan", hwChan);
        }
        return new int[]{0, 0};
    }


   /**
    *  Reads a channel.
    *
    *  @param  ch  The channel to read
    *  @return  The read value
    */
    @Override
    protected double readChannel(Channel ch)
    {
        double value = Double.NaN;

        try {
            switch (ch.getHwChan()) {

            case CHAN_STATUS:
                value = cpfe.readStatus(); break;

            case CHAN_TEMPERATURE:
                value = cpfe.readTemperature(); break;

            case CHAN_VOLTAGE:
                value = cpfe.readVoltage(); break;

            case CHAN_CURRENT:
                value = cpfe.readCurrent(); break;
            }
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error reading CPFE1000 data: {0}", e);
            setOnline(false);
        }

        return value;
    }

}
