package org.lsst.ccs.subsystem.utility;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.localdb.statusdb.server.TrendingData;
import org.lsst.ccs.localdb.trendserver.TrendingClientService;
import org.lsst.ccs.localdb.utils.TrendingConnectionUtils;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.DerivedChannel;

/**
 * A derived channel to accumulate the on time for a channel
 * 
 * @author The LSST CCS Team
 */
public class AccumTimeChannel extends DerivedChannel
{
    private static final int CHECK_INTERVAL = 10000;

    @LookupPath
    private String path;
    @LookupField(strategy = LookupField.Strategy.TOP)
    private Subsystem subsys;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private TrendingClientService trendingService;

    private Channel channel;  // From Groovy file

    private static final Logger LOG = Logger.getLogger(AccumTimeChannel.class.getName());
    private final Object syncSavedTime = new Object();
    private String fullName;
    private TrendingConnectionUtils trendingUtils;
    private boolean trendingUtilsError = false;
    private long readTime = 0, checkTime = 0;
    private double accumTime = 0.0, savedTime = Double.NaN;

    /**
     *  Init phase
     */
    @Override
    public void init()
    {
        fullName = subsys.getName() + "/" + path;
        if (channel == null) {
            throw new RuntimeException(getPath() + " parameter error: channel has not been defined");
        }
        super.init();
    }

    /**
     *  Start phase
     */
    @Override
    public void start()
    {
        initSavedTime();
        super.start();
    }

    /**
     *  Calculate the value.
     * 
     *  @return  The value
     */
    @Override
    public double evaluateDerivedValue()
    {
        long currTime = System.currentTimeMillis();
        if (currTime - checkTime >= CHECK_INTERVAL) {
            initSavedTime();
            checkTime = currTime;
        }
        if (readTime != 0 && channel.getValue() > 0.0) {  // Fails test if NaN
            accumTime += (currTime - readTime) / 3.6e6;   // Add hours
        }
        readTime = currTime;
        return accumTime + savedTime;  // NaN if savedTime = NaN
    }

    @Command(type=Command.CommandType.ACTION, description="Set the accumulated time")
    public void setAccumTime(@Argument(description="Time period (hrs)") double time)
    {
        synchronized(syncSavedTime) {
            savedTime = time - accumTime;
        }
    }

    private void initSavedTime()
    {
        if (!Double.isNaN(savedTime)) return;
        trendingUtils = trendingService.getBusTrendingConnection();
        if (trendingUtils == null) {
            if (!trendingUtilsError) {
                LOG.log(Level.SEVERE, "Trending data cannot be accessed");
                trendingUtilsError = true;
            }
            return;
        }
        if (trendingUtilsError) {
            LOG.log(Level.INFO, "Trending data can now be accessed");
            trendingUtilsError = false;
        }
        synchronized(syncSavedTime) {
            if (Double.isNaN(savedTime)) {
                TrendingData data = trendingUtils.getLatestData(fullName);
                savedTime = data == null ? Double.NaN : data.getValue("value");
                if (Double.isNaN(savedTime)) {
                    LOG.log(Level.WARNING, "No trending data present for channel {0}", fullName);
                    savedTime = 0.0;
                }
                else {
                    LOG.log(Level.INFO, "Got trending data for channel {0}", fullName);
                }
            }
        }
    }

}
