package org.lsst.ccs.subsystem.utility;

import java.util.ArrayList;
import java.util.List;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.subsystem.utility.constants.MpmConditions;
import org.lsst.ccs.subsystem.utility.constants.MpmLatches;
import org.lsst.ccs.subsystem.utility.constants.MpmSwitches;

/**
 *  Handles the vacuum system Pluto PLC.
 *
 *  @author Owen Saxton
 */
public class MpmSimPlutoDevice extends MpmPlutoDevice {

    /**
     *  Constants
     */
    static final List<Integer>[] clearList = new ArrayList[MpmLatches.NUM_LATCHES];
    static {
        for (int cond = 0; cond < MpmLatches.NUM_LATCHES; cond++) {
            clearList[cond] = new ArrayList<>();
        }
        clearList[MpmLatches.LATCH_COLD_TEMP_HIGH].add(MpmLatches.LATCH_COLD_TEMP_HIGH);
        clearList[MpmLatches.LATCH_COLD_TEMP_LOW].add(MpmLatches.LATCH_COLD_TEMP_LOW);
        clearList[MpmLatches.LATCH_CRYO_TEMP_HIGH].add(MpmLatches.LATCH_CRYO_TEMP_HIGH);
        clearList[MpmLatches.LATCH_CRYO_TEMP_LOW].add(MpmLatches.LATCH_CRYO_TEMP_LOW);
        clearList[MpmLatches.LATCH_CRYO_VACUUM].add(MpmLatches.LATCH_CRYO_VACUUM);
        clearList[MpmLatches.LATCH_HEX_VACUUM].add(MpmLatches.LATCH_HEX_VACUUM);
        clearList[MpmLatches.LATCH_UT_LEAK].add(MpmLatches.LATCH_UT_LEAK);
        clearList[MpmLatches.LATCH_UT_LEAK].add(MpmLatches.LATCH_UT_LEAK_FAULT);
        clearList[MpmLatches.LATCH_UT_LEAK_FAULT].add(MpmLatches.LATCH_UT_LEAK);
        clearList[MpmLatches.LATCH_UT_LEAK_FAULT].add(MpmLatches.LATCH_UT_LEAK_FAULT);
        clearList[MpmLatches.LATCH_UT_SMOKE].add(MpmLatches.LATCH_UT_SMOKE);
        clearList[MpmLatches.LATCH_UT_SMOKE].add(MpmLatches.LATCH_UT_SMOKE_FAULT);
        clearList[MpmLatches.LATCH_UT_SMOKE_FAULT].add(MpmLatches.LATCH_UT_SMOKE);
        clearList[MpmLatches.LATCH_UT_SMOKE_FAULT].add(MpmLatches.LATCH_UT_SMOKE_FAULT);
        clearList[MpmLatches.LATCH_UT_TEMP].add(MpmLatches.LATCH_UT_TEMP);
    }

    /**
     *  Data fields.
     */
    private final boolean[] switches = new boolean[MpmSwitches.NUM_SWITCHES];
    private final Boolean[] latches = new Boolean[MpmLatches.NUM_LATCHES];
    private final boolean[] conditions = new boolean[MpmConditions.NUM_CONDITIONS];
    private final int[] simTempLimits = {-39, 39, -139, 29};


    /**
     *  Performs basic initialization.
     */
    @Override
    protected void initDevice()
    {
    }


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        setOnline(true);
    }


    /**
     *  Closes device connection.
     */
    @Override
    protected void close()
    {
    }


    /**
     *  Checks a channel's parameters for validity.
     *
     *  @param  ch  The channel to check
     *  @return  A two-element array containing the encoded type [0] and subtype [1] values.
     */
    @Override
    protected int[] checkChannel(Channel ch)
    {
        return new int[]{0, 0};
    }


    /**
     *  Reads a channel.
     *
     *  @param  ch  The channel to read
     *  @return  The read value
     */
    @Override
    protected double readChannel(Channel ch)
    {
        return 315.0 + ch.getHwChan();
    }


    /**
     *  Tests whether a PLC is active
     *
     *  @param  plc  The PLC number
     *  @return  Whether the PLC is active, or null if offline
     */
    @Override
    public Boolean isPlcActive(int plc)
    {
        return true;
    }


    /**
     *  Sets a switch on or off.
     *
     *  For the vacuum Pluto, this is implemented as a pair of push buttons,
     *  one for on, one for off.
     *
     *  @param  sw  The switch number.
     *  @param  on  The on state to set: true or false
     */
    @Override
    public void setSwitchOn(int sw, boolean on)
    {
        switches[sw] = on;
    }


    /**
     *  Gets the on state of a switch.
     *
     *  The state is not the state of the bit that was toggled, but is read back
     *  either directly from the controlled hardware, or from the PLC output line.
     *
     *  @param  sw  The switch number.
     *  @return  Whether the switch is on
     */
    @Override
    public Boolean isSwitchOn(int sw)
    {
        return switches[sw];
    }


    /**
     *  Gets whether a latched condition is active.
     * 
     *  @param  cond  The condition number
     *  @return  Whether active - indicated by the bit being 0
     */
    @Override
    public Boolean isLatchActive(int cond)
    {
        Boolean latch = latches[cond];
        return latch != null && latch;
    }


    /**
     *  Gets whether a latched condition is latched.
     * 
     *  @param  cond  The condition number
     *  @return  Whether latched - indicated by the bit being 1
     */
    @Override
    public Boolean isLatchLatched(int cond)
    {
        Boolean latch = latches[cond];
        return latch != null && !latch;
    }


    /**
     *  Clears a latched condition.
     * 
     *  @param  cond  The condition number
     */
    @Override
    public void clearLatch(int cond)
    {
        for (int cnd : clearList[cond]) {
            Boolean latch = latches[cnd];
            latches[cnd] = latch != null && !latch ? null : latch;
        }
    }


    /**
     *  Gets whether a condition is active.
     * 
     *  @param  cond  The condition number
     *  @return  Whether active - indicated by the bit being 1
     */
    @Override
    public Boolean isConditionActive(int cond)
    {
        return conditions[cond];
    }


    /**
     *  Gets the temperature limits.
     *
     *  @return  The 4-element array of temperature limits
     */
    @Override
    public int[] getTempLimits()
    {
        return simTempLimits;
    }


    @Command(type = Command.CommandType.ACTION)
    public void setLatchActive(@Argument(description="Latched condition number") int cond) {
        latches[cond] = true;
    }

    @Command(type = Command.CommandType.ACTION)
    public void setLatchLatched(@Argument(description="Latched condition number") int cond) {
        latches[cond] = false;
    }

    @Command(type = Command.CommandType.ACTION)
    public void setCondition(@Argument(description="Running condition number") int cond,
                             @Argument(description="Condition on state") boolean set) {
        conditions[cond] = set;
    }

}
