package org.lsst.ccs.subsystem.utility;

import java.util.HashSet;
import java.util.Set;
import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.dataforth.Maq20;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.common.devices.dataforth.Maq20AnalogControl;
import org.lsst.ccs.subsystem.common.devices.dataforth.Maq20Device;
import org.lsst.ccs.subsystem.utility.constants.UtilTrunkValves;
import org.lsst.ccs.subsystem.utility.data.UtilityException;

/**
 *  Controls utility trunk Maq20 device (operates valves)
 * 
 *  @author saxton
 */
public class Maq20DeviceUT extends Maq20Device {

    public static final int
        CHAN_UT_VALVE_SET   = 0,
        CHAN_VPC_VALVE_SET  = 1,
        CHAN_MPC_VALVE_SET  = 2,

        CHAN_VPC_SPLY_AIR_VEL  = 0,
        CHAN_VPC_SPLY_AIR_TEMP = 1,
        CHAN_VPC_RETN_PRESS    = 2,
        CHAN_VPC_SPLY_PRESS    = 3,
        CHAN_VPC_PREFILT_PRESS = 4,
        CHAN_VPC_HUMIDITY      = 5,
        CHAN_MPC_RETN_PRESS    = 6,
        CHAN_MPC_PREFILT_PRESS = 7,
        CHAN_MPC_SPLY_PRESS    = 8,
        CHAN_MPC_HUMIDITY      = 9,
        CHAN_UT_FLOW_RATE      = 10,
        CHAN_VPC_VALVE_POSN    = 11,
        CHAN_UT_VALVE_POSN     = 12,
        CHAN_MPC_VALVE_POSN    = 13;

    private static final double VALVE_OFFSET = 0.004, VALVE_SCALE = 0.016;

    private static final int[] valveSetChans = new int[UtilTrunkValves.NUM_VALVES];
    static {
        valveSetChans[UtilTrunkValves.VALVE_UT_ID] = CHAN_UT_VALVE_SET;
        valveSetChans[UtilTrunkValves.VALVE_VPC_ID] = CHAN_VPC_VALVE_SET;
        valveSetChans[UtilTrunkValves.VALVE_MPC_ID] = CHAN_MPC_VALVE_SET;
    }
    private static final Set<Integer> valveReadChans = new HashSet<>();
    static {
        valveReadChans.add(CHAN_UT_VALVE_POSN);
        valveReadChans.add(CHAN_VPC_VALVE_POSN);
        valveReadChans.add(CHAN_MPC_VALVE_POSN);
    }

    @LookupField (strategy=LookupField.Strategy.CHILDREN)
    private Maq20AnalogControl maqCtrl;

    private static final Logger LOG = Logger.getLogger(Maq20DeviceUT.class.getName());
    private int ctrlModIndex, currModIndex = -1;
    private final long[] setTime = new long[UtilTrunkValves.NUM_VALVES];


    /**
     *  Basic initialization.
     */
    @Override
    public void initDevice()
    {
        super.initDevice();
        if (maqCtrl == null) {
            ErrorUtils.reportConfigError(LOG, path, "analog control", "not defined");
        }
        ctrlModIndex = maqCtrl.getModIndex();
        for (int ix = 0; ix < getModuleCount(); ix++) {
            ModuleData modData = getModuleData(ix);
            if (modData.modDef.type == Maq20.ModuleType.IS) {
                currModIndex = ix;
                break;
            }
        }
    }


   /**
    *  Reads a channel.
    *
    *  @param  ch  The channel to read
    *  @return  Channel value
    */
    @Override
    protected double readChannel(Channel ch)
    {
        double value = super.readChannel(ch);
        int hwChan = ch.getHwChan();
        int index = ch.getType() & 0xff;
        if (index == ctrlModIndex) {
            value = 100.0 * (value - VALVE_OFFSET) / VALVE_SCALE;
        }
        else if (index == currModIndex && valveReadChans.contains(hwChan)) {
            double posn = (value - VALVE_OFFSET) / VALVE_SCALE;
            value = 100.0 * (hwChan == CHAN_MPC_VALVE_POSN ? 1.0 - posn : posn);
        }
        return value;
    }


    /**
     *  Sets a valve position.
     *
     *  @param  valveId  The valve ID
     *  @param  posn  Requested position (0 - 1)
     *  @throws  UtilityException
     */
    public void setValvePosition(int valveId, double posn) throws UtilityException
    {
        int chan = valveSetChans[valveId];
        double value = VALVE_SCALE * Math.min(Math.max(chan == CHAN_MPC_VALVE_SET ? (1.0 - posn) : posn, 0.0), 1.0) + VALVE_OFFSET;
        try {
            maqCtrl.writeValue(chan, value);
            setTime[valveId] = System.currentTimeMillis();
        }
        catch (DriverException e) {
            throw new UtilityException("Error setting valve " + UtilTrunkValves.getName(valveId)
                                         + " (channel " + chan + ") position: " + e);
        }
    }


    /**
     *  Gets a valve position.
     *
     *  @param  valveId  The valve ID
     *  @return  The set position (0 - 1)
     *  @throws  UtilityException
     */
    public double getValvePosition(int valveId) throws UtilityException
    {
        int chan = valveSetChans[valveId];
        try {
            double posn = (maqCtrl.readValue(chan) - VALVE_OFFSET) / VALVE_SCALE;
            return chan == CHAN_MPC_VALVE_SET ? (1.0 - posn) : posn;
        }
        catch (DriverException e) {
            throw new UtilityException("Error getting valve " + UtilTrunkValves.getName(valveId)
                                         + " (channel " + chan + ") position: " + e);
        }
    }


    /**
     *  Gets the set time for a valve.
     *
     *  Need to know this because it can take several seconds to reach its set point
     * 
     *  @param  valveId  The valve ID
     *  @return  The set time (ms)
     */
    public long getValveSetTime(int valveId)
    {
        return setTime[valveId];
    }

}
