package org.lsst.ccs.subsystem.utility;

import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.Map;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.DerivedChannel;
import org.lsst.ccs.subsystem.utility.data.UtilityException;

/**
 * A derived channel to calculate the weighted average of a set of channels
 * 
 * @author The LSST CCS Team
 */
public class WeightedAverageChannel extends DerivedChannel {
    
    private Channel[] channels;  // From Groovy file
    private double[] weights;    // From Groovy file

    private final Map<String, Integer> nameMap = new LinkedHashMap<>();

    @Override
    public void init()
    {        
        if (channels == null) {
            throw new RuntimeException(getPath() + " parameter error: channels array has not been defined");
        }
        if (weights == null) {
            weights = new double[channels.length];
            Arrays.fill(weights, 1.0);
        }
        else if (weights.length != channels.length) {
            throw new RuntimeException(getPath() + " parameter error : channels and weights arrays have different sizes");
        }
        for (int ix = 0; ix < channels.length; ix++) {
            nameMap.put(channels[ix].getPath(), ix);
        }
        super.init();
    }

    @Override
    public double evaluateDerivedValue()
    {
        double valueSum = 0.0, weightSum = 0.0;
        for (int ix = 0; ix < channels.length; ix++) {
            double value = channels[ix].getValue();
            if (!Double.isNaN(value)) {
                valueSum += weights[ix] * value;
                weightSum += weights[ix];
            }
        }
        return weightSum != 0.0 ? valueSum / weightSum : Double.NaN;
    }

    @Command(type=Command.CommandType.QUERY, description="Get the channel weights", level=0)
    public Map<String, Double> getChannelWeights()
    {
        Map<String, Double> weightMap = new LinkedHashMap<>();
        for (int ix = 0; ix < channels.length; ix++) {
            weightMap.put(channels[ix].getPath(), weights[ix]);
        }
        return weightMap;
    }
    
    @Command(type=Command.CommandType.ACTION, description="Set a channel weight")
    public void setChannelWeight(@Argument(description="Channel name") String name,
                                 @Argument(description="Channel weight") double weight) throws UtilityException
    {
        Integer ix = nameMap.get(name);
        if (ix == null) {
            throw new UtilityException("Invalid channel name: " + name);
        }
        weights[ix] = weight;
    }

}
