package org.lsst.ccs.subsystem.utility;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.subsystem.utility.data.UtilityException;

/**
 *  Handles the simulated MAQ20 device.
 *
 *  @author Owen Saxton
 */
public class SimMaq20DeviceUT extends Maq20DeviceUT {

    @LookupField(strategy=LookupField.Strategy.TREE)
    private final Map<String, Channel> allChannels = new HashMap<>();

    private static final Logger LOG = Logger.getLogger(SimMaq20DeviceUT.class.getName());
    private final Map<String, Integer> channelMap = new LinkedHashMap<>();
    private final List<Double> channelValues = new ArrayList<>();
    private final List<Double> channelScales = new ArrayList<>();
    private final List<Double> channelOffsets = new ArrayList<>();


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        LOG.log(Level.INFO, "Connected to {0} (simulated MAQ20)", path);
        setOnline(true);
    }


    /**
     *  Closes device connection.
     */
    @Override
    protected void close()
    {
    }


    /**
     *  Checks a channel's parameters for validity.
     *
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel number
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string
     *  @return  A two-element array containing the encoded type [0] and subtype [1] values.
     *  @throws  Exception if there is a parameter error
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        int index = channelValues.size();
        channelMap.put(name, index);
        channelValues.add((double)index);
        Channel channel = allChannels.get(name);
        double offset = 0.0, scale = 1.0;
        if (channel != null) {
            offset = channel.convertRawValue(0.0);
            scale = channel.convertRawValue(1.0) - offset;
        }
        channelOffsets.add(offset);
        channelScales.add(scale);
        return new int[]{index, 0};
    }


    /**
     *  Reads all referenced channels.
     */
    @Override
    protected void readChannelGroup()
    {
    }


    /**
     *  Reads a channel.
     *
     *  @param  hwChan   The hardware channel number.
     *  @param  type     The encoded channel type returned by checkChannel.
     *  @return  The read value
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        return (channelValues.get(type) - channelOffsets.get(type)) / channelScales.get(type);
    }


    @Command(type=Command.CommandType.ACTION, level=0, description="Set a channel value")
    public void setChannelValue(@Argument(description="Channel name") String chan,
                                @Argument(description="Channel value") double value) throws UtilityException
    {
        Integer index = channelMap.get(chan);
        if (index == null) {
            throw new UtilityException("Invalid channel name");
        }
        channelValues.set(index, value);
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="Get the list of channel names")
    public String getChannelNames()
    {
        return channelMap.keySet().toString();
    }

}
