package org.lsst.ccs.subsystem.utility;

import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.DerivedChannel;
import org.lsst.ccs.subsystem.utility.constants.UtilTrunkValves;
import org.lsst.ccs.subsystem.utility.data.UtilityException;

/**
 *  A DerivedChannel to compute the deviation of the measured valve position from the set value.
 * 
 *  This channel's value does not need to be trended: it's purpose is just to generate
 *  alerts when the deviation exceeds the specified limits.
 *
 *  @author The CCS team
 */
public class ValvePositionErrorChannel extends DerivedChannel {

    @LookupPath
    private String path;

    private Channel positionChannel;
    private Maq20DeviceUT valveControl;
    private Integer valveId;

   /**
    *  Check that required items have been specified.
    */
    @Override
    public void postInit() {
        if (positionChannel == null) {
            throw new RuntimeException(path + ": positionChannel has not been specified");
        }
        if (valveControl == null) {
            throw new RuntimeException(path + ": valveControl has not been specified");
        }
        if (valveId == null) {
            throw new RuntimeException(path + ": valveId has not been specified");
        }
        if (valveId < 0 || valveId >= UtilTrunkValves.NUM_VALVES) {
            throw new RuntimeException(path + ": valveId must be non-negative and less than " + UtilTrunkValves.NUM_VALVES);
        }
    }    

   /**
    *  Evaluate the absolute value of the deviation of measured from set position.
    *
    *  @return  double 
    */
    @Override
    public double evaluateDerivedValue() {
        double setPosition;
        try {
            setPosition = 100.0 * valveControl.getValvePosition(valveId);
        }
        catch (UtilityException e) {
            return Double.NaN;
        }
        if (System.currentTimeMillis() >= valveControl.getStableValveTime(valveId)) {
            return Math.abs(positionChannel.getValue() - setPosition);
        }
        else {
            return Double.NaN;
        }
    }

}

