package org.lsst.ccs.subsystem.utility;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.utility.data.UtilityException;

/**
 *  Provides measured quantities from the telescope.
 *
 *  @author Owen Saxton
 */
public class TelescopeDevice extends Device {

    public static final int
        CHAN_DOME_XMINUS_TEMP = 0,
        CHAN_DOME_YMINUS_TEMP = 1,
        NUM_CHANNELS = 2;

    private static final Logger LOG = Logger.getLogger(TelescopeDevice.class.getName());
    private final Map<String, Integer> channelMap = new LinkedHashMap<>();
    private final List<Double> channelValues = new ArrayList<>();


    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice()
    {
        for (int chan = 0; chan < NUM_CHANNELS; chan++) {
            channelValues.add(chan, 18.0);
        }
    }


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        setOnline(true);
    }


    /**
     *  Closes device connection.
     */
    @Override
    protected void close()
    {
    }


    /**
     *  Checks a channel's parameters for validity.
     *
     *  @param  ch  The channel to check
     *  @return  A two-element array containing the encoded type [0] and subtype [1] values.
     *  @throws  Exception if there is a parameter error
     */
    @Override
    protected int[] checkChannel(Channel ch) throws Exception
    {
        int hwChan = ch.getHwChan();
        if (hwChan < 0 || hwChan >= NUM_CHANNELS) {
            ErrorUtils.reportChannelError(LOG, ch.getPath(), "hwChan", hwChan);
        }
        channelMap.put(ch.getName(), hwChan);
        return new int[]{0, 0};
    }

    /**
     *  Reads a channel.
     *
     *  @param  ch  The channel to read
     *  @return  The read value
     */
    @Override
    protected double readChannel(Channel ch)
    {
        return channelValues.get(ch.getHwChan()) + 0.2 * (Math.random() - 0.5);
    }


    @Command(type=Command.CommandType.ACTION, level=0, description="Set a channel value")
    public void setChannelValue(@Argument(description="Channel name") String chName,
                                @Argument(description="Channel value") double value) throws UtilityException
    {
        Integer index = channelMap.get(chName);
        if (index == null) {
            throw new UtilityException("Invalid channel name: " + chName);
        }
        channelValues.set(index, value);
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="Get the list of channel names")
    public String getChannelNames()
    {
        return channelMap.keySet().toString();
    }

}
