package org.lsst.ccs.subsystem.utility;

import java.util.Set;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.RaisedAlertHistory;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.MessagingService;
import org.lsst.ccs.services.alert.AlertEvent;
import org.lsst.ccs.services.alert.AlertListener;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.actions.MpmAction;
import org.lsst.ccs.subsystem.utility.constants.MpmSwitches;

/**
 * This class manages the MPM actions in response to Alert payloads.
 *
 * It uses the AgentPresenceManager to detect which Agents on the buses might be
 * sending Alert payloads to trigger actions.
 *
 * @author The LSST CCS Team
 */
public class MpmActionManager implements HasLifecycle, AlertListener, AgentPresenceListener {

    private static final Logger LOG = Logger.getLogger(MpmActionManager.class.getName());

    @LookupField(strategy = LookupField.Strategy.TOP)
    private Agent agent;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService propertiesService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private MpmPlutoDevice plutoDevice;

    private final Set<String>[] active = new ConcurrentSkipListSet[MpmAction.Action.values().length];  // Active actions with agents

    /**
     *  Constructor.
     */
    public MpmActionManager()
    {
        for (int j = 0; j < active.length; j++) {
            active[j] = new ConcurrentSkipListSet<>();
        }
    }

    /**
     *  Build phase.
     */
    @Override
    public void build()
    {
        if (plutoDevice == null) {
            throw new RuntimeException("Could not find an instance of MpmPlutoDevice in the tree.");
        }
    }

    /**
     *  Init phase.
     */
    @Override
    public void init()
    {
        // Set a property to define agent as MPM action manager.
        propertiesService.setAgentProperty(MpmAction.getManagerName(), "");
        // Start alert listening
        alertService.startStatusAlertListening((info) -> {
            return info.hasAgentProperty(MpmAction.getClientName());
        });
    }

    /**
     *  Start phase.
     */
    @Override
    public void start()
    {
        //Alert listener responsible to listen for Alert emergency payloads
        alertService.addListener(this);
    }

    /**
     *  Shutdown phase.
     */
    @Override
    public void shutdown()
    {
        //Remove agent presence and alert listeners
        agent.getAgentService(MessagingService.class).getMessagingAccess().getAgentPresenceManager().removeAgentPresenceListener(this);
        alertService.removeListener(this);
    }

    /**
     *  Raised alert handler.
     * 
     *  @param  event  The alert event
     */
    @Override
    public void onAlert(AlertEvent event)
    {
        String origin = event.getSource();
        //LOG.log(Level.INFO, "onAlert called: origin = {0}, type = {1}", new Object[]{origin, event.getType()});
        switch (event.getType()) {
        case AGENT_CONNECTION:
            for (RaisedAlertHistory alertHistory : event.getSummary().getAllRaisedAlertHistories()) {
                processAlert(origin, alertHistory.getLatestAlert(), alertHistory.getLatestAlertState());                    
            }
            break;
        case AGENT_DISCONNECTION:
            for (MpmAction.Action action : MpmAction.Action.values()) {
                if (active[action.ordinal()].remove(origin) && active[action.ordinal()].isEmpty()) {
                    performAction(action, false);
                }
            }
            break;
        case ALERT_RAISED:
            processAlert(origin, event.getAlert(), event.getLevel());
            break;
        }

    }

    /**
     *  Processes an alert.
     * 
     *  @param  origin  Name of originating agent
     *  @param  alert  The alert
     *  @param  state  The alert state (ALARM, WARNING, NOMINAL)
     */
    private synchronized void processAlert(String origin, Alert alert, AlertState state)
    {
        MpmAction.Action action = MpmAction.getData(alert);
        if (action != null) {
            LOG.log(Level.INFO, "Received {0} level alert from {1}: action = {2}", new Object[]{state, origin, action});
            int index = action.ordinal();
            boolean on = state == AlertState.ALARM;
            if (on) {
                active[index].add(origin);
            }
            else {
                active[index].remove(origin);
                if (!active[index].isEmpty()) return;
            }
            performAction(action, on);
        }
    }

    /**
     *  Performs the requested action.
     * 
     *  @param  action  The action
     *  @param  on  Whether to turn on
     */
    private void performAction(MpmAction.Action action, boolean on)
    {
        switch (action) {
        case BLOCK_UT_POWER:
            plutoDevice.setSwitchOn(MpmSwitches.SW_BLOCK_UT_POWER, on, true);
            break;
        case BLOCK_REFRIG:
            plutoDevice.setSwitchOn(MpmSwitches.SW_BLOCK_COLD_REFG, on, true);
            plutoDevice.setSwitchOn(MpmSwitches.SW_BLOCK_CRYO_REFG, on, true);
            break;
        }
    }

}
