package org.lsst.ccs.subsystem.utility;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.DerivedChannel;

/**
 *  A DerivedChannel to compute a heat transfer rate from three
 *  specified input Channels: cooling fluid flow rate and inlet &
 *  outlet temperatures.
 *
 *  The formula is constant * flow * Delts-T, where the constant
 *  factor is settable, and should be a product of density, specific
 *  heat, and any units-adjusting factors.
 *
 *  @author The CCS team
 */
public class HeatTransferChannel extends DerivedChannel {

    @LookupPath
    private String path;

    private static final Logger LOG = Logger.getLogger(HeatTransferChannel.class.getName());

    private double factor = 1.0;
    private Channel flowChannel, inletTChannel, outletTChannel;

   /**
    *  Check that required channels have been specified.
    */
    @Override
    public void postInit() {
        if (flowChannel == null) {
            throw new RuntimeException(path + ": flowChannel has not been specified");
        }       
        if (inletTChannel == null) {
            throw new RuntimeException(path + ": inletTChannel has not been specified");
        }       
        if (outletTChannel == null) {
            throw new RuntimeException(path + ": outletTChannel has not been specified");
        }       
    }

   /**
    *  Set constant factor in evaluation formula.
    *
    *  @param  factor  The constant factor
    */
    public void setFactor(double factor) {
        this.factor = factor;
        LOG.log(Level.INFO, "{0}: Constant factor set to {1}", new Object[]{path, factor});
    }

   /**
    *  Evaluate heat-transfer rate.
    *
    *  @return  double 
    */
    @Override
    public double evaluateDerivedValue() {
        return (outletTChannel.getValue() - inletTChannel.getValue()) * flowChannel.getValue() * factor;
    }

}

