package org.lsst.ccs.subsystem.utility;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.drivers.abb.Cms;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.subsystem.common.ErrorUtils;

/**
 *  Handles a Lambda CPFE1000 power supply system.
 *
 *  @author Owen Saxton
 */
public class CmsDevice extends Device {

   /**
    *  Constants.
    */
    public static final int
        TYPE_RMS = 0,
        TYPE_AC  = 1,
        TYPE_DC  = 2;

   /**
    *  Private lookup maps.
    */
    private static final Map<String, Cms.ConnType> cTypeMap = new HashMap<>();
    static {
        cTypeMap.put("SERIAL", Cms.ConnType.SERIAL);
        cTypeMap.put("FTDI", Cms.ConnType.FTDI);
    }

    private static final Map<String, Integer> sTypeMap = new HashMap<>();
    static {
        sTypeMap.put("RMS", TYPE_RMS);
        sTypeMap.put("AC", TYPE_AC);
        sTypeMap.put("DC", TYPE_DC);
    }

   /**
    *  Data fields.
    */
    private String connType;     // Connection type string
    private String devcId;       // Device identification
    private int baudRate = 0;    // Baud rate (default)
    private int address = 1;     // Modbus address

    private static final Logger LOG = Logger.getLogger(CmsDevice.class.getName());
    private final Cms cms = new Cms();  // Associated CMS object
    private Cms.ConnType cType;
    private boolean initError = false;


   /**
    *  Performs basic initialization.
    */
    @Override
    protected void initDevice()
    {
        if (connType == null) {
            ErrorUtils.reportConfigError(LOG, name, "connType", "is missing");
        }
        cType = cTypeMap.get(connType.toUpperCase());
        if (cType == null) {
            ErrorUtils.reportConfigError(LOG, name, "connType", "is invalid");
        }
        if (devcId == null) {
            ErrorUtils.reportConfigError(LOG, name, "devcId", "is missing");
        }
        fullName = "CMS100 (" + devcId + ":" + address + ")";
    }


   /**
    *  Performs full initialization.
    */
    @Override
    protected void initialize()
    {
        try {
            cms.open(cType, devcId, baudRate);
            initSensors();
            LOG.log(Level.INFO, "Connected to {0}", fullName);
            initError = false;
            setOnline(true);
        }
        catch (DriverException e) {
            if (!initError) {
                LOG.log(Level.SEVERE, "Error connecting to {0}: {1}", new Object[]{fullName, e});
                initError = true;
            }
        }
    }


   /**
    *  Closes the connection.
    */
    @Override
    protected void close()
    {
        try {
            cms.close();
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error disconnecting from {0}: {1}", new Object[]{fullName, e});
        }
    }


   /**
    *  Checks a channel's parameters for validity.
    *
    *  @param  name     The channel name
    *  @param  hwChan   The hardware channel number
    *  @param  type     The channel type string
    *  @param  subtype  The channel subtype string
    *  @return  A two-element array containing the encoded type [0] and
    *           subtype [1] values.
    *  @throws  Exception if any errors found in the parameters.
    */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        if (hwChan <= 0 || hwChan > Cms.NUM_SENSORS) {
            ErrorUtils.reportChannelError(LOG, name, "hwchan", hwChan);
        }
        Integer sType = sTypeMap.get((type == null || type.isEmpty()) ? "RMS" : type.toUpperCase());
        if (sType == null) {
            ErrorUtils.reportChannelError(LOG, name, "type", type);
        }
        return new int[]{sType, 0};
    }


   /**
    *  Reads a channel.
    *
    *  @param  hwChan   The hardware channel number.
    *  @param  type     The encoded channel type returned by checkChannel.
    *  @return  The read value
    */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = super.readChannel(hwChan, type);

        try {
            switch (type) {
                
            case TYPE_RMS:
                value = cms.readRMS(address, hwChan); break;
                
            case TYPE_AC:
                value = cms.readAC(address, hwChan); break;
                
            case TYPE_DC:
                value = cms.readDC(address, hwChan); break;
            }
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error reading CMS100 data: {0}", e);
            setOnline(false);
        }

        return value;
    }

}
