package org.lsst.ccs.subsystem.utility;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.drivers.auxelex.Bfr;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.utility.constants.UtilityConfig;
import org.lsst.ccs.subsystem.utility.constants.PurgeTestSwitches;

/**
 *  Interface to the SLAC BFR.
 *
 *  @author Owen Saxton
 */
public class BfrDevicePT extends Device implements SwitchControl {

    /**
     *  Constants.
     */
    public static final int
        CHAN_HEATER_1    = 0,
        CHAN_HEATER_2    = 1,
        CHAN_24V_PDU     = 3,
        CHAN_48V_PDU     = 2,
        CHAN_HEATER_SIM  = 4;

    private static final String
        NODE   = "node";
    private static final Logger LOG = Logger.getLogger(BfrDevicePT.class.getName());
    private boolean initError = false;


    /**
     *  Data fields.
     */
    @ConfigurationParameter(name=NODE, category=UtilityConfig.CATEGORY_UTIL, isFinal=true)
    private volatile Integer node;

    private final Bfr bfr = new Bfr();
    private double currents[] = null;


    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice() {
        if (node == null) {
            ErrorUtils.reportConfigError(LOG, name, NODE, "is missing");
        }
        fullName = "BFR board (" + node + ")";
    }


    /**
     *  Performs full initialization.
     */
    @Override
    public void initialize()
    {
        try {
            bfr.open(node);
            initSensors();
            LOG.log(Level.INFO, "Connected to {0}", fullName);
            initError = false;
            setOnline(true);
        }
        catch (DriverException e) {
            if (!initError) {
                LOG.log(Level.SEVERE, "Error connecting to {0}: {1}", new Object[]{fullName, e});
                initError = true;
            }
        }
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        try {
            bfr.close();
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error disconnecting from {0}: {1}", new Object[]{fullName, e});
        }
    }


    /**
     *  Checks a monitoring channel's parameters for validity.
     *
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string
     *  @return  Two-element array containing the encoded type [0] and subtype [1] 
     *  @throws  Exception  If parameters are invalid
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        if (hwChan < 0 || hwChan >= Bfr.NUM_SENSORS) {
            ErrorUtils.reportChannelError(LOG, name, "hw channel number", hwChan);
        }
        return new int[]{0, 0};
    }


    /**
     *  Reads all monitored channels.
     *
     */
    @Override
    protected void readChannelGroup()
    {
        currents = null;
        if (isOnline()) {
            try {
                currents = bfr.readCurrent(Bfr.CurrentType.AC);
            }
            catch (DriverException e) {
                LOG.log(Level.SEVERE, "Error reading {0} currents: {1}", new Object[]{fullName, e});
                setOnline(false);
            }
        }
    }


    /**
     *  Reads a monitored channel.
     *
     *  @param  hwChan  The hardware channel number
     *  @param  type    The encoded channel type
     *  @return  The read value
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        return currents == null ? Double.NaN : currents[hwChan];
    }


    /**
     *  Gets the switch device type.
     * 
     *  @return  The switch device type enumeration
     */
    @Override
    public int getSwitchDevice()
    {
        return PurgeTestSwitches.DEVC_BFR;
    }


    /**
     *  Turns a relay on.
     *
     *  @param  relay  The relay number
     */
    @Override
    public void switchOn(int relay)
    {
        try {
            bfr.setRelayOn(relay);
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error setting relay {0} on {1}: {2}", new Object[]{relay, fullName, e});
        }
        
    }


    /**
     *  Turns a relay off.
     *
     *  @param  relay  The relay number
     */
    @Override
    public void switchOff(int relay)
    {
        try {
            bfr.setRelayOff(relay);
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error setting relay {0} on {1}: {2}", new Object[]{relay, fullName, e});
        }
        
    }


    /**
     *  Gets the state of a relay.
     *
     *  @param  relay  The relay number
     *  @return  Whether the relay is on or off
     */
    @Override
    public Boolean isSwitchOn(int relay)
    {
        if (!isOnline() || relay < 0) return null;
        try {
            return bfr.isRelayOn(relay);
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error getting state for relay {0} on {1}: {2}", new Object[]{relay, fullName, e});
            return null;
        }
    }

}
