package org.lsst.ccs.subsystem.utility;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.wattsup.WattsUp;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.subsystem.common.ErrorUtils;

/**
 *  Handles a WattsUp power meter.
 *
 *  @author Owen Saxton
 */
public class WattsUpDevice extends Device implements WattsUp.Listener {

   /**
    *  Public constants.
    */
    public static final int
        CHAN_WATTS  = 0,
        CHAN_VOLTS  = 1,
        CHAN_FREQ   = 2;

   /**
    *  Data fields.
    */
    private static final int
        N_DEV_CHANS = 3,
        LOG_PERIOD  = 1;

    private static final Map<String, WattsUp.ConnType> cTypeMap = new HashMap<>();
    static {
        cTypeMap.put("SERIAL", WattsUp.ConnType.SERIAL);
        cTypeMap.put("FTDI",   WattsUp.ConnType.FTDI);
    }

    private String connType;   // Connection type string (SERIAL or FTDI)
    private String devcId;     // Identifier string (device name or serial number)

    private static final Logger LOG = Logger.getLogger(WattsUpDevice.class.getName());
    private WattsUp.ConnType connTypeE;
    private final WattsUp wtu = new WattsUp();    // WattsUp hardware object
    private final double[] value = new double[N_DEV_CHANS];  // Current values
    private boolean initError = false;


   /**
    *  Performs basic initialization.
    */
    @Override
    protected void initDevice()
    {
        if (connType == null) {
            ErrorUtils.reportConfigError(LOG, name, "connType", "is missing");
        }
        connTypeE = cTypeMap.get(connType.toUpperCase());
        if (connTypeE == null) {
            ErrorUtils.reportConfigError(LOG, name, "connType", "is invalid: " + connType);
        }
        if (devcId == null) {
            ErrorUtils.reportConfigError(LOG, name, "devcId", "is missing");
        }
        wtu.addListener(this);
        fullName = "WattsUp meter (" + devcId + ")";
    }


   /**
    *  Performs full initialization.
    */
    @Override
    protected void initialize()
    {
        try {
            wtu.open(connTypeE, devcId);
            wtu.restart();
            initSensors();
            setOnline(true);
            LOG.log(Level.INFO, "Connected to {0}", fullName);
            initError = false;
        }
        catch (DriverException e) {
            if (!initError) {
                LOG.log(Level.SEVERE, "Error connecting to {0}: {1}", new Object[]{fullName, e});
                initError = true;
            }
        }
    }


   /**
    *  Closes the connection.
    *
    *  This is a no-op since the driver closes the connection automatically
    *  when a connection problem is detected.
    */
    @Override
    protected void close()
    {
    }


   /**
    *  Checks a channel's parameters for validity.
    *
    *  @param  name     The channel name
    *  @param  hwChan   The hardware channel number
    *  @param  type     The channel type string
    *  @param  subtype  The channel subtype string
    *  @return  A two-element array containing the encoded type [0] and
    *           subtype [1] values.
    *  @throws  Exception if any errors found in the parameters.
    */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        if (hwChan < 0 || hwChan >= N_DEV_CHANS) {
            ErrorUtils.reportChannelError(LOG, name, "hw channel number", hwChan);
        }
        return new int[]{0, 0};
    }


   /**
    *  Initializes a channel.
    *
    *  @param  chan     The hardware channel number
    *  @param  type     The channel type string
    *  @param  subtype  The channel subtype string
    */
    @Override
    protected void initChannel(int chan, int type, int subtype)
    {
    }


   /**
    *  Reads a channel.
    *
    *  @param  chan  The hardware channel number.
    *  @param  type  The encoded channel type returned by checkChannel.
    *  @return  The value read from the channel
    */
    @Override
    protected double readChannel(int chan, int type)
    {
        return isOnline() ? value[chan] : super.readChannel(chan, type);
    }


   /**
    *  Changes the WattsUp? meter powered state.
    *
    *  Performs meter setup upon power-on.
    *
    *  @param  on  Whether the meter is powered on or not
    */
    @Override
    public void setPowered(boolean on)
    {
        if (on) {
            try {
                wtu.setLoggedFields((1 << WattsUp.FLD_WATTS) |
                                    (1 << WattsUp.FLD_VOLTS) |
                                    (1 << WattsUp.FLD_FREQUENCY));
                wtu.setExternalLogging(LOG_PERIOD);
            }
            catch (DriverException e) {
                LOG.log(Level.SEVERE, "Error configuring {0}: {1}", new Object[]{fullName, e});
            }
        }
        else {
            for (int j = 0; j < N_DEV_CHANS; j++) {
                value[j] = 0;
            }
        }
    }


   /**
    *  Changes the WattsUp? meter open state.
    */
    @Override
    public void setClosed()
    {
        setOnline(false);
    }


   /**
    *  Receives data periodically from the WattsUp? meter.
    *
    *  @param  data  The array of data from the meter.
    */
    @Override
    public void processData(double[] data)
    {
        value[CHAN_WATTS] = data[WattsUp.FLD_WATTS];
        value[CHAN_VOLTS] = data[WattsUp.FLD_VOLTS];
        value[CHAN_FREQ]  = data[WattsUp.FLD_FREQUENCY];
    }

}
