package org.lsst.ccs.subsystem.utility;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.localdb.statusdb.server.TrendingData;
import org.lsst.ccs.localdb.trendserver.TrendingClientService;
import org.lsst.ccs.localdb.utils.TrendingConnectionUtils;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.DerivedChannel;

/**
 * A derived channel to accumulate the on time for a channel
 * 
 * @author The LSST CCS Team
 */
public class AccumTimeChannel extends DerivedChannel
{
    @LookupPath
    private String path;
    @LookupField(strategy = LookupField.Strategy.TOP)
    private Subsystem subsys;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private TrendingClientService trendingService;

    private Channel channel;  // From Groovy file

    private static final Logger LOG = Logger.getLogger(AccumTimeChannel.class.getName());
    private long readTime = 0;
    private double accumTime = 0.0;

    /**
     *  Init phase
     */
    @Override
    public void init()
    {        
        if (channel == null) {
            throw new RuntimeException(getPath() + " parameter error: channel has not been defined");
        }
        super.init();
    }

    /**
     *  Start phase
     */
    @Override
    public void start()
    {
        TrendingConnectionUtils trendingUtils = trendingService.getBusTrendingConnection();
        if (trendingUtils == null) {
            LOG.log(Level.SEVERE, "Trending data cannot be accessed");
        }
        else {
            String chName = subsys.getName() + "/" + path;
            TrendingData data = trendingUtils.getLatestData(chName);
            accumTime = data == null ? Double.NaN : data.getValue("value");
            if (Double.isNaN(accumTime)) {
                LOG.log(Level.WARNING, "No trending data present for channel {0}", chName);
                accumTime = 0.0;
            }
            else {
                LOG.log(Level.INFO, "Got trending data for channel {0}", chName);
            }
        }
        super.start();
    }

    /**
     *  Calculate the value.
     * 
     *  @return  The value
     */
    @Override
    public double evaluateDerivedValue()
    {
        long currTime = System.currentTimeMillis();
        if (readTime != 0 && channel.getValue() > 0.0) {  // Fails test if NaN
            accumTime += (currTime - readTime) / 3.6e6;   // Add hours
        }
        readTime = currTime;
        return accumTime;
    }

    @Command(type=Command.CommandType.ACTION, description="Set the accumulated time")
    public void setAccumTime(@Argument(description="Time period (hrs)") double time)
    {
        accumTime = time;
    }

}
