package org.lsst.ccs.subsystem.utility;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.common.PIController;
import org.lsst.ccs.subsystem.utility.data.UtilityException;

/**
 *  Implements a temperature controller for the volume purge cabinet.
 *
 *  @author Owen Saxton
 */
public class VpcPIControl implements HasLifecycle {

    private static final String PIC = "Pic";

    @LookupField(strategy = LookupField.Strategy.TOP)
    Subsystem subsys;
    @LookupName
    private String name;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskServices;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private final Map<String, Channel> allChannels = new HashMap<>();
    @LookupField(strategy = LookupField.Strategy.TREE)
    private PurgeMaq20Device maqDevc;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private BfrDevice bfrDevc;
    
    @ConfigurationParameter(category=PIC, isFinal=true)
    private Double  gain = Double.NaN;              // loop gain
    @ConfigurationParameter(category=PIC, isFinal=true)
    private Double  timeConst = Double.NaN;         // integration time constant (secs)
    @ConfigurationParameter(category=PIC, isFinal=true)
    private Double  smoothTime = Double.NaN;        // input smoothing time (secs)
    @ConfigurationParameter(category=PIC, isFinal=true)
    private Double  awGain = Double.NaN;            // anti-windup gain
    @ConfigurationParameter(category=PIC, isFinal=true)
    private Double  baseDuty = Double.NaN;          // base duty cycle input
    @ConfigurationParameter(category=PIC, isFinal=true)
    private Double  tolerance = Double.NaN;         // maximum on-target error (%)
    @ConfigurationParameter(category=PIC, isFinal=true)
    private double  minInput = -10.0;               // minimum input
    @ConfigurationParameter(category=PIC, isFinal=true)
    private double  maxInput = 10.0;                // maximum input
    @ConfigurationParameter(category=PIC, isFinal=true)
    private double  minOutput = -500.0;             // minimum PID output (cold valve fully open)
    @ConfigurationParameter(category=PIC, isFinal=true)
    private Double  maxOutput = 500.0;              // maximum PID output (both heaters on)
    @ConfigurationParameter(category=PIC, isFinal=true)
    private Integer updateTime = Integer.MAX_VALUE; // The update time interval (msec)

    private String[] refTempChans;     // Reference temperature channels to use
    private String[] ctrlTempChans;    // Controltemperature channels to use
    private double coolCapacity = 220.0;  // Maximum cooling power available
    private double heatCapacity = 150.0;  // Heating power of each heater

    private final List<Channel> refTempChanList = new ArrayList<>();   // Reference temperature channels to use
    private final List<Channel> ctrlTempChanList = new ArrayList<>();  // Control temperature channels to use

    private static final Logger LOG = Logger.getLogger(VpcPIControl.class.getName());
    private PIController pic;          // The PI controller
    private double lastPower;          // The last output power value set (pos = heat, neg = cool)
    private boolean active = false;    // True if loop is active
    private double setTemp;            // Temperature set point
    private boolean loopFailed = false;
    private int numHeaters = 0;
    private long lastLoopTime;


    /**
     *  Sets up the VPC temperature control loop timer task.
     */
    @Override
    public void build() {
        AgentPeriodicTask apt = new AgentPeriodicTask("vpc-loop-" + name,
                                                      () -> iterateLoop()).withPeriod(Duration.ofMillis(500));
        periodicTaskServices.scheduleAgentPeriodicTask(apt);
    }


    /**
     *  Initializes the parameters.
     */
    @Override
    public void init()
    {
        if (maqDevc == null) {
            ErrorUtils.reportConfigError(LOG, name, "PurgeMaq20Device component", "is missing");
        }
        if (bfrDevc == null) {
            ErrorUtils.reportConfigError(LOG, name, "BfrDevice component", "is missing");
        }
        if (refTempChans == null || refTempChans.length == 0) {
            ErrorUtils.reportConfigError(LOG, name, "refTempChans", "is missing or empty");
        }
        else {
            for (String cName : refTempChans) {
                Channel cmpt = allChannels.get(cName);
                if (cmpt != null) {
                    refTempChanList.add(cmpt);
                }
                else {
                    ErrorUtils.reportConfigError(LOG, name, "refTempChans", "contains non-Channel item");
                }
            }
        }
        if (ctrlTempChans == null || ctrlTempChans.length == 0) {
            ErrorUtils.reportConfigError(LOG, name, "ctrlTempChans", "is missing or empty");
        }
        else {
            for (String cName : ctrlTempChans) {
                Channel cmpt = allChannels.get(cName);
                if (cmpt != null) {
                    ctrlTempChanList.add(cmpt);
                }
                else {
                    ErrorUtils.reportConfigError(LOG, name, "ctrlTempChans", "contains non-Channel item");
                }
            }
        }

        if (gain == null) {
            ErrorUtils.reportConfigError(LOG, name, "gain", "is missing");
        }
        if (timeConst.isNaN()) {
            ErrorUtils.reportConfigError(LOG, name, "timeConst", "is missing");
        }
        if (smoothTime.isNaN()) {
            ErrorUtils.reportConfigError(LOG, name, "smoothTime", "is missing");
        }
        if (awGain.isNaN()) {
            ErrorUtils.reportConfigError(LOG, name, "awGain", "is missing");
        }
        if (baseDuty.isNaN()) {
            ErrorUtils.reportConfigError(LOG, name, "baseDuty", "is missing");
        }
        if (tolerance.isNaN()) {
            ErrorUtils.reportConfigError(LOG, name, "tolerance", "is missing");
        }
        if (updateTime == Integer.MAX_VALUE) {
            ErrorUtils.reportConfigError(LOG, name, "updateTime", "is missing");
        }
        pic = new PIController(gain, timeConst);
        pic.setSmoothTime(smoothTime);
        pic.setAwGain(awGain);
        pic.setBaseOutput(baseDuty);
        pic.setInputRange(minInput, maxInput);
        pic.setOutputRange(minOutput, maxOutput);
        pic.setTolerance(tolerance);
    }


    /**
     *  Initializes the heater state.
     */
    @Override
    public void postStart()
    {
        Boolean heater1On = bfrDevc.isSwitchOn(BfrDevice.CHAN_HEATER_1);
        Boolean heater2On = bfrDevc.isSwitchOn(BfrDevice.CHAN_HEATER_2);
        if (heater1On == Boolean.TRUE && heater2On == Boolean.TRUE) {
            numHeaters = 2;
        }
        else {
            if (heater2On == Boolean.TRUE) {
                bfrDevc.switchOff(BfrDevice.CHAN_HEATER_2);
                bfrDevc.switchOn(BfrDevice.CHAN_HEATER_1);
                heater1On = true;
            }
            numHeaters = heater1On == Boolean.TRUE ? 1 : 0;
        }
    }


    /**
     *  Sets the power to be produced.
     *
     *  @param  power  The power value: pos = heat, neg = cool
     */
    public void setPower(double power)
    {
        if (numHeaters > 0 && power < -(numHeaters + 0.5) * heatCapacity) {
            bfrDevc.switchOff(numHeaters == 1 ? BfrDevice.CHAN_HEATER_1 : BfrDevice.CHAN_HEATER_2);
            numHeaters--;
        }
        else if (numHeaters < 2 && power > numHeaters * heatCapacity) {
            bfrDevc.switchOn(numHeaters == 0 ? BfrDevice.CHAN_HEATER_1 : BfrDevice.CHAN_HEATER_2);
            numHeaters++;
        }
        try {
            maqDevc.setVpcValve((-power + numHeaters * heatCapacity) / coolCapacity);
        }
        catch (Exception ex) {
            LOG.log(Level.SEVERE, "Error setting VPC valve position: {0}", ex.getMessage());
        }
    }


    /**
     *  Sets the target temperature.
     *
     *  @param  temp  The temperature to set
     *  @throws  UtilityException
     */
    public void setTemperature(double temp) throws UtilityException
    {
        setTemp = temp;
        pic.setSetpoint(setTemp);
    }


    /**
     *  Gets the target temperature.
     *
     *  @return  The set temperature
     */
    public double getTemperature()
    {
        return setTemp;
   }


    /**
     *  Sets the gain.
     *
     *  @param  gain  The gain to set
     *  @throws  UtilityException
     */
    public void setGain(double gain) throws UtilityException
    {
        this.gain = gain;
        if (pic != null) {  // Can get called during startup before pic exists
            pic.setPID(gain, timeConst);
        }
    }


    /**
     *  Gets the gain.
     *
     *  @return  The gain
     */
    public double getGain()
    {
        return gain;
    }


    /**
     *  Sets the time constant.
     *
     *  @param  time  The time constant to set
     *  @throws  UtilityException
     */
    public void setTimeConstant(double time) throws UtilityException
    {
        timeConst = time;
        pic.setPID(gain, timeConst);
    }


    /**
     *  Gets the time constant.
     *
     *  @return  The time constant
     */
    public double getTimeConstant()
    {
        return timeConst;
   }


    /**
     *  Starts the control loop.
     *
     *  @param  power  The initial power value
     *  @throws  UtilityException
     */
    public void startLoop(double power) throws UtilityException
    {
        if (active) return;
        lastPower = power;
        startLoop();
    }


    /**
     *  Starts the control loop.
     *
     *  @throws  UtilityException
     */
    public void startLoop() throws UtilityException
    {
        if (active) return;
        pic.reset();
        pic.setIntegral(lastPower - baseDuty);
        active = true;
    }


    /**
     *  Stops the control loop.
     *
     *  @throws  UtilityException
     */
    public void stopLoop() throws UtilityException
    {
        if (!active) return;
        active = false;
    }


    /**
     *  Gets the control loop state.
     *
     *  @return  Whether the control loop is active
     */
    public boolean isLoopActive()
    {
        return active;
    }


    /**
     *  Resets the controller.
     * 
     *  @throws  UtilityException
     */
    public void reset() throws UtilityException
    {
        pic.reset();
    }


    /**
     *  Timer routine for control loop iteration.
     */
    private void iterateLoop()
    {
        if (!active) return;
        long time = System.currentTimeMillis();
        if (time < lastLoopTime + updateTime) return;
        lastLoopTime = time;
        double refTemp = 0.0, ctrlTemp = 0.0;
        int count = 0;
        for (Channel tempChan : refTempChanList) {
            double value = tempChan.getValue();
            if (!Double.isNaN(value)) {
                refTemp += value;
                count++;
            }
        }
        if (count > 0) {
            refTemp /= count;
        }
        else {
            if (!loopFailed) {
                LOG.severe("Control loop iteration failed: no valid reference temperature values available");
                loopFailed = true;
            }
            return;
        }
        count = 0;
        for (Channel tempChan : ctrlTempChanList) {
            double value = tempChan.getValue();
            if (!Double.isNaN(value)) {
                ctrlTemp += value;
                count++;
            }
        }
        if (count > 0) {
            ctrlTemp /= count;
        }
        else {
            if (!loopFailed) {
                LOG.severe("Control loop iteration failed: no valid control temperature values available");
                loopFailed = true;
            }
            return;
        }
        lastPower = pic.performPI(new double[]{ctrlTemp - refTemp}, time / 1000.0);
        setPower(lastPower);
        loopFailed = false;
    }

}
