package org.lsst.ccs.subsystem.utility;

import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.common.devices.dataforth.Maq20AnalogControl;
import org.lsst.ccs.subsystem.common.devices.dataforth.Maq20Device;
import org.lsst.ccs.subsystem.utility.constants.HeaterState;
import org.lsst.ccs.subsystem.utility.constants.PurgeTestHeaters;
import org.lsst.ccs.subsystem.utility.data.UtilityException;

/**
 *  Controls Maq20 device
 * 
 *  @author saxton
 */
public class PurgeMaq20Device extends Maq20Device {

    public static final int
        CHAN_SIM_HEATER = 0,
        CHAN_VPC_VALVE  = 1,
        CHAN_UT_VALVE   = 2;

    private static final double[][] convAmpsToVFract =
       {{0.004, 0.000}, {0.005, 0.000}, {0.006, 0.145}, {0.007, 0.190}, {0.008, 0.248}, {0.009, 0.302},
        {0.010, 0.371}, {0.011, 0.394}, {0.012, 0.459}, {0.013, 0.541}, {0.014, 0.624}, {0.015, 0.625},
        {0.016, 0.655}, {0.017, 0.744}, {0.018, 0.787}, {0.019, 0.858}, {0.020, 1.000}};
               
    @LookupField (strategy=LookupField.Strategy.ANCESTORS)
    private PurgeTestMain main;
    @LookupField (strategy=LookupField.Strategy.CHILDREN)
    private Maq20AnalogControl maqCtrl;

    private static final Logger LOG = Logger.getLogger(PurgeMaq20Device.class.getName());
    private int modIndex;


    /**
     *  Basic initialization.
     */
    @Override
    public void initDevice()
    {
        super.initDevice();
        if (maqCtrl == null) {
            ErrorUtils.reportConfigError(LOG, name, "analog control", "not defined");
        }
        modIndex = maqCtrl.getModIndex();
    }


   /**
    *  Reads a channel.
    *
    *  @param  hwChan   The hardware channel number.
    *  @param  type     The encoded channel type returned by checkChannel.
    *  @return  Channel value
    */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = super.readChannel(hwChan, type);
        if ((type & 0xff) == modIndex) {
            if (hwChan == CHAN_VPC_VALVE || hwChan == CHAN_UT_VALVE) {
                value = 62.5 * value - 0.25;
            }
            else if (hwChan == CHAN_SIM_HEATER) {
                HeaterState state = main.getFullState().getHeaterState(PurgeTestHeaters.HEATER_SIM_ID);
                value = state == HeaterState.OFF ? 0.0 : state == HeaterState.OFFLINE ? Double.NaN
                          : convVFractAndAmps(false, value);
            }
        }
        return value;
    }


    /**
     *  Sets the VPC valve position.
     *
     *  @param  posn  Requested position (0 - 1)
     *  @throws  UtilityException
     */
    public void setVpcValve(double posn) throws UtilityException
    {
        setValve(CHAN_VPC_VALVE, posn);
    }


    /**
     *  Sets the UT valve position.
     *
     *  @param  posn  Requested position (0 - 1)
     *  @throws  UtilityException
     */
    public void setUtValve(double posn) throws UtilityException
    {
        setValve(CHAN_UT_VALVE, posn);
    }


    /**
     *  Sets a valve position.
     *
     *  @param  chan  The channel number
     *  @param  posn  Requested position (0 - 1)
     *  @throws  UtilityException
     */
    public void setValve(int chan, double posn) throws UtilityException
    {
        double value = 0.016 * Math.min(Math.max(posn, 0.0), 1.0) + 0.004;
        try {
            maqCtrl.writeValue(chan, value);
        }
        catch (DriverException e) {
            throw new UtilityException("Error operating valve (channel " + chan + "): " + e);
        }
    }


    /**
     *  Sets the simulator heater duty cycle (voltage fraction).
     *
     *  @param  duty  Requested duty cycle (0 - 1)
     *  @throws  UtilityException     * 
     */
    public void setSimHeater(double duty) throws UtilityException
    {
        try {
            maqCtrl.writeValue(CHAN_SIM_HEATER, convVFractAndAmps(true, duty));
        }
        catch (DriverException e) {
            throw new UtilityException("Error operating simulator heater: " + e);
        }
    }


    /**
     *  Converts between voltage fraction and current
     * 
     *  @param  toAmps  True if converting voltage to amps; false otherwise
     *  @param  value   The value to convert
     *  @return  The converted value
     */
    private double convVFractAndAmps(boolean toAmps, double value)
    {
        double conv;
        int in = toAmps ? 1 : 0;
        int out = 1 - in;
        int j = 0;
        double[][] convTable = convAmpsToVFract;
        for (; j < convTable.length; j++) {
            if (convTable[j][in] >= value) break;
        }
        if (j == 0) {
            conv = convTable[j][out];
        }
        else if (j >= convTable.length) {
            conv = convTable[j - 1][out];
        }
        else {
            double[] cPrev = convTable[j - 1];
            double[] cCurr = convTable[j];
            conv = cPrev[out] + (cCurr[out] - cPrev[out]) / (cCurr[in] - cPrev[in]) * (value - cPrev[in]);
        }
        return conv;
    }

}
