package org.lsst.ccs.subsystem.utility;

import java.util.Arrays;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.subsystem.common.devices.pluto.PlutoDevice;
import org.lsst.ccs.subsystem.utility.constants.MpmConditions;
import org.lsst.ccs.subsystem.utility.constants.MpmLatches;
import org.lsst.ccs.subsystem.utility.constants.MpmLimits;
import org.lsst.ccs.subsystem.utility.constants.MpmPlcs;

/**
 *  Handles the protection system Pluto PLC.
 *
 *  @author Owen Saxton
 */
public class MpmPlutoDevice extends PlutoDevice {

    /**
     *  Constants.
     */
    public static final int
        SW_BLOCK_REB_POWER  = 0,
        SW_BLOCK_UT_POWER   = 1,
        SW_BLOCK_COOLANT    = 2,
        SW_BLOCK_COLD_HEAT  = 3,
        SW_BLOCK_COLD_REFG  = 4,
        SW_BLOCK_CRYO_HEAT  = 5,
        SW_BLOCK_CRYO_REFG  = 6,
        NUM_SWITCHES        = 7;

    private static final int
        NUM_AREAS = 30,
        SWDI_ON_BIT = 0,
        SWDI_OFF_BIT = 1,
        SWDI_READ_AREA = 2,
        SWDI_READ_BIT = 3,
        LTDI_RESET_BIT = 0,
        LTDI_READ_AREA = 1,
        LTDI_READ_BIT = 2,
        LTDI_PEND_AREA = 3,
        LTDI_PEND_BIT = 4,
        CNDI_READ_AREA = 0,
        CNDI_READ_BIT = 1;
    private static final int
        OFFSET_CELSIUS = -273;

    /**
     *  Private lookup maps, etc.
     */
    private static final int[] statusMask = new int[MpmPlcs.NUM_PLCS];
    static {
        statusMask[MpmPlcs.PLC_TRUNK] = 0x02;
        statusMask[MpmPlcs.PLC_COLD]  = 0x04;
        statusMask[MpmPlcs.PLC_CRYO]  = 0x08;
    }
    private static final int[][] switches = new int[NUM_SWITCHES][];
    static {                              // ON bit, OFF bit, read area, read bit 
        switches[SW_BLOCK_REB_POWER] = new int[]{3, 4, 3, 6};
        switches[SW_BLOCK_UT_POWER]  = new int[]{5, 6, 3, 1};
        switches[SW_BLOCK_COOLANT]   = new int[]{7, 8, 3, 11};
        switches[SW_BLOCK_COLD_HEAT] = new int[]{18, 19, 16, 6};
        switches[SW_BLOCK_COLD_REFG] = new int[]{20, 21, 17, 6};
        switches[SW_BLOCK_CRYO_HEAT] = new int[]{34, 35, 26, 6};
        switches[SW_BLOCK_CRYO_REFG] = new int[]{36, 37, 27, 6};
    }
    private static final int[][] latches = new int[MpmLatches.NUM_LATCHES][];
    static {                                        // Reset bit, on area, on bit, ind area, ind bit
        latches[MpmLatches.LATCH_UT_TEMP]        = new int[]{2, 0, 6, 0, 9};   
        latches[MpmLatches.LATCH_UT_LEAK]        = new int[]{1, 1, 8, 1, 14};  //    
        latches[MpmLatches.LATCH_UT_LEAK_FAULT]  = new int[]{1, 1, 5, 1, 13};
        latches[MpmLatches.LATCH_UT_SMOKE]       = new int[]{0, 2, 7, 2, 15};
        latches[MpmLatches.LATCH_UT_SMOKE_FAULT] = new int[]{0, 2, 4, 2, 14};
        latches[MpmLatches.LATCH_COLD_TEMP_HIGH] = new int[]{16, 16, 2, 16, 4};
        latches[MpmLatches.LATCH_COLD_TEMP_LOW]  = new int[]{17, 17, 2, 17, 4};
        latches[MpmLatches.LATCH_CRYO_TEMP_HIGH] = new int[]{32, 26, 2, 26, 4};
        latches[MpmLatches.LATCH_CRYO_TEMP_LOW]  = new int[]{33, 27, 2, 27, 4};
        latches[MpmLatches.LATCH_CRYO_VACUUM]    = new int[]{39, 28, 7, 28, 10};
        latches[MpmLatches.LATCH_HEX_VACUUM]     = new int[]{38, 28, 0, 28, 3};
    }
    private static final int[][] conditions = new int[MpmConditions.NUM_CONDITIONS][];
    static {                                                // Area, bit
        conditions[MpmConditions.COND_UT_POWER]  = new int[]{3, 0};  
        conditions[MpmConditions.COND_REB_POWER] = new int[]{3, 5};
        conditions[MpmConditions.COND_COOLANT]   = new int[]{3, 10};
        conditions[MpmConditions.COND_COLD_REFG] = new int[]{17, 9};
        conditions[MpmConditions.COND_COLD_HEAT] = new int[]{16, 9};
        conditions[MpmConditions.COND_CRYO_REFG] = new int[]{27, 9};
        conditions[MpmConditions.COND_CRYO_HEAT] = new int[]{26, 9};
    }

    /**
     *  Data fields.
     */
    private final int[] tempLimits = new int[MpmLimits.NUM_LIMITS];
    private boolean limitsRead = false;


    /**
     *   Constructor.
     */
    public MpmPlutoDevice()
    {
        super(NUM_AREAS);
        Arrays.fill(tempLimits, Integer.MAX_VALUE);
    }


    /**
     *  Tests whether a PLC is active
     *
     *  @param  plc  The PLC number
     *  @return  Whether the PLC is active, or null if offline
     */
    public Boolean isPlcActive(int plc)
    {
        try {
            int status = plu.readModuleStatus();
            return (status & statusMask[plc]) != 0;
        }
        catch (DriverException e) {
            return null;
        }
    }


    /**
     *  Sets a switch on or off.
     *
     *  For the vacuum Pluto, this is implemented as a pair of push buttons,
     *  one for on, one for off.
     *
     *  @param  sw  The switch number.
     *  @param  on  The on state to set: true or false
     */
    public void setSwitchOn(int sw, boolean on)
    {
        int bitNum = switches[sw][on ? SWDI_ON_BIT : SWDI_OFF_BIT];
        toggleBit(bitNum / 16, bitNum & 0x0f);
    }


    /**
     *  Gets the on state of a switch.
     *
     *  The state is not the state of the bit that was toggled, but is read back
     *  either directly from the controlled hardware, or from the PLC output line.
     *
     *  @param  sw  The switch number.
     *  @return  Whether the switch is on
     */
    public Boolean isSwitchOn(int sw)
    {
        int[] swData = switches[sw];
        Integer value = readAddBit(swData[SWDI_READ_AREA], swData[SWDI_READ_BIT] + 16);
        return value != null ? value != 0 : null;
    }


    /**
     *  Gets whether a latched condition is active.
     * 
     *  @param  cond  The condition number
     *  @return  Whether active - indicated by the bit being 0
     */
    public Boolean isLatchActive(int cond)
    {
        int[] condData = latches[cond];
        Integer value = readAddBit(condData[LTDI_READ_AREA], condData[LTDI_READ_BIT] + 16);
        return value != null ? value == 0 : null;
    }


    /**
     *  Gets whether a latched condition is latched.
     * 
     *  @param  cond  The condition number
     *  @return  Whether latched - indicated by the bit being 1
     */
    public Boolean isLatchLatched(int cond)
    {
        int[] condData = latches[cond];
        Integer value = readAddBit(condData[LTDI_PEND_AREA], condData[LTDI_PEND_BIT] + 16);
        return value != null ? value != 0 : null;
    }


    /**
     *  Clears a latched condition.
     * 
     *  @param  cond  The condition number
     */
    public void clearLatch(int cond)
    {
        int bitNum = latches[cond][LTDI_RESET_BIT];
        toggleBit(bitNum / 16, bitNum & 0x0f);
    }


    /**
     *  Gets whether a condition is active.
     * 
     *  @param  cond  The condition number
     *  @return  Whether active - indicated by the bit being 1
     */
    public Boolean isConditionActive(int cond)
    {
        int[] condData = conditions[cond];
        Integer value = readAddBit(condData[CNDI_READ_AREA], condData[CNDI_READ_BIT] + 16);
        return value != null ? value != 0 : null;
    }


    /**
     *  Gets the temperature limits.
     *
     *  @return  The 4-element array of temperature limits
     */
    public int[] getTempLimits()
    {
        if (!limitsRead && isOnline()) {
            tempLimits[MpmLimits.LIMIT_COLD_HIGH] = readAddWord(14, 0) + OFFSET_CELSIUS;
            tempLimits[MpmLimits.LIMIT_COLD_LOW] = readAddWord(14, 1) + OFFSET_CELSIUS;
            tempLimits[MpmLimits.LIMIT_CRYO_HIGH] = readAddWord(24, 0) + OFFSET_CELSIUS;
            tempLimits[MpmLimits.LIMIT_CRYO_LOW] = readAddWord(24, 1) + OFFSET_CELSIUS;
            limitsRead = true;
        }
        return tempLimits;
    }

}
