package org.lsst.ccs.subsystem.utility;

import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.common.devices.dataforth.Maq20PWMControl;

/**
 *  Implements control of a PWM-controlled fan
 *
 *  @author saxton
 */
public class FanControl extends Maq20PWMControl {

    private static final Logger LOG = Logger.getLogger(FanControl.class.getName());

    @ConfigurationParameter(category="Device", isFinal=true)
    private Integer minRpm;
    @ConfigurationParameter(category="Device", isFinal=true)
    private Integer maxRpm;
    @ConfigurationParameter(category="Device", isFinal=true)
    private Double minDutyCycle;
    @ConfigurationParameter(category="Device", isFinal=true)
    private Double maxDutyCycle;

    private double dcSlope;


    /**
     *  Life-cycle initialization.
     */
    @Override
    public void init()
    {
        super.init();
        if (minRpm == null) {
            ErrorUtils.reportConfigError(LOG, name, "minRpm", "is missing");
        }
        if (maxRpm == null) {
            ErrorUtils.reportConfigError(LOG, name, "maxRpm", "is missing");
        }
        if (minDutyCycle == null) {
            ErrorUtils.reportConfigError(LOG, name, "minDutyCycle", "is missing");
        }
        if (maxDutyCycle == null) {
            ErrorUtils.reportConfigError(LOG, name, "maxDutyCycle", "is missing");
        }
        dcSlope = (maxDutyCycle - minDutyCycle) / (maxRpm - minRpm);
        if (!Double.isFinite(dcSlope)) {
            ErrorUtils.reportConfigError(LOG, name, "conversion constants", "are invalid");
        }
    }


    /**
     *  Sets the RPM
     *
     *  @param  rpm  The RPM value
     *  @throws  DriverException
     */
    public void setRpm(int rpm) throws DriverException
    {
        setDutyCycle1(minDutyCycle + dcSlope * (Math.max(minRpm, Math.min(maxRpm, rpm)) - minRpm));
    }


    /**
     *  Gets the set RPM
     *
     *  @return  rpm  The RPM value
     *  @throws  DriverException
     */
    public int getRpm() throws DriverException
    {
        return Math.max(minRpm, Math.min(maxRpm, (int)(minRpm + (getDutyCycle1() - minDutyCycle) / dcSlope)));
    }

}
