package org.lsst.ccs.subsystem.utility;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.utility.constants.FanState;
import org.lsst.ccs.subsystem.utility.constants.HeaterState;
import org.lsst.ccs.subsystem.utility.constants.PurgeTestFans;
import org.lsst.ccs.subsystem.utility.constants.PurgeTestHeaters;
import org.lsst.ccs.subsystem.utility.constants.PurgeTestSwitches;
import org.lsst.ccs.subsystem.utility.constants.PurgeTestValves;
import org.lsst.ccs.subsystem.utility.constants.SwitchState;
import org.lsst.ccs.subsystem.utility.constants.UtilityAgentProperties;
import org.lsst.ccs.subsystem.utility.constants.ValveState;
import org.lsst.ccs.subsystem.utility.constants.VpcControlState;
import org.lsst.ccs.subsystem.utility.data.PurgeTestState;
import org.lsst.ccs.subsystem.utility.data.UtilityException;

/**
 *  Implements the purge test subsystem.
 *
 *  @author Owen Saxton
 */
public class PurgeTestMain extends Subsystem implements HasLifecycle {

    private static final int[] devNums = new int[PurgeTestSwitches.NUM_SWITCHES];
    static {
        devNums[PurgeTestSwitches.SW_BFR_24V_PDU] = PurgeTestSwitches.DEVC_BFR;
        devNums[PurgeTestSwitches.SW_BFR_48V_PDU] = PurgeTestSwitches.DEVC_BFR;
        devNums[PurgeTestSwitches.SW_BFR_HEATER_1] = PurgeTestSwitches.DEVC_BFR;
        devNums[PurgeTestSwitches.SW_BFR_HEATER_2] = PurgeTestSwitches.DEVC_BFR;
        devNums[PurgeTestSwitches.SW_BFR_HTR_SIM] = PurgeTestSwitches.DEVC_BFR;
        devNums[PurgeTestSwitches.SW_PDU_24V_MAIN] = PurgeTestSwitches.DEVC_PDU_24V;
        devNums[PurgeTestSwitches.SW_PDU_ELEX] = PurgeTestSwitches.DEVC_PDU_24V;
        devNums[PurgeTestSwitches.SW_PDU_48V_MAIN] = PurgeTestSwitches.DEVC_PDU_48V;
        devNums[PurgeTestSwitches.SW_PDU_UT_FAN] = PurgeTestSwitches.DEVC_PDU_48V;
        devNums[PurgeTestSwitches.SW_PDU_MPC_FAN] = PurgeTestSwitches.DEVC_PDU_48V;
        devNums[PurgeTestSwitches.SW_PDU_VPC_FAN] = PurgeTestSwitches.DEVC_PDU_48V;
    }
    private static final int[] swNums = new int[PurgeTestSwitches.NUM_SWITCHES];
    static {
        swNums[PurgeTestSwitches.SW_BFR_24V_PDU] = BfrDevice.CHAN_24V_PDU;
        swNums[PurgeTestSwitches.SW_BFR_48V_PDU] = BfrDevice.CHAN_48V_PDU;
        swNums[PurgeTestSwitches.SW_BFR_HEATER_1] = BfrDevice.CHAN_HEATER_1;
        swNums[PurgeTestSwitches.SW_BFR_HEATER_2] = BfrDevice.CHAN_HEATER_2;
        swNums[PurgeTestSwitches.SW_BFR_HTR_SIM] = BfrDevice.CHAN_HEATER_SIM;
        swNums[PurgeTestSwitches.SW_PDU_24V_MAIN] = -1;
        swNums[PurgeTestSwitches.SW_PDU_ELEX] = Pdu24VDevice.CHAN_ELECTRONICS;
        swNums[PurgeTestSwitches.SW_PDU_48V_MAIN] = -1;
        swNums[PurgeTestSwitches.SW_PDU_UT_FAN] = Pdu48VDevice.CHAN_UT_FAN;
        swNums[PurgeTestSwitches.SW_PDU_MPC_FAN] = Pdu48VDevice.CHAN_MPC_FAN;
        swNums[PurgeTestSwitches.SW_PDU_VPC_FAN] = Pdu48VDevice.CHAN_VPC_FAN; 
    }
    private static final Map<Integer, String> typeMap = new HashMap<>();
    static {
        typeMap.put(PurgeTestSwitches.DEVC_BFR, "BFR");
        typeMap.put(PurgeTestSwitches.DEVC_PDU_24V, "24V PDU");
        typeMap.put(PurgeTestSwitches.DEVC_PDU_48V, "48V PDU");
    }

    private static final Logger LOG = Logger.getLogger(PurgeTestMain.class.getName());

    @LookupName
    String name;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService propertiesService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private final List<SwitchControl> swDeviceList = new ArrayList<>();
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private final Map<String, FanPIControl> fanDeviceMap = new LinkedHashMap<>();
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private VpcPIControl vpcControl;
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private PurgeMaq20Device purgeControl;

    private String mpcFan, vpcFan, utFan;

    private final SwitchControl[] swDevices = new SwitchControl[PurgeTestSwitches.NUM_DEVICES];
    private final PurgeTestState purgeState = new PurgeTestState();
    private final FanPIControl[] fanControl = new FanPIControl[PurgeTestFans.NUM_FANS];


    /**
     *  Constructor.
     */
    public PurgeTestMain() {
        super("purgetest", AgentInfo.AgentType.WORKER);
    }


    /**
     *  Build phase
     */
    @Override
    public void build() {
        // Create and schedule an AgentPeriodicTask to update and publish the purge system state
        AgentPeriodicTask pt = new AgentPeriodicTask("Purge-state",
                                                     () -> updateState()).withPeriod(Duration.ofMillis(1000));
        periodicTaskService.scheduleAgentPeriodicTask(pt);
    }


    /**
     *  Subsystem post-initialization
     */
    @Override
    public void postInit()
    {
        // Set a property to define that this Agent's type.
        propertiesService.setAgentProperty(UtilityAgentProperties.PURGE_TEST_TYPE, getClass().getCanonicalName());

        // Check power devices and control channels
        for (SwitchControl sd : swDeviceList) {
            swDevices[sd.getSwitchDevice()] = sd;
        }
        for (int j = 0; j < swDevices.length; j++) {
            if (swDevices[j] == null) {
                ErrorUtils.reportConfigError(LOG, name, typeMap.get(j), "has not been defined");
            }
        }

        // Check fan controllers
        if (mpcFan != null) {
            fanControl[PurgeTestFans.FAN_MPC_ID] = fanDeviceMap.get(mpcFan);
        }
        if (fanControl[PurgeTestFans.FAN_MPC_ID] == null) {
            ErrorUtils.reportConfigError(LOG, name, "mpcFan", "not specified or not defined");
        }
        if (vpcFan != null) {
            fanControl[PurgeTestFans.FAN_VPC_ID] = fanDeviceMap.get(vpcFan);
        }
        if (fanControl[PurgeTestFans.FAN_VPC_ID] == null) {
            ErrorUtils.reportConfigError(LOG, name, "vpcFan", "not specified or not defined");
        }
        if (utFan != null) {
            fanControl[PurgeTestFans.FAN_UT_ID] = fanDeviceMap.get(utFan);
        }
        if (fanControl[PurgeTestFans.FAN_UT_ID] == null) {
            ErrorUtils.reportConfigError(LOG, name, "utFan", "not specified or not defined");
        }
        if (vpcControl == null) {
            ErrorUtils.reportConfigError(LOG, name, "VpcPIControl", "has not been defined");
        }
        if (purgeControl == null) {
            ErrorUtils.reportConfigError(LOG, name, "PurgeMaq20Device", "has not been defined");
        }
    }


    /**
     *  Starts the subsystem.
     */
    @Override
    public void postStart()
    {
        //updateState();
        //publishState();        // For any GUIs
        LOG.info("Purge test subsystem started");
    }


    /**
     *  Gets the subsystem state.
     *
     *  @return  The full PDU system state
     */
    @Command(type=CommandType.QUERY, description="Get the full state")
    public PurgeTestState getFullState()
    {
        purgeState.setTickMillis(getTickPeriod());
        return purgeState;
    }


    /**
     *  Gets the list of switch names.
     * 
     *  @return  The list of names
     */
    @Command(type=Command.CommandType.QUERY, description="Get list of switch names")
    public List<String> getSwitchNames()
    {
        return new ArrayList<>(PurgeTestSwitches.switchNameToId.keySet());
    }


    /**
     *  Turns a switch on or off (using its name).
     *
     *  @param  name  The switch name.
     *  @param  on  Whether to turn on or off
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Turn on/off a named switch")
    public void setSwitchOn(@Argument(description="The switch name") String name,
                            @Argument(description="Whether to turn on") boolean on) throws UtilityException
    {
        try {
            Integer sw = PurgeTestSwitches.switchNameToId.get(name);
            if (sw == null) {
                throw new UtilityException("Invalid switch name: " + name);
            }
            synchronized (purgeState) {
                SwitchState state = on ? SwitchState.ON : SwitchState.OFF;
                if (state == purgeState.getSwitchBaseState(sw)) return;
                purgeState.setSwitchBaseState(sw, state);
                if (purgeState.getSwitchState(sw) == SwitchState.OFFLINE) return;
                purgeState.setSwitchState(sw, state);
                setSwitchOn(sw, on);
            }
        }
        finally {
            publishState();
        }
    }


    /**
     *  Gets the list of fan names.
     * 
     *  @return  The list of names
     */
    @Command(type=Command.CommandType.QUERY, description="Get list of fan names")
    public List<String> getFanNames()
    {
        return new ArrayList<>(PurgeTestFans.fanNameToId.keySet());
    }


    /**
     *  Sets a fan state (using its name).
     *
     *  @param  name   The fan name
     *  @param  state  The state to set
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Set a fan control state")
    public void setFanState(@Argument(description="The fan name") String name,
                            @Argument(description="The control state") FanState state) throws UtilityException
    {
        try {
            int fanId = getFanId(name);
            if (state == FanState.OFFLINE || (state == FanState.TEMP && !PurgeTestFans.hasTempState[fanId])) {
                throw new UtilityException("Invalid " + name + " state: " + state);
            }
            synchronized (purgeState) {
                if (state == purgeState.getFanBaseState(fanId)) return;
                purgeState.setFanBaseState(fanId, state);
                if (purgeState.getFanState(fanId) == FanState.OFFLINE) return;
                purgeState.setFanState(fanId, state);
                FanPIControl ctrlr = fanControl[fanId];
                if (state == FanState.TEMP) {
                    ctrlr.setTemperature(purgeState.getDeltaTemp(fanId));
                    ctrlr.startLoop(purgeState.getFanSpeed(fanId));
                }
                else {
                    if (PurgeTestFans.hasTempState[fanId]) {
                        ctrlr.stopLoop();
                    }
                    if (state == FanState.SPEED) {
                        ctrlr.setSpeed(purgeState.getFanSpeed(fanId));
                    }
                }
                setSwitchOn(PurgeTestFans.fanSwitchMap.get(fanId), state != FanState.OFF);
            }
        }
        finally {
            publishState();
        }
    }


    /**
     *  Sets a fan speed (using its name).
     *
     *  @param  name   The fan name
     *  @param  speed  The fan speed
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Set a fan speed")
    public void setFanSpeed(@Argument(description="The fan name") String name,
                            @Argument(description="The fan speed") double speed) throws UtilityException
    {
        try {
            int fanId = getFanId(name);
            synchronized (purgeState) {
                purgeState.setFanSpeed(fanId, speed);
                if (purgeState.getFanState(fanId) == FanState.SPEED) {
                    fanControl[fanId].setSpeed(speed);
                }
            }
        }
        finally {
            publishState();
        }
    }


    /**
     *  Sets the VPC control state.
     *
     *  @param  state  The state to set
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Set the VPC control state")
    public void setVpcState(@Argument(description="The control state") VpcControlState state) throws UtilityException
    {
        try {
            synchronized (purgeState) {
                if (state == purgeState.getVpcState()) return;
                purgeState.setVpcState(state);
                HeaterState h1State = purgeState.getHeaterState(PurgeTestHeaters.HEATER_VPC1_ID);
                HeaterState h2State = purgeState.getHeaterState(PurgeTestHeaters.HEATER_VPC2_ID);
                ValveState vState = purgeState.getValveState(PurgeTestValves.VALVE_VPC_ID);
                if (state == VpcControlState.TEMP) {
                    if (h1State != HeaterState.OFFLINE) {
                        purgeState.setHeaterState(PurgeTestHeaters.HEATER_VPC1_ID, HeaterState.TEMP);
                    }
                    if (h2State != HeaterState.OFFLINE) {
                        purgeState.setHeaterState(PurgeTestHeaters.HEATER_VPC2_ID, HeaterState.TEMP);
                    }
                    if (vState != ValveState.OFFLINE) {
                        purgeState.setValveState(PurgeTestValves.VALVE_VPC_ID, ValveState.TEMP);
                    }
                    vpcControl.startLoop();
                }
                else {
                    vpcControl.stopLoop();
                    if (h1State != HeaterState.OFFLINE) {
                        purgeState.setHeaterState(PurgeTestHeaters.HEATER_VPC1_ID,
                                                  purgeState.getHeaterBaseState(PurgeTestHeaters.HEATER_VPC1_ID));
                        operateHeater(PurgeTestHeaters.HEATER_VPC1_ID);
                    }
                    if (h2State != HeaterState.OFFLINE) {
                        purgeState.setHeaterState(PurgeTestHeaters.HEATER_VPC2_ID,
                                                  purgeState.getHeaterBaseState(PurgeTestHeaters.HEATER_VPC2_ID));
                        operateHeater(PurgeTestHeaters.HEATER_VPC2_ID);
                    }
                    if (vState != ValveState.OFFLINE) {
                        purgeState.setValveState(PurgeTestValves.VALVE_VPC_ID,
                                                 purgeState.getValveBaseState(PurgeTestValves.VALVE_VPC_ID));
                        operateValve(PurgeTestValves.VALVE_VPC_ID);
                    }
                }
            }
        }
        finally {
            publishState();
        }
    }


    /**
     *  Gets the list of valve names.
     * 
     *  @return  The list of names
     */
    @Command(type=Command.CommandType.QUERY, description="Get list of valve names")
    public List<String> getValveNames()
    {
        return new ArrayList<>(PurgeTestValves.valveNameToId.keySet());
    }


    /**
     *  Sets a valve state (using its name).
     *
     *  @param  name   The valve name
     *  @param  state  The state to set
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Set a valve control state")
    public void setValveState(@Argument(description="The valve name") String name,
                              @Argument(description="The control state") ValveState state) throws UtilityException
    {
        try {
            int valveId = getValveId(name);
            if (state == ValveState.OFFLINE || state == ValveState.TEMP) {
                throw new UtilityException("Invalid " + name + " state: " + state);
            }
            synchronized (purgeState) {
                if (state == purgeState.getValveBaseState(valveId)) return;
                purgeState.setValveBaseState(valveId, state);
                ValveState currState = purgeState.getValveState(valveId);
                if (currState == ValveState.OFFLINE || currState == ValveState.TEMP) return;
                purgeState.setValveState(valveId, state);
                operateValve(valveId);
            }
        }
        finally {
            publishState();
        }
    }


    /**
     *  Sets a valve position (using its name).
     *
     *  @param  name   The valve name
     *  @param  position  The valve position
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Set a valve position")
    public void setValvePosition(@Argument(description="The valve name") String name,
                                 @Argument(description="The valve position") double position) throws UtilityException
    {
        try {
            int valveId = getValveId(name);
            synchronized (purgeState) {
                purgeState.setValvePosition(valveId, position);
                if (purgeState.getValveState(valveId) == ValveState.POSN) {
                    operateValve(valveId);
                }
            }
        }
        finally {
            publishState();
        }
    }


    /**
     *  Sets a delta temperature (using fan name).
     *
     *  @param  name  The fan name
     *  @param  temp  The delta temperature to set
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Set a delta temperature")
    public void setDeltaTemp(@Argument(description="The fan name") String name,
                             @Argument(description="The delta temperature") double temp) throws UtilityException
    {
        try {
            int fanId = getFanId(name);
            synchronized (purgeState) {
                purgeState.setDeltaTemp(fanId, temp);
                if (purgeState.getFanState(fanId) == FanState.TEMP) {
                    if (fanId == PurgeTestFans.FAN_VPC_ID) {
                        vpcControl.setTemperature(temp);
                    }
                    else {
                        fanControl[fanId].setTemperature(temp);
                    }
                }
            }
        }
        finally {
            publishState();
        }
    }


    /**
     *  Gets the list of heater names.
     * 
     *  @return  The list of names
     */
    @Command(type=Command.CommandType.QUERY, description="Get list of heater names")
    public List<String> getHeaterNames()
    {
        return new ArrayList<>(PurgeTestHeaters.heaterNameToId.keySet());
    }


    /**
     *  Sets a heater control state (using its name).
     *
     *  @param  name   The heater name
     *  @param  state  The state to set
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Set a heater control state")
    public void setHeaterState(@Argument(description="The heater name") String name,
                               @Argument(description="The control state") HeaterState state) throws UtilityException
    {
        try {
            int htrId = getHeaterId(name);
            if (state == HeaterState.OFFLINE || state == HeaterState.TEMP) {
                throw new UtilityException("Invalid " + name + " state: " + state);
            }
            synchronized (purgeState) {
                if (state == purgeState.getHeaterBaseState(htrId)) return;
                purgeState.setHeaterBaseState(htrId, state);
                HeaterState currState = purgeState.getHeaterState(htrId);
                if (currState == HeaterState.OFFLINE || currState == HeaterState.TEMP) return;
                purgeState.setHeaterState(htrId, state);
                operateHeater(htrId);
            }
        }
        finally {
            publishState();
        }
    }


    /**
     *  Sets a heater power (using its name).
     *
     *  @param  name   The heater name
     *  @param  power  The heater power
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Set a heater power")
    public void setHeaterPower(@Argument(description="The heater name") String name,
                               @Argument(description="The power value") double power) throws UtilityException
    {
        try {
            int htrId = getHeaterId(name);
            synchronized (purgeState) {
                if (htrId != PurgeTestHeaters.HEATER_SIM_ID) {
                    throw new UtilityException("Cannot set power for " + name);
                }
                purgeState.setHeaterValue(htrId, power);
                operateHeater(htrId);
            }
        }
        finally {
            publishState();
        }
    }


    /**
     *  Sets the update (tick) period.
     *
     *  @param  period  The update period (milliseconds) to set.
     */
    @Command(type=CommandType.ACTION, description="Set the update period")
    public void setUpdatePeriod(@Argument(name="period", description="The tick period (msecs)")
                                int period)
    {
        setTickPeriod(period);
        //updateState();
        publishState();    // Must do this for GUI
    }


    /**
     *  Turns a switch on or off using its ID.
     *
     *  @param  sw  The switch ID.
     *  @param  on  Whether to turn on or off
     *  @throws  UtilityException
     */
    private void setSwitchOn(int sw, boolean on) throws UtilityException
    {
        if (sw < 0 || sw >= PurgeTestSwitches.NUM_SWITCHES) {
            throw new UtilityException("Invalid switch number: " + sw);
        }
        if (on) {
            swDevices[devNums[sw]].switchOn(swNums[sw]);
        }
        else {
            swDevices[devNums[sw]].switchOff(swNums[sw]);
        }
        //updateState();
        //publishState();
    }


    /**
     *  Operates a heater.
     *
     *  @param  htrId  The heater ID
     *  @throws  UtitliyException
     */
    private void operateHeater(int htrId) throws UtilityException
    {
        HeaterState state = purgeState.getHeaterState(htrId);
        if (state == HeaterState.OFFLINE || state == HeaterState.TEMP) return;
        setSwitchOn(PurgeTestHeaters.heaterSwitchMap.get(htrId), state == HeaterState.ON);
        if (htrId == PurgeTestHeaters.HEATER_SIM_ID && state == HeaterState.ON) {
            purgeControl.setSimHeater(purgeState.getHeaterValue(htrId));
        }
    }


    /**
     *  Operates a coolant valve.
     *
     *  @param  valveId  The valve ID
     *  @throws  UtilityException
     */
    private void operateValve(int valveId) throws UtilityException
    {
        ValveState state = purgeState.getValveState(valveId);
        if (state == ValveState.OFFLINE || state == ValveState.TEMP) return;
        double position = state == ValveState.SHUT ? 0.0 : purgeState.getValvePosition(valveId);
        if (valveId == PurgeTestValves.VALVE_VPC_ID) {
            purgeControl.setVpcValve(position);
        }
        else {
            purgeControl.setUtValve(position);
        }
        
    }


    /**
     *  Updates the purge system state when devices go offline or online.
     */
    private void updateState()
    {
        synchronized (purgeState) {
            boolean changed = false;
            for (int sw = 0; sw < PurgeTestSwitches.NUM_PWR_SWITCHES; sw++) {
                Boolean st = swDevices[devNums[sw]].isSwitchOn(swNums[sw]);
                SwitchState ss = st == null ? SwitchState.OFFLINE : st ? SwitchState.ON : SwitchState.OFF;
                if (ss != purgeState.getSwitchState(sw)) {
                    if (ss != SwitchState.OFFLINE) {
                        purgeState.setSwitchBaseState(sw, ss);
                    }
                    purgeState.setSwitchState(sw, ss);
                    changed = true;
                }
            }
            for (int fan = 0; fan < PurgeTestFans.NUM_FANS; fan++) {
                int sw = PurgeTestFans.fanSwitchMap.get(fan);
                Boolean st = swDevices[devNums[sw]].isSwitchOn(swNums[sw]);
                FanState fs = st == null ? FanState.OFFLINE : !st ? FanState.OFF : purgeState.getFanBaseState(fan);
                if (fs != purgeState.getFanState(fan)) {
                    if (fs != FanState.OFFLINE) {
                        purgeState.setFanBaseState(fan, fs);
                    }
                    purgeState.setFanState(fan, fs);
                    FanPIControl ctrlr = fanControl[fan];
                    if (ctrlr != null && fs == FanState.SPEED) {
                        ctrlr.setSpeed(purgeState.getFanSpeed(fan));
                    }
                    changed = true;
                }
            }
            for (int htr = 0; htr < PurgeTestHeaters.NUM_HEATERS; htr++) {
                int sw = PurgeTestHeaters.heaterSwitchMap.get(htr);
                Boolean st = swDevices[devNums[sw]].isSwitchOn(swNums[sw]);
                HeaterState hs = st == null ? HeaterState.OFFLINE :
                                 PurgeTestHeaters.hasAutoState[htr] && purgeState.getVpcState() == VpcControlState.TEMP
                                   ? HeaterState.TEMP : st ? HeaterState.ON : HeaterState.OFF;
                if (hs != purgeState.getHeaterState(htr)) {
                    if (hs != HeaterState.TEMP && hs != HeaterState.OFFLINE) {
                        purgeState.setHeaterBaseState(htr, hs);
                    }
                    purgeState.setHeaterState(htr, hs);
                    try {
                        operateHeater(htr);
                    }
                    catch (UtilityException e) {
                        LOG.log(Level.SEVERE, "Error operating heater {0}: {1}", new Object[]{htr, e});
                    }
                    changed = true;
                }
            }
            for (int valve = 0; valve < PurgeTestValves.NUM_VALVES; valve++) {
                ValveState vs = !purgeControl.isOnline() ? ValveState.OFFLINE :
                                PurgeTestValves.hasTempState[valve] && purgeState.getVpcState() == VpcControlState.TEMP
                                  ? ValveState.TEMP : purgeState.getValveBaseState(valve);
                if (vs != purgeState.getValveState(valve)) {
                    purgeState.setValveState(valve, vs);
                    try {
                        operateValve(valve);
                    }
                    catch (UtilityException e) {
                        LOG.log(Level.SEVERE, "Error operating valve {0}: {1}", new Object[]{valve, e});
                    }
                    changed = true;
                }
            }
            if (changed) {
                publishState();
            }
        }
    }


    /**
     *  Sets the tick period
     */
    private void setTickPeriod(long period)
    {
        periodicTaskService.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }
    

    /**
     *  Gets the tick period
     */
    private int getTickPeriod()
    {
        return (int)periodicTaskService.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }
    

    /**
     *  Publishes the state of the system.
     *
     *  This is intended to be called whenever any element of the state is changed.
     */
    private void publishState()
    {
        purgeState.setTickMillis(getTickPeriod());
        publishSubsystemDataOnStatusBus(new KeyValueData(PurgeTestState.KEY, purgeState));
    }    


    /**
     *  Gets a fan ID from its name.
     */
    private int getFanId(String name) throws UtilityException
    {
        Integer id = PurgeTestFans.fanNameToId.get(name);
        if (id == null) {
            throw new UtilityException("Invalid fan name: " + name);
        }
        return id;
    }


    /**
     *  Gets a valve ID from its name.
     */
    private int getValveId(String name) throws UtilityException
    {
        Integer id = PurgeTestValves.valveNameToId.get(name);
        if (id == null) {
            throw new UtilityException("Invalid valve name: " + name);
        }
        return id;
    }


    /**
     *  Gets a heater ID from its name.
     */
    private int getHeaterId(String name) throws UtilityException
    {
        Integer id = PurgeTestHeaters.heaterNameToId.get(name);
        if (id == null) {
            throw new UtilityException("Invalid heater name: " + name);
        }
        return id;
    }

}
