package org.lsst.ccs.subsystem.utility;

import java.util.ArrayList;
import java.time.Duration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.Alert;
//import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.alert.AlertEvent;
import org.lsst.ccs.services.alert.AlertListener;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.utility.constants.MpmAlert;
import org.lsst.ccs.subsystem.utility.constants.ConditionState;
import org.lsst.ccs.subsystem.utility.constants.LatchState;
import org.lsst.ccs.subsystem.utility.constants.MpmConditions;
import org.lsst.ccs.subsystem.utility.constants.MpmLatches;
import org.lsst.ccs.subsystem.utility.constants.MpmLimits;
import org.lsst.ccs.subsystem.utility.constants.MpmPlcs;
import org.lsst.ccs.subsystem.utility.constants.MpmSwitches;
import org.lsst.ccs.subsystem.utility.constants.PLCState;
import org.lsst.ccs.subsystem.utility.constants.SwitchState;
import org.lsst.ccs.subsystem.utility.constants.UtilityAgentProperties;
import org.lsst.ccs.subsystem.utility.data.MpmSysState;
import org.lsst.ccs.subsystem.utility.data.UtilityException;

/**
 * The master protection system
 *
 * @author The LSST CCS Team
 */
public class MpmMain /*extends Subsystem*/ implements HasLifecycle, ClearAlertHandler, AlertListener {

    private static final int[] switchChannels = new int[MpmSwitches.NUM_SWITCHES];
    static {
        switchChannels[MpmSwitches.SW_BLOCK_COLD_HEAT] = MpmPlutoDevice.SW_BLOCK_COLD_HEAT;
        switchChannels[MpmSwitches.SW_BLOCK_COLD_REFG] = MpmPlutoDevice.SW_BLOCK_COLD_REFG;
        switchChannels[MpmSwitches.SW_BLOCK_CRYO_HEAT] = MpmPlutoDevice.SW_BLOCK_CRYO_HEAT;
        switchChannels[MpmSwitches.SW_BLOCK_CRYO_REFG] = MpmPlutoDevice.SW_BLOCK_CRYO_REFG;
        switchChannels[MpmSwitches.SW_BLOCK_COOLANT] = MpmPlutoDevice.SW_BLOCK_COOLANT;
        switchChannels[MpmSwitches.SW_BLOCK_REB_POWER] = MpmPlutoDevice.SW_BLOCK_REB_POWER;
        switchChannels[MpmSwitches.SW_BLOCK_UT_POWER] = MpmPlutoDevice.SW_BLOCK_UT_POWER;
    }
    private static final Map<Integer, MpmAlert> alertMap = new HashMap<>();
    static {
        alertMap.put(MpmLatches.LATCH_COLD_TEMP_HIGH, MpmAlert.COLD_TEMP_HIGH);
        alertMap.put(MpmLatches.LATCH_COLD_TEMP_LOW, MpmAlert.COLD_TEMP_LOW);
        alertMap.put(MpmLatches.LATCH_CRYO_TEMP_HIGH, MpmAlert.CRYO_TEMP_HIGH);
        alertMap.put(MpmLatches.LATCH_CRYO_TEMP_LOW, MpmAlert.CRYO_TEMP_LOW);
        alertMap.put(MpmLatches.LATCH_CRYO_VACUUM, MpmAlert.CRYO_VACUUM_BAD);
        alertMap.put(MpmLatches.LATCH_HEX_VACUUM, MpmAlert.HEX_VACUUM_BAD);
        alertMap.put(MpmLatches.LATCH_UT_LEAK, MpmAlert.UT_COOLANT_LEAK);
        alertMap.put(MpmLatches.LATCH_UT_LEAK_FAULT, MpmAlert.UT_COOLANT_LEAK);
        alertMap.put(MpmLatches.LATCH_UT_SMOKE, MpmAlert.UT_SMOKE_DETC);
        alertMap.put(MpmLatches.LATCH_UT_SMOKE_FAULT, MpmAlert.UT_SMOKE_DETC);
        alertMap.put(MpmLatches.LATCH_UT_TEMP, MpmAlert.UT_TEMP_HIGH);
    }
    private static final Map<String, String> revAlertMap = new HashMap<>();
    static {
        for (Map.Entry e : alertMap.entrySet()) {
            revAlertMap.put(((MpmAlert)e.getValue()).getId(), MpmLatches.ID_MAP.get((int)e.getKey()));
        }
    }
    private static final String[] plcNames = new String[MpmPlcs.NUM_PLCS];
    static {
        plcNames[MpmPlcs.PLC_TRUNK] = "Trunk";
        plcNames[MpmPlcs.PLC_COLD] = "Cold";
        plcNames[MpmPlcs.PLC_CRYO] = "Cryo";
    }

    @LookupName
    private String name;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService propertiesService;
    @LookupField( strategy=LookupField.Strategy.TOP)
    private Subsystem subsys;

    @LookupField(strategy=LookupField.Strategy.DESCENDANTS)
    private MpmPlutoDevice plutoDevc;

    // General
    private static final Logger LOG = Logger.getLogger(MpmMain.class.getName());
    private final MpmSysState mpmState = new MpmSysState();
    private final Device[] switchDevices = new Device[MpmSwitches.NUM_SWITCHES];
    private boolean running = false;
    private final Map<String, Boolean> activeAlertMap = new HashMap<>();

    /*
    public MpmMain() {
        super("mpm", AgentInfo.AgentType.WORKER);
    }
    */

    /**
     *  Build phase
     */
    @Override
    public void build() {
        //setAgentProperty("org.lsst.ccs.use.full.paths", "true");
        //Create and schedule an AgentPeriodicTask to update the protection state
        AgentPeriodicTask pt;
        pt = new AgentPeriodicTask("Protection-state",
                                   () -> updateMpmState()).withPeriod(Duration.ofMillis(1000));
        periodicTaskService.scheduleAgentPeriodicTask(pt);
        //ass.registerState(ProtectionState.class, "Protection state", this);
        //ass.updateAgentState(ProtectionState.UNKNOWN);
        for (MpmAlert alert : MpmAlert.values()) {
            activeAlertMap.put(alert.getId(), false);
        }
    }


    /**
     *  Post initialization
     */
    @Override
    public void postInit() {
        // Set a property to define that this Agent is a protection subsystem.
        propertiesService.setAgentProperty(UtilityAgentProperties.MPM_TYPE, MpmMain.class.getCanonicalName());

        // Add alert listener
        alertService.addListener(this);

        if (plutoDevc != null) {
            for (int cond = 0; cond < MpmLatches.NUM_LATCHES; cond++) {
                mpmState.setLatch(cond, LatchState.CLEAR);
            }
            for (int cond = 0; cond < MpmConditions.NUM_CONDITIONS; cond++) {
                mpmState.setCondition(cond, ConditionState.NO);
            }
        }
        else {
            ErrorUtils.reportConfigError(LOG, name, "Pluto device", "not specified");
        }

        for (int sw : switchChannels) {
            mpmState.setSwitchState(sw, SwitchState.OFFLINE);
            switchDevices[sw] = plutoDevc;
        }
    }


    /**
     *  Post start
     */
    @Override
    public void postStart() {
        LOG.info("Protection subsystem started");
        running = true;
    }


    /**
     *  Gets the state of the protection system.
     *
     *  @return  The protection system state
     */
    @Command(type=Command.CommandType.QUERY, description="Get the protection system state")
    public MpmSysState getSystemState() {
        mpmState.setTickMillis(getTickPeriod());
        return mpmState;
    }    


    /**
     *  Gets the list of switch names.
     *
     *  @return  The switch names.
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.QUERY, description="Get switch names")
    public List<String> getSwitchNames() throws UtilityException
    {
        return new ArrayList(MpmSwitches.NAME_MAP.keySet());
    }


    /**
     *  Turns a (named) switch on or off.
     *
     *  @param  swch  The switch name.
     *  @param  on    Whether to turn on or off
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Turn on/off a named switch")
    public void setNamedSwitchOn(@Argument(description="The switch name") String swch,
                                 @Argument(description="Whether to turn on") boolean on) throws UtilityException
    {
        try {
            Integer sw = MpmSwitches.NAME_MAP.get(swch);
            if (sw == null) {
                throw new UtilityException("Invalid switch name: " + swch);
            }
            SwitchState state = mpmState.getSwitchState(sw);
            if (state == SwitchState.OFFLINE) return;
            Device swDevice = switchDevices[sw];
            if (swDevice == plutoDevc) {
                plutoDevc.setSwitchOn(switchChannels[sw], on);
            }
        }
        finally {
            updateMpmState();
        }
    }


    /**
     *  Gets whether a switch is on.
     *
     *  @param  sw  The switch number.
     *  @return  Whether the switch is on
     *  @throws  UtilityException
     */
    private Boolean isSwitchOn(int sw)
    {
        Boolean value = false;
        Device swDevice = switchDevices[sw];
        if (swDevice == plutoDevc) {
            value = plutoDevc.isSwitchOn(switchChannels[sw]);
        }
        return value;
    }


    /**
     *  Gets the list of latched condition names.
     *
     *  @return  The condition names.
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.QUERY, description="Get latched condition names")
    public List<String> getLatchNames() throws UtilityException
    {
        return new ArrayList(MpmLatches.NAME_MAP.keySet());
    }


    /**
     *  Clears a (named) latched condition.
     *
     *  @param  cond  The condition name.
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Clear a latched condition")
    public void clearLatch(@Argument(description="The condition name") String cond) throws UtilityException
    {
        try {
            Integer id = MpmLatches.NAME_MAP.get(cond);
            if (id == null) {
                throw new UtilityException("Invalid condition name: " + cond);
            }
            plutoDevc.clearLatch(id);
        }
        finally {
            updateMpmState();
        }
    }


    /**
     *  Sets the update period.
     *
     *  @param  value  The update period (milliseconds) to set.
     */
    @Command(type=Command.CommandType.ACTION, description="Set the update interval")
    public void setUpdatePeriod(@Argument(description="The tick period (ms)") int value)
    {
        setTickPeriod(value);
        publishState();
    }


    /**
     *  Updates the protection system state periodically.
     *
     *  The protection state consists mainly of the state of the switches (lines) being
     *  controlled, along with whether they can be turned on.
     */
    private void updateMpmState()
    {
        if (!running) return;

        boolean changed = false;

        for (int plc = 0; plc < MpmPlcs.NUM_PLCS; plc++) {
            PLCState oldState = mpmState.getPlcState(plc);
            Boolean active = plutoDevc.isPlcActive(plc);
            PLCState newState = active == null ? PLCState.OFFLINE : active ? PLCState.ALIVE : PLCState.DEAD;
            if (newState != oldState) {
                changed = true;
                mpmState.setPlcState(plc, newState);
                String desc = plcNames[plc] + " protection PLC ";
                if (newState == PLCState.ALIVE) {
                    lowerAlert(MpmAlert.PROT_PLC_NOT_ALIVE, desc + "is alive");
                }
                else {
                    raiseAlert(MpmAlert.PROT_PLC_NOT_ALIVE, desc + (newState == PLCState.DEAD ? "has died" : "is offline"));
                }
            }
        }

        for (int sw = 0; sw < MpmSwitches.NUM_SWITCHES; sw++) {
            SwitchState oldState = mpmState.getSwitchState(sw);
            Boolean isOn = isSwitchOn(sw);
            SwitchState state = isOn != null ? isOn ? SwitchState.ON : SwitchState.OFF : SwitchState.OFFLINE;
            if (state != oldState) {
                mpmState.setSwitchState(sw, state);
                changed = true;
            }
        }

        Set<MpmAlert> raisedAlerts = new HashSet<>(), loweredAlerts = new HashSet<>();
        for (int cond = 0; cond < MpmLatches.NUM_LATCHES; cond++) {
            Boolean active = plutoDevc.isLatchActive(cond);
            Boolean latched = plutoDevc.isLatchLatched(cond);
            LatchState state = active == null || latched == null ? LatchState.OFFLINE :
                               latched ? LatchState.LATCHED :
                               active ? LatchState.ACTIVE : LatchState.CLEAR;
            MpmAlert alert = alertMap.get(cond);
            if (state == LatchState.ACTIVE) {
                raisedAlerts.add(alert);
            }
            LatchState oldState = mpmState.getLatch(cond); 
            if (state != oldState) {
                mpmState.setLatch(cond, state);
                if (state == LatchState.ACTIVE) {
                    raiseAlert(alert, "Protection PLC error condition set");
                }
                else if (state != LatchState.OFFLINE && oldState == LatchState.ACTIVE) {
                    loweredAlerts.add(alert);
                }
                changed = true;
            }
        }
        for (MpmAlert alert : loweredAlerts) {
            if (!raisedAlerts.contains(alert)) {
                lowerAlert(alert, "Protection PLC error condition cleared");
            }
        }

        for (int cond = 0; cond < MpmConditions.NUM_CONDITIONS; cond++) {
            Boolean active = plutoDevc.isConditionActive(cond);
            ConditionState state = active == null ? ConditionState.OFF :
                                   active ? ConditionState.YES : ConditionState.NO;
            if (state != mpmState.getCondition(cond)) {
                mpmState.setCondition(cond, state);
                changed = true;
            }
        }

        if (mpmState.getLimit(0) == Integer.MAX_VALUE) {
            int[] limits = plutoDevc.getTempLimits();
            for (int j = 0; j < MpmLimits.NUM_LIMITS; j++) {
                mpmState.setLimit(j, limits[j]);
            }
            changed = true;
        }

        if (changed) {
            publishState();
        }
    }


    /**
     *  Raises an alert.
     *
     *  @param  alert  The protection alert to raise
     *  @param  cond   The alert condition
     */
    private void raiseAlert(MpmAlert alert, String cond)
    {
        alertService.raiseAlert(alert.getAlert(), AlertState.ALARM, cond);
        activeAlertMap.put(alert.getId(), true);
    }


    /**
     *  Lowers an alert.
     *
     *  @param  alert  The protection alert to lower
     *  @param  cond   The alert condition
     */
    private void lowerAlert(MpmAlert alert, String cond)
    {
        alertService.raiseAlert(alert.getAlert(), AlertState.NOMINAL, cond);
        activeAlertMap.put(alert.getId(), false);
    }


    /**
     * Callback to clear an {@code Alert} instance.
     * 
     * @param alert The Alert instance to clear.
     * @param alertState The AlertState for the provided Alert.
     * @return A ClearAlertCode to indicate which action is to be taken
     *         by the framework.
     * 
     */
    @Override
    public ClearAlertCode canClearAlert(Alert alert, AlertState alertState)
    {
        Boolean active = activeAlertMap.get(alert.getAlertId());
        return active == null ? ClearAlertCode.UNKNOWN_ALERT
                              : active ? ClearAlertCode.DONT_CLEAR_ALERT : ClearAlertCode.CLEAR_ALERT;
    }


    /**
     *  Alert event handler.
     *
     *  Resets PLC latch when corresponding alert is cleared.
     *
     *  @param  event  The alert event
     */
    @Override
    public void onAlert(AlertEvent event)
    {
        if (event.getType() != AlertEvent.AlertEventType.ALERT_CLEARED) return;
        for (String id : event.getClearedIds()) {
            String cond = revAlertMap.get(id);
            if (cond != null) {
                try {
                    clearLatch(cond);
                }
                catch (UtilityException e) {
                    LOG.log(Level.SEVERE, "Error clearing latched PLC condition ({0}): {1}", new Object[]{cond, e});
                }
            }
        }
    }


    /**
     *  Publishes the state of the protection system.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    private void publishState()
    {
        mpmState.setTickMillis(getTickPeriod());
        subsys.publishSubsystemDataOnStatusBus(new KeyValueData(MpmSysState.KEY, mpmState));
    }


    /**
     *  Sets the monitoring publishing period.
     */
    private void setTickPeriod(long period)
    {
        periodicTaskService.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }
    

    /**
     *  Gets the monitoring publishing period.
     */
    private int getTickPeriod()
    {
        return (int)periodicTaskService.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }

}
